package fr.ifremer.adagio.synchro.intercept.data;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.collect.ImmutableList;
import com.google.common.eventbus.Subscribe;
import fr.ifremer.adagio.core.dao.technical.synchronization.SynchronizationStatus;
import fr.ifremer.adagio.synchro.service.SynchroDirection;
import fr.ifremer.common.synchro.dao.Daos;
import fr.ifremer.common.synchro.dao.SynchroTableDao;
import fr.ifremer.common.synchro.intercept.SynchroOperationRepository;
import fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.common.synchro.meta.SynchroTableMetadata.DuplicateKeyStrategy;
import fr.ifremer.common.synchro.meta.event.CreateQueryEvent;
import fr.ifremer.common.synchro.meta.event.LoadJoinEvent;
import fr.ifremer.common.synchro.meta.event.LoadTableEvent;
import fr.ifremer.common.synchro.query.SynchroQueryBuilder;
import fr.ifremer.common.synchro.query.SynchroQueryOperator;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.List;

/**
 * <p>
 * FishingTripInterceptor class.
 * </p>
 * 
 * @author Benoit Lavenier (benoit.lavenier@e-is.pro)
 * @since 3.5.0
 */
public class FishingTripInterceptor extends AbstractDataInterceptor {

	private static final String TABLE_LANDING = "LANDING";
	private static final String LANDING_COLUMN_FISHING_TRIP_FK = "fishing_trip_fk";
	private static final String FK_COLUMN_NAME = "ID";
	private static final String SYNCHRONIZATION_STATUS_COLUMN_NAME = "SYNCHRONIZATION_STATUS";

	private static final String UNIQUE_KEY_NAME = "FISHING_TRIP_UNIQUE_KEY";
	private static final List<String> UNIQUE_KEY_COLS = ImmutableList.of("DEPARTURE_DATE_TIME", "VESSEL_FK", "PROGRAM_FK",
			"DECLARED_DOCUMENT_REFERENCE_FK");

	private PreparedStatement updateLandingFishingTripFkStatement = null;

	/**
	 * <p>
	 * Constructor for FishingTripInterceptor.
	 * </p>
	 */
	public FishingTripInterceptor() {
		super();
		setEnableOnWrite(true);
	}

	/** {@inheritDoc} */
	@Override
	public boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table) {
		return "FISHING_TRIP".equalsIgnoreCase(table.getName());
	}

	/**
	 * <p>
	 * handleCreateQuery.
	 * </p>
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.CreateQueryEvent} object.
	 */
	@Subscribe
	public void handleCreateQuery(CreateQueryEvent e) {
		switch (e.queryName) {
		// Select queries : remove unused columns
		case count:
		case countFromUpdateDate:
		case select:
		case selectFromUpdateDate:
		case selectMaxUpdateDate:
			// Add restriction on person session
			e.sql = SynchroQueryBuilder.newBuilder(e.sql)
					// Evict if linked to scientific cruise
					.addWhere(SynchroQueryOperator.AND, "t.scientific_cruise_fk is null")
					// Evict if already linked to ObservedLocation (by Landing)
					.addJoin(" LEFT OUTER JOIN landing la ON la.fishing_trip_fk=t.id ")
					// WARNING: make sure there is left & right parenthesis (mantis #25174)
					.addWhere(SynchroQueryOperator.AND, "(la.id is null OR la.observed_location_fk is null)")
					.build();
			break;

		default:
			break;
		}
	}

	/**
	 * <p>
	 * handleLoadJoin.
	 * </p>
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.LoadJoinEvent} object.
	 */
	@Subscribe
	public void handleLoadJoin(LoadJoinEvent e) {
		if (!e.join.isValid()) {
			return;
		}

		// Disable child join to LANDING
		// (LANDING should not be seen as a child of FISHING_TRIP)
		String targetTableName = e.join.getTargetTable().getName();
		String targetColumnName = e.join.getTargetColumn().getName();
		boolean isJoinToLandingTable = TABLE_LANDING.equalsIgnoreCase(targetTableName)
				&& LANDING_COLUMN_FISHING_TRIP_FK.equalsIgnoreCase(targetColumnName);

		if (isJoinToLandingTable) {
			e.join.setIsValid(false);
		}
	}

	/**
	 * <p>
	 * handleTableLoad.
	 * </p>
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.LoadTableEvent} object.
	 */
	@Subscribe
	public void handleTableLoad(LoadTableEvent e) {
		e.table.setRoot(true);

		// Define a natural Id, on server
		if (getConfig().getDirection() == SynchroDirection.EXPORT_TEMP2SERVER) {
			e.table.addUniqueConstraint(UNIQUE_KEY_NAME, UNIQUE_KEY_COLS, DuplicateKeyStrategy.REJECT);
		} else if (getConfig().getDirection() == SynchroDirection.EXPORT_LOCAL2TEMP) {
			// When select by LANDING.FISHING_TRIP_FK == FISHING_TRIP.ID, SYNCHRONIZATION_STATUS must not be DELETED
			e.table.addSelectByFksWhereClause(FK_COLUMN_NAME,
					String.format("t.%s<>'%s'", SYNCHRONIZATION_STATUS_COLUMN_NAME, SynchronizationStatus.DELETED.getValue()));
		}
	}

	/** {@inheritDoc} */
	@Override
	protected void doOnDelete(List<Object> pk, SynchroTableDao sourceDao, SynchroTableDao targetDao, SynchroOperationRepository buffer)
			throws SQLException {

		int localId = Integer.parseInt(pk.get(0).toString());

		// remove reference contains by LANDING.FISHING_TRIP_FK
		removeLinkToLanding(targetDao.getConnection(), localId);
	}

	/** {@inheritDoc} */
	@Override
	protected void doClose() throws IOException {
		super.doClose();

		Daos.closeSilently(updateLandingFishingTripFkStatement);
		updateLandingFishingTripFkStatement = null;
	}

	/* -- Internal methods -- */

	/**
	 * <p>
	 * removeLinkToLanding.
	 * </p>
	 * 
	 * @param connection
	 *            a {@link java.sql.Connection} object.
	 * @param fishingTripId
	 *            a int.
	 * @throws java.sql.SQLException
	 *             if any.
	 */
	protected void removeLinkToLanding(Connection connection, int fishingTripId) throws SQLException {
		if (updateLandingFishingTripFkStatement == null) {
			updateLandingFishingTripFkStatement = connection.prepareStatement(initUpdateLandingFishingTripFkStatement());
		}

		updateLandingFishingTripFkStatement.setInt(1, fishingTripId);
		updateLandingFishingTripFkStatement.executeUpdate();
	}

	/**
	 * <p>
	 * initUpdateLandingFishingTripFkStatement.
	 * </p>
	 * 
	 * @return a {@link java.lang.String} object.
	 */
	protected String initUpdateLandingFishingTripFkStatement() {
		return String.format("UPDATE %s SET %s = null WHERE %s = ?",
				TABLE_LANDING,
				LANDING_COLUMN_FISHING_TRIP_FK,
				LANDING_COLUMN_FISHING_TRIP_FK
				);
	}
}
