package fr.ifremer.adagio.synchro.intercept.data;

/*
 * #%L
 * SIH-Adagio :: Core for Allegro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.eventbus.Subscribe;
import fr.ifremer.adagio.synchro.intercept.data.internal.ExportFkRemoteIdInterceptor;
import fr.ifremer.adagio.synchro.intercept.data.internal.ExportPkRemoteIdInterceptor;
import fr.ifremer.adagio.synchro.intercept.data.internal.ImportEditedRowInterceptor;
import fr.ifremer.adagio.synchro.intercept.data.internal.ImportRemoteIdInterceptor;
import fr.ifremer.adagio.synchro.meta.data.DataSynchroTables;
import fr.ifremer.adagio.synchro.service.SynchroDirection;
import fr.ifremer.common.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.common.synchro.meta.SynchroJoinMetadata;
import fr.ifremer.common.synchro.meta.SynchroTableMetadata.DuplicateKeyStrategy;
import fr.ifremer.common.synchro.meta.SynchroTableMetadata;
import fr.ifremer.common.synchro.meta.event.CreateQueryEvent;
import fr.ifremer.common.synchro.meta.event.LoadJoinEvent;
import fr.ifremer.common.synchro.meta.event.LoadTableEvent;
import fr.ifremer.common.synchro.query.SynchroQueryBuilder;
import fr.ifremer.common.synchro.query.SynchroQueryOperator;
import fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration;
import org.apache.commons.collections.CollectionUtils;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import java.sql.Timestamp;
import java.util.Set;

/**
 * Manage only data table with columns 'id' AND 'remote_id' :
 * <ul>
 * <li>Set as root only if the table as an object_type (OBJECT_TYPE is need for join with PERSON_SESION_VESSEL)</li>
 * <li>when exporting to server, override 'update_date' column with a systimestamp</li>
 * </ul>
 * 
 * @author Benoit Lavenier (benoit.lavenier@e-is.pro)
 * @since 3.6.3
 */
public class DataTableInterceptor extends AbstractDataInterceptor {

	private Set<String> dataTableIncludes;
	private Timestamp systimestamp = null;

	/**
	 * <p>
	 * Constructor for DataTableInterceptor.
	 * </p>
	 */
	public DataTableInterceptor() {
		super();
		dataTableIncludes = DataSynchroTables.getImportTablesIncludes();
	}

	/** {@inheritDoc} */
	@Override
	public boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table) {
		if (CollectionUtils.isEmpty(dataTableIncludes)) {
			return false;
		}

		boolean isDataTable = dataTableIncludes.contains(table.getName());

		return isDataTable && hasColumns(table, getConfig().getColumnId(), getConfig().getColumnRemoteId());
	}

	/** {@inheritDoc} */
	@Override
	public SynchroInterceptorBase clone() {
		DataTableInterceptor result = (DataTableInterceptor) super.clone();
		result.dataTableIncludes = this.dataTableIncludes;
		result.systimestamp = this.systimestamp;
		return result;
	}

	/**
	 * <p>
	 * handleQuery.
	 * </p>
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.CreateQueryEvent} object.
	 */
	@Subscribe
	public void handleQuery(CreateQueryEvent e) {
		SynchroDirection direction = getConfig().getDirection();

		// IMPORT: Server DB -> Temp DB
		if (direction == SynchroDirection.IMPORT_SERVER2TEMP) {

			switch (e.queryName) {
			// Select queries : remove unsed columns
			case select:
			case selectFromUpdateDate:
				e.sql = SynchroQueryBuilder.newBuilder(e.queryName, e.sql)
						.deleteColumnIfExists(getConfig().getColumnRemoteId())
						.deleteColumnIfExists(getConfig().getColumnSynchronizationStatus())
						.build();
				break;
			case insert:
			case update: {
				SynchroQueryBuilder qb = SynchroQueryBuilder.newBuilder(e.queryName, e.sql)
						.deleteColumnIfExists(getConfig().getColumnRemoteId());
				if (qb.constainsColumn(getConfig().getColumnSynchronizationStatus())) {
					qb.setColumnValue(getConfig().getColumnSynchronizationStatus(), "'SYNC'");
				}
				e.sql = qb.build();
				break;
			}
			default:
				break;
			}
		}

		// IMPORT: Temp DB -> Local DB
		else if (direction == SynchroDirection.IMPORT_TEMP2LOCAL) {

			switch (e.queryName) {
			// Select queries : remove unsed columns
			case select:
			case selectFromUpdateDate:
				e.sql = SynchroQueryBuilder.newBuilder(e.queryName, e.sql)
						.deleteColumnIfExists(getConfig().getColumnRemoteId())
						.deleteColumnIfExists(getConfig().getColumnSynchronizationStatus())
						.build();
				break;
			case insert: {
				SynchroQueryBuilder qb = SynchroQueryBuilder.newBuilder(e.queryName, e.sql)
						.deleteColumn(getConfig().getColumnRemoteId())
						.replaceColumn(getConfig().getColumnId(), getConfig().getColumnRemoteId());
				if (qb.constainsColumn(getConfig().getColumnSynchronizationStatus())) {
					qb.setColumnValue(getConfig().getColumnSynchronizationStatus(), "'SYNC'");
				}
				qb.addColumn(getConfig().getColumnId(), e.source.getSelectSequenceNextValString());
				e.sql = qb.build();
				break;
			}
			case update: {
				SynchroQueryBuilder qb = SynchroQueryBuilder.newBuilder(e.queryName, e.sql)
						.deleteColumn(getConfig().getColumnRemoteId())
						.replaceColumn(getConfig().getColumnId(), getConfig().getColumnRemoteId());
				if (qb.constainsColumn(getConfig().getColumnSynchronizationStatus())) {
					qb.setColumnValue(getConfig().getColumnSynchronizationStatus(), "'SYNC'");
				}
				e.sql = qb.build();
				break;
			}
			default:
				break;
			}
		}

		// EXPORT: Local DB -> Temp DB
		else if (direction == SynchroDirection.EXPORT_LOCAL2TEMP) {

			// Nothing to do ?
		}

		// EXPORT: Temp DB -> Server DB
		else if (direction == SynchroDirection.EXPORT_TEMP2SERVER) {

			switch (e.queryName) {
			case selectMaxUpdateDate:
				// Do not try to retrieve max update_date on server
				e.sql = SynchroQueryBuilder.newBuilder(e.queryName, e.sql)
						.addWhere(SynchroQueryOperator.AND, "1=2")
						.build();
				break;
			// Select queries : remove unsed columns
			case select:
			case selectFromUpdateDate:
				e.sql = SynchroQueryBuilder.newBuilder(e.queryName, e.sql)
						.deleteColumnIfExists(getConfig().getColumnSynchronizationStatus())
						.deleteColumnIfExists(getConfig().getColumnRemoteId())
						.build();
				break;
			case insert: {
				SynchroQueryBuilder qb = SynchroQueryBuilder.newBuilder(e.queryName, e.sql)
						.deleteColumnIfExists(getConfig().getColumnSynchronizationStatus())
						.deleteColumnIfExists(getConfig().getColumnRemoteId());
				e.sql = qb.build();
				break;
			}
			case update: {
				SynchroQueryBuilder qb = SynchroQueryBuilder.newBuilder(e.queryName, e.sql)
						.deleteColumnIfExists(getConfig().getColumnSynchronizationStatus())
						.deleteColumnIfExists(getConfig().getColumnRemoteId());
				e.sql = qb.build();
				break;
			}
			default:
				break;
			}
		}
	}

	/**
	 * <p>
	 * handleTableLoad.
	 * </p>
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.LoadTableEvent} object.
	 */
	@Subscribe
	public void handleTableLoad(LoadTableEvent e) {

		SynchroTableMetadata table = e.table;
		String remoteIdColumn = getConfig().getColumnRemoteId();
		String idColumn = getConfig().getColumnId();
		String updateDateColumn = getConfig().getColumnUpdateDate();

		// Set table as root if synchronization status column exists
		boolean hasSynchronizationStatusColumn = table.getColumnIndex(getConfig().getColumnSynchronizationStatus()) != -1;
		boolean hasUpdateDateColumn = table.getColumnIndex(updateDateColumn) != -1;
		boolean hasObjectType = ObjectTypeHelper.getObjectTypeFromTableName(table.getName()) != null;
		boolean hasRemoteIdColumn = table.getColumnIndex(remoteIdColumn) != -1;

		// Set isRoot if not already done
		// for example, DeletedItemHistoryInterceptor already set as true, so do not override this to false !
		if (!table.isRoot() && hasSynchronizationStatusColumn && hasObjectType) {
			// BLA 23/09/2014 : avoid Landing to be set as a root table
			// table.setRoot(true);
		}

		SynchroDirection direction = getConfig().getDirection();

		// Temp DB -> Local DB
		if (hasRemoteIdColumn && direction == SynchroDirection.IMPORT_TEMP2LOCAL) {
			// Define a natural id on REMOTE_ID, if not already define
			// (could have been override - e.g. ProduceSortingMeasurementInterceptor)
			if (!table.hasUniqueConstraint(remoteIdColumn)) {
				table.addUniqueConstraint(remoteIdColumn, ImmutableList.of(remoteIdColumn), DuplicateKeyStrategy.REPLACE);
			}

			// Make sure status = SYNC before override a row
			if (!getConfig().isForceEditedRowOverride()
					&& hasSynchronizationStatusColumn
					&& hasUpdateDateColumn) {
				ImportEditedRowInterceptor synchronizationStatusInterceptor = new ImportEditedRowInterceptor(
						getConfig(),
						table.getName().toLowerCase(),
						table.getSelectColumnIndex(idColumn),
						table.getSelectColumnIndex(updateDateColumn)
						);
				table.addInterceptor(synchronizationStatusInterceptor);
			}
		}

		// Temp DB -> Server DB
		else if (hasRemoteIdColumn && direction == SynchroDirection.EXPORT_TEMP2SERVER) {
			if (systimestamp == null) {
				systimestamp = checkAndGetSystemTimestamp(getConfig());
			}

			// Define a natural id on ID column, if not already define
			// (could have been override - e.g. ProduceSortingMeasurementInterceptor)
			if (!table.hasUniqueConstraint(remoteIdColumn)) {
				table.addUniqueConstraint(remoteIdColumn, ImmutableList.of(idColumn), DuplicateKeyStrategy.REPLACE);
			}

			int updateDateColumnIndex = table.getSelectColumnIndex(getConfig().getColumnUpdateDate());
			boolean hasUpdateDate = updateDateColumnIndex != -1;

			ExportPkRemoteIdInterceptor remoteIdInterceptor = new ExportPkRemoteIdInterceptor(
					getConfig(),
					table.getName(),
					idColumn,
					table.getSelectColumnIndex(idColumn),
					hasSynchronizationStatusColumn,
					hasUpdateDate,
					updateDateColumnIndex,
					systimestamp,
					table.hasChildJoins());

			table.addInterceptor(remoteIdInterceptor);
		}

	}

	/**
	 * <p>
	 * handleJoinLoad.
	 * </p>
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.LoadJoinEvent} object.
	 */
	@Subscribe
	public void handleJoinLoad(LoadJoinEvent e) {
		SynchroJoinMetadata join = e.join;
		// Do not apply if mirror database
		if (!join.isValid()) {
			return;
		}

		SynchroTableMetadata fkTable = join.getFkTable();
		SynchroDirection direction = getConfig().getDirection();

		// If the FK table is the current table, or not processed by this interceptor
		if (fkTable == e.source || isNotInterceptedTable(fkTable)) {

			if (direction == SynchroDirection.IMPORT_TEMP2LOCAL) {
				SynchroTableMetadata pkTable = join.getPkTable();

				// Create and configure a interceptor, to rewrite remote id
				String pkTableName = pkTable.getName().toLowerCase();
				String fkColumnName = join.getFkColumn().getName().toLowerCase();
				int fkColumnIndex = fkTable.getSelectColumnIndex(fkColumnName);

				ImportRemoteIdInterceptor remoteIdInterceptor = new ImportRemoteIdInterceptor(
						getConfig(),
						pkTableName,
						fkColumnName,
						fkColumnIndex,
						join.getFkColumn().isNullable());

				if (!fkTable.containsInterceptor(remoteIdInterceptor)) {
					fkTable.addInterceptor(remoteIdInterceptor);
				}
			}

			else if (direction == SynchroDirection.EXPORT_TEMP2SERVER) {
				SynchroTableMetadata pkTable = join.getPkTable();

				// Create and configure a interceptor, to rewrite remote id
				String pkTableName = pkTable.getName().toLowerCase();
				String fkColumnName = join.getFkColumn().getName().toLowerCase();
				int fkColumnIndex = fkTable.getSelectColumnIndex(fkColumnName);

				ExportFkRemoteIdInterceptor remoteIdInterceptor = new ExportFkRemoteIdInterceptor(
						getConfig(),
						pkTableName,
						fkColumnName,
						fkColumnIndex,
						join.getFkColumn().isNullable());

				if (!fkTable.containsInterceptor(remoteIdInterceptor)) {
					fkTable.addInterceptor(remoteIdInterceptor);
				}

			}

		}

	}

	/* -- Internal methods -- */

	/**
	 * <p>
	 * isNotInterceptedTable.
	 * </p>
	 * 
	 * @param table
	 *            a {@link fr.ifremer.common.synchro.meta.SynchroTableMetadata} object.
	 * @return a boolean.
	 */
	protected boolean isNotInterceptedTable(SynchroTableMetadata table) {

		Set<String> columnNames = table.getColumnNames();

		return !columnNames.contains(getConfig().getColumnId())
				|| !columnNames.contains(getConfig().getColumnRemoteId());
	}

	/* -- Protected methods -- */

	/**
	 * <p>
	 * checkAndGetSystemTimestamp.
	 * </p>
	 * 
	 * @param configuration
	 *            a {@link fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration} object.
	 * @return a {@link java.sql.Timestamp} object.
	 */
	protected Timestamp checkAndGetSystemTimestamp(SynchroDatabaseConfiguration configuration) {
		Timestamp systimestamp = configuration.getSystemTimestamp();
		Preconditions.checkNotNull(systimestamp,
				String.format("Could not found system timestamp in database configuration. This is need for %s", getClass().getSimpleName()));
		return systimestamp;
	}
}
