package fr.ifremer.adagio.synchro.intercept.data;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.eventbus.Subscribe;
import fr.ifremer.adagio.synchro.intercept.data.internal.ExportPkRemoteIdInterceptor;
import fr.ifremer.adagio.synchro.intercept.data.internal.TableWithAttachmentsInterceptor;
import fr.ifremer.adagio.synchro.meta.data.DataSynchroTables;
import fr.ifremer.common.synchro.SynchroTechnicalException;
import fr.ifremer.common.synchro.dao.Daos;
import fr.ifremer.common.synchro.dao.SynchroBaseDao;
import fr.ifremer.common.synchro.dao.SynchroTableDao;
import fr.ifremer.common.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.common.synchro.intercept.SynchroOperationRepository;
import fr.ifremer.adagio.synchro.intercept.data.internal.ExportFkRemoteIdInterceptor;
import fr.ifremer.adagio.synchro.intercept.data.internal.ImportRemoteIdInterceptor;
import fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.common.synchro.meta.SynchroTableMetadata;
import fr.ifremer.common.synchro.meta.event.LoadJoinEvent;
import fr.ifremer.common.synchro.meta.event.LoadTableEvent;
import fr.ifremer.adagio.synchro.service.SynchroDirection;
import fr.ifremer.adagio.synchro.service.data.DataSynchroDatabaseConfiguration;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.List;

/**
 * <p>
 * BatchInterceptor class.
 * </p>
 * 
 * @author Lionel Touseau (lionel.touseau@e-is.pro)
 * @since 3.8.1
 */
public class BatchInterceptor extends AbstractDataInterceptor {

	private static final String TABLE_BATCH = DataSynchroTables.BATCH.name();
	private static final String TABLE_OPERATION = DataSynchroTables.OPERATION.name();
	private static final String TABLE_QUANTIFICATION_MEASUREMENT = DataSynchroTables.QUANTIFICATION_MEASUREMENT.name();
	private static final String TABLE_SORTING_MEASUREMENT = DataSynchroTables.SORTING_MEASUREMENT.name();
	private static final String COLUMN_ID = "id";
	private static final String COLUMN_ROOT_BATCH = "root_batch_fk";
	private static final String COLUMN_PARENT_BATCH = "parent_batch_fk";
	private static final String COLUMN_OPERATION_CATCH_BATCH_FK = "catch_batch_fk";
	private static final String COLUMN_QM_BATCH_FK = "batch_fk";
	private static final String COLUMN_SM_BATCH_FK = "sorting_batch_fk";

	private int rootBatchFkColumnIndex = -1;
	private int parentBatchFkColumnIndex = -1;
	private int batchIdColumnIndex = -1;

	private PreparedStatement updateOperationCatchBatchFkStatement = null;
	private PreparedStatement deleteChildrenBatchesByRootBatchFkStatement = null;
	private PreparedStatement updateParentBatchFkStatement = null;
	private PreparedStatement deleteQuantificationMeasurementByBatchFkStatement = null;
	private PreparedStatement deleteSortingMeasurementByBatchFkStatement = null;

	/**
	 * <p>
	 * Constructor for BatchInterceptor.
	 * </p>
	 */
	public BatchInterceptor() {
		super();
		setEnableOnWrite(true);
	}

	/** {@inheritDoc} */
	@Override
	public SynchroInterceptorBase clone() {
		BatchInterceptor newBean = (BatchInterceptor) super.clone();
		newBean.batchIdColumnIndex = this.batchIdColumnIndex;
		newBean.rootBatchFkColumnIndex = this.rootBatchFkColumnIndex;
		newBean.parentBatchFkColumnIndex = this.parentBatchFkColumnIndex;
		return newBean;
	}

	/** {@inheritDoc} */
	@Override
	public boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table) {
		return TABLE_BATCH.equalsIgnoreCase(table.getName());
	}

	/**
	 * <p>
	 * handleTableLoad.
	 * </p>
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.LoadTableEvent} object.
	 */
	@Subscribe
	public void handleTableLoad(LoadTableEvent e) {
		// batches are not root
		e.table.setRoot(false);

		// get column indexes
		batchIdColumnIndex = e.table.getSelectColumnIndex(COLUMN_ID);
		rootBatchFkColumnIndex = e.table.getSelectColumnIndex(COLUMN_ROOT_BATCH);
		parentBatchFkColumnIndex = e.table.getSelectColumnIndex(COLUMN_PARENT_BATCH);
	}

	/**
	 * Disables parent-child relationships when loading
	 * 
	 * @param e
	 *            a {@link fr.ifremer.common.synchro.meta.event.LoadJoinEvent} object.
	 */
	@Subscribe
	public void handleJoinLoad(LoadJoinEvent e) {
		if (!e.join.isValid()) {
			return;
		}

		// Disable reverse Operation-CatchBatch relationship
		// (OPERATION should not be seen as a child of BATCH)
		if (TABLE_OPERATION.equalsIgnoreCase(e.join.getTargetTable().getName())) {
			// this parent link can be disabled
			e.join.setIsValid(false);
		}

		// Disable join between BATCH and BATCH (root_batch_fk, parent_batch_fk)
		String targetTableName = e.join.getTargetTable().getName();
		String targetColumnName = e.join.getTargetColumn().getName();
		boolean isJoinToRootOrParentTable = TABLE_BATCH.equalsIgnoreCase(targetTableName)
				&& (COLUMN_ROOT_BATCH.equalsIgnoreCase(targetColumnName)
				|| COLUMN_PARENT_BATCH.equalsIgnoreCase(targetColumnName));

		if (isJoinToRootOrParentTable) {
			// disable join
			e.join.setIsValid(false);

			// create ExportFk interceptor for root_batch_fk and parent_batch_fk when exporting from temp 2 server
			// ensure we register only once
			if (getConfig().getDirection() == SynchroDirection.EXPORT_TEMP2SERVER) {

				SynchroTableMetadata batchTable = e.join.getFkTable();

				// if (!hasRemoteIdInterceptor(ExportFkRemoteIdInterceptor.class, targetColumnName, batchTable)) {
				// if it does not exist
				// Create and configure an interceptor, to rewrite remote id
				ExportFkRemoteIdInterceptor batchRemoteIdInterceptor = new ExportFkRemoteIdInterceptor(
						getConfig(),
						TABLE_BATCH.toLowerCase(),
						targetColumnName.toLowerCase(),
						batchTable.getSelectColumnIndex(targetColumnName),
						e.join.getFkColumn().isNullable());

				if (!batchTable.containsInterceptor(batchRemoteIdInterceptor)) {
					batchTable.addInterceptor(batchRemoteIdInterceptor);
				}
				// }

			}

			// create ImportRemoteId Interceptor for root_batch_fk and parent_batch_fk when importing from temp 2 local
			// ensure we register only once
			if (getConfig().getDirection() == SynchroDirection.IMPORT_TEMP2LOCAL) {

				SynchroTableMetadata batchTable = e.join.getPkTable();

				// Create and configure an interceptor, to rewrite remote id
				ImportRemoteIdInterceptor batchRemoteIdInterceptor = new ImportRemoteIdInterceptor(
						getConfig(),
						TABLE_BATCH.toLowerCase(),
						targetColumnName.toLowerCase(),
						batchTable.getSelectColumnIndex(targetColumnName),
						e.join.getFkColumn().isNullable());

				// Add it (only if not exist)
				if (!batchTable.containsInterceptor(batchRemoteIdInterceptor)) {
					batchTable.addInterceptor(batchRemoteIdInterceptor);
				}
			}
		}

	}

	/** {@inheritDoc} */
	@Override
	protected void doOnWrite(Object[] data,
			List<Object> pk,
			SynchroTableDao sourceDao,
			SynchroTableDao targetDao,
			SynchroOperationRepository buffer,
			boolean insert)
			throws SQLException {

		// If root entity: no transformation
		if (buffer == null) {
			return;
		}

		// delay rootBatchFk write
		Object rootBatchFk = data[rootBatchFkColumnIndex];

		// root (catch) batch
		if (rootBatchFk == null) {
			// manually add root batch children
			Object rootBatchId = data[batchIdColumnIndex];
			buffer.addChildToUpdateFromOneColumn(TABLE_BATCH, COLUMN_ROOT_BATCH, rootBatchId);
		}

		// also delay parentBatchFk write
		Object parentBatchFk = data[parentBatchFkColumnIndex];
		if (parentBatchFk != null) {
			if (enableIntegrityConstraints) {
				// set as Null, and update later
				data[parentBatchFkColumnIndex] = null;

				buffer.addMissingColumnUpdate(COLUMN_PARENT_BATCH, pk, parentBatchFk);
			}
		}

	}

	/** {@inheritDoc} */
	@Override
	protected void doOnDelete(List<Object> pk, SynchroTableDao sourceDao, SynchroTableDao targetDao, SynchroOperationRepository buffer)
			throws SQLException {

		int localId = Integer.parseInt(pk.get(0).toString());
		// delete children batches (if root - first will always be root since we get there by operation, or by deleted
		// item history)
		deleteChildrenBatchesByRootBatch(targetDao, localId);
	}

	/** {@inheritDoc} */
	@Override
	protected void doClose() throws IOException {
		super.doClose();

		Daos.closeSilently(updateOperationCatchBatchFkStatement);
		Daos.closeSilently(deleteChildrenBatchesByRootBatchFkStatement);
		updateOperationCatchBatchFkStatement = null;
		deleteChildrenBatchesByRootBatchFkStatement = null;
	}

	/* -- Internal methods -- */

	/**
	 * <p>
	 * deleteChildrenBatchesByRootBatch.
	 * </p>
	 * 
	 * @param targetDao
	 *            a {@link fr.ifremer.common.synchro.dao.SynchroBaseDao} object.
	 * @param rootBatchId
	 *            a int.
	 * @throws java.sql.SQLException
	 *             if any.
	 */
	protected void deleteChildrenBatchesByRootBatch(SynchroBaseDao targetDao, int rootBatchId) throws SQLException {

		// remove reference to catchBatch by OPERATION.CATCH_BATCH_FK
		removeLinkToOperation(targetDao, rootBatchId);

		// remove link to parent batch (to avoid integrity constraint violation)
		removeLinkToParentBatchByRootBatch(targetDao, rootBatchId);

		// delete links to batches QuantificationMeasurement and SortingMeasurement
		deleteQuantificationMeasurementByRootBatch(targetDao, rootBatchId);
		deleteSortingMeasurementByRootBatch(targetDao, rootBatchId);

		if (deleteChildrenBatchesByRootBatchFkStatement == null || deleteChildrenBatchesByRootBatchFkStatement.isClosed()) {
			deleteChildrenBatchesByRootBatchFkStatement = targetDao.getPreparedStatement(getDeleteBatchByRootBatchFkQuery());
		}
		deleteChildrenBatchesByRootBatchFkStatement.setInt(1, rootBatchId);
		deleteChildrenBatchesByRootBatchFkStatement.executeUpdate();
	}

	/**
	 * <p>
	 * removeLinkToOperation.
	 * </p>
	 * 
	 * @param targetDao
	 *            a {@link fr.ifremer.common.synchro.dao.SynchroBaseDao} object.
	 * @param catchBatchId
	 *            a int.
	 * @throws java.sql.SQLException
	 *             if any.
	 */
	protected void removeLinkToOperation(SynchroBaseDao targetDao, int catchBatchId) throws SQLException {
		if (updateOperationCatchBatchFkStatement == null || updateOperationCatchBatchFkStatement.isClosed()) {

			updateOperationCatchBatchFkStatement = targetDao.getPreparedStatement(getUpdateOperationCatchBatchFkQuery());
		}

		updateOperationCatchBatchFkStatement.setInt(1, catchBatchId);
		updateOperationCatchBatchFkStatement.executeUpdate();
	}

	/**
	 * <p>
	 * removeLinkToParentBatchByRootBatch.
	 * </p>
	 * 
	 * @param targetDao
	 *            a {@link fr.ifremer.common.synchro.dao.SynchroBaseDao} object.
	 * @param catchBatchId
	 *            a int.
	 * @throws java.sql.SQLException
	 *             if any.
	 */
	protected void removeLinkToParentBatchByRootBatch(SynchroBaseDao targetDao, int catchBatchId) throws SQLException {
		if (updateParentBatchFkStatement == null || updateParentBatchFkStatement.isClosed()) {

			updateParentBatchFkStatement = targetDao.getPreparedStatement(getUpdateParentBatchFkQuery());
		}

		updateParentBatchFkStatement.setInt(1, catchBatchId);
		updateParentBatchFkStatement.executeUpdate();
	}

	/**
	 * <p>
	 * deleteQuantificationMeasurementByRootBatch.
	 * </p>
	 * 
	 * @param targetDao
	 *            a {@link fr.ifremer.common.synchro.dao.SynchroBaseDao} object.
	 * @param catchBatchId
	 *            a int.
	 * @throws java.sql.SQLException
	 *             if any.
	 */
	protected void deleteQuantificationMeasurementByRootBatch(SynchroBaseDao targetDao, int catchBatchId) throws SQLException {
		if (deleteQuantificationMeasurementByBatchFkStatement == null || deleteQuantificationMeasurementByBatchFkStatement.isClosed()) {

			deleteQuantificationMeasurementByBatchFkStatement = targetDao.getPreparedStatement(getDeleteQuantificationMeasurementByBatchFkQuery());
		}

		deleteQuantificationMeasurementByBatchFkStatement.setInt(1, catchBatchId);
		deleteQuantificationMeasurementByBatchFkStatement.setInt(2, catchBatchId);
		deleteQuantificationMeasurementByBatchFkStatement.executeUpdate();
	}

	/**
	 * <p>
	 * deleteSortingMeasurementByRootBatch.
	 * </p>
	 * 
	 * @param targetDao
	 *            a {@link fr.ifremer.common.synchro.dao.SynchroBaseDao} object.
	 * @param catchBatchId
	 *            a int.
	 * @throws java.sql.SQLException
	 *             if any.
	 */
	protected void deleteSortingMeasurementByRootBatch(SynchroBaseDao targetDao, int catchBatchId) throws SQLException {
		if (deleteSortingMeasurementByBatchFkStatement == null || deleteSortingMeasurementByBatchFkStatement.isClosed()) {

			deleteSortingMeasurementByBatchFkStatement = targetDao.getPreparedStatement(getDeleteSortingMeasurementByBatchFkQuery());
		}

		deleteSortingMeasurementByBatchFkStatement.setInt(1, catchBatchId);
		deleteSortingMeasurementByBatchFkStatement.executeUpdate();
	}

	/**
	 * <p>
	 * getUpdateOperationCatchBatchFkQuery.
	 * </p>
	 * 
	 * @return a {@link java.lang.String} object.
	 */
	protected String getUpdateOperationCatchBatchFkQuery() {
		return String.format("UPDATE %s SET %s = null WHERE %s = ?",
				TABLE_OPERATION,
				COLUMN_OPERATION_CATCH_BATCH_FK,
				COLUMN_OPERATION_CATCH_BATCH_FK
				);
	}

	/**
	 * <p>
	 * getUpdateParentBatchFkQuery.
	 * </p>
	 * 
	 * @return a {@link java.lang.String} object.
	 */
	protected String getUpdateParentBatchFkQuery() {
		return String.format("UPDATE %s SET %s = null WHERE %s = ?",
				TABLE_BATCH,
				COLUMN_PARENT_BATCH,
				COLUMN_ROOT_BATCH
				);
	}

	/**
	 * requires 2 bound parameters: root_batch_fk (twice)
	 * 
	 * @return query that updates QuantificationMeasurement.batchFk=null for root batch and its children
	 */
	protected String getDeleteQuantificationMeasurementByBatchFkQuery() {
		return String.format("DELETE FROM %s"
				+ " WHERE %s"
				+ " IN (SELECT %s FROM %s"
				+ " WHERE %s = ?"
				+ " OR %s = ?)",
				TABLE_QUANTIFICATION_MEASUREMENT,
				COLUMN_QM_BATCH_FK,
				COLUMN_ID,
				TABLE_BATCH,
				COLUMN_ID,
				COLUMN_ROOT_BATCH
				);
	}

	/**
	 * requires 1 bound parameter: root_batch_fk
	 * 
	 * @return query that updates SortingMeasurement.sortingBatchFk=null for root batch children
	 */
	protected String getDeleteSortingMeasurementByBatchFkQuery() {
		return String.format("DELETE FROM %s"
				+ " WHERE %s"
				+ " IN (SELECT %s FROM %s"
				+ " WHERE %s = ?)",
				TABLE_SORTING_MEASUREMENT,
				COLUMN_SM_BATCH_FK,
				COLUMN_ID,
				TABLE_BATCH,
				COLUMN_ROOT_BATCH
				);
	}

	/**
	 * <p>
	 * getDeleteBatchByRootBatchFkQuery.
	 * </p>
	 * 
	 * @return a {@link java.lang.String} object.
	 */
	protected String getDeleteBatchByRootBatchFkQuery() {
		return String.format("DELETE FROM %s WHERE %s = ? AND IS_CATCH_BATCH<>1",
				TABLE_BATCH,
				COLUMN_ROOT_BATCH
				);
	}

}
