package fr.ifremer.adagio.synchro.intercept.data;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.collect.ImmutableSet;
import fr.ifremer.adagio.synchro.intercept.AbstractSynchroInterceptor;
import fr.ifremer.adagio.synchro.service.SynchroDirection;
import fr.ifremer.adagio.synchro.service.data.DataSynchroDatabaseConfiguration;
import fr.ifremer.common.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata;
import org.apache.commons.collections.CollectionUtils;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import java.util.Collection;
import java.util.Set;

/**
 * <p>
 * Abstract AbstractDataInterceptor class.
 * </p>
 * 
 */
public abstract class AbstractDataInterceptor extends AbstractSynchroInterceptor<DataSynchroDatabaseConfiguration> {

	protected boolean enableIntegrityConstraints = true;

	/**
	 * <p>
	 * Constructor for AbstractDataInterceptor.
	 * </p>
	 */
	public AbstractDataInterceptor() {
		super();
	}

	/**
	 * <p>
	 * Constructor for AbstractDataInterceptor.
	 * </p>
	 * 
	 * @param directions
	 *            a {@link fr.ifremer.adagio.synchro.service.SynchroDirection} object.
	 */
	public AbstractDataInterceptor(SynchroDirection... directions) {
		super(directions);
	}

	/**
	 * <p>
	 * Constructor for AbstractDataInterceptor.
	 * </p>
	 * 
	 * @param tableNames
	 *            a {@link java.util.Set} object.
	 */
	public AbstractDataInterceptor(Set<String> tableNames) {
		super(tableNames);
	}

	/**
	 * <p>
	 * Constructor for AbstractDataInterceptor.
	 * </p>
	 * 
	 * @param tableNames
	 *            a {@link java.util.Set} object.
	 * @param directions
	 *            a {@link fr.ifremer.adagio.synchro.service.SynchroDirection} object.
	 */
	public AbstractDataInterceptor(Set<String> tableNames, SynchroDirection... directions) {
		super(tableNames, directions);
	}

	/**
	 * <p>
	 * Constructor for AbstractDataInterceptor.
	 * </p>
	 * 
	 * @param tableName
	 *            a {@link java.lang.String} object.
	 * @param directions
	 *            a {@link fr.ifremer.adagio.synchro.service.SynchroDirection} object.
	 */
	public AbstractDataInterceptor(String tableName, SynchroDirection... directions) {
		this(ImmutableSet.of(tableName), directions);
	}

	/**
	 * <p>
	 * Constructor for AbstractDataInterceptor.
	 * </p>
	 * 
	 * @param tableName
	 *            a {@link java.lang.String} object.
	 */
	public AbstractDataInterceptor(String tableName) {
		this(ImmutableSet.of(tableName));
	}

	/** {@inheritDoc} */
	@Override
	public void init(DataSynchroDatabaseConfiguration config) {
		super.init(config);
		enableIntegrityConstraints = config.getDirection() == SynchroDirection.EXPORT_TEMP2SERVER
				|| config.getDirection() == SynchroDirection.IMPORT_TEMP2LOCAL;
	}

	/** {@inheritDoc} */
	@Override
	public SynchroInterceptorBase clone() {
		AbstractDataInterceptor newBean = (AbstractDataInterceptor) super.clone();
		newBean.enableIntegrityConstraints = this.enableIntegrityConstraints;
		return newBean;
	}

	/** {@inheritDoc} */
	@Override
	protected Class<DataSynchroDatabaseConfiguration> getConfigClass() {
		return DataSynchroDatabaseConfiguration.class;
	}

	/** {@inheritDoc} */
	@Override
	public boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table) {
		return true;
	}

	/**
	 * <p>
	 * createPkFilter.
	 * </p>
	 * 
	 * @param tableName
	 *            a {@link java.lang.String} object.
	 * @return a {@link java.lang.String} object.
	 */
	protected String createPkFilter(String tableName, String pkName) {
		if (getConfig().getPkIncludes() == null
				|| getConfig().getPkIncludes().isEmpty()) {
			return null;
		}

		Collection<String> pkStrs = getConfig()
				.getPkIncludes()
				.get(tableName.toUpperCase());

		if (CollectionUtils.isEmpty(pkStrs)) {
			// There is filter by PK, but not on this table
			// so exclude this table
			return "1=2";
		}

		StringBuilder inBuilder = new StringBuilder();
		for (String pkStr : pkStrs) {
			inBuilder.append(',').append(pkStr);
		}

		return String.format("t.%s IN (%s)",
				pkName,
				inBuilder.substring(1)
				);
	}
}
