package fr.ifremer.adagio.synchro.intercept;

/*
 * #%L
 * SIH-Adagio :: Core for Allegro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2016 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import fr.ifremer.adagio.synchro.service.AbstractSynchroDatabaseConfiguration;
import fr.ifremer.adagio.synchro.service.SynchroDirection;
import fr.ifremer.common.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.common.synchro.service.SynchroDatabaseConfiguration;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import java.util.Arrays;
import java.util.Set;

/**
 * <p>
 * Abstract AbstractSynchroInterceptor class.
 * </p>
 * 
 */
public abstract class AbstractSynchroInterceptor<T extends AbstractSynchroDatabaseConfiguration> extends SynchroInterceptorBase {

	private SynchroDatabaseMetadata meta;
	private Set<SynchroDirection> allowDirections = Sets.newHashSet();
	private Set<String> allowTables;

	/**
	 * <p>
	 * Constructor for AbstractSynchroInterceptor.
	 * </p>
	 */
	public AbstractSynchroInterceptor() {
		// All direction are allowed
		this.allowDirections = null;
		// All table are allowed
		this.allowTables = null;
	}

	/**
	 * <p>
	 * Constructor for AbstractSynchroInterceptor.
	 * </p>
	 * 
	 * @param directions
	 *            a {@link fr.ifremer.adagio.synchro.service.SynchroDirection} object.
	 */
	public AbstractSynchroInterceptor(SynchroDirection... directions) {
		Preconditions.checkArgument(directions != null && directions.length > 0);

		// Fill allowed directions
		this.allowDirections = ImmutableSet.copyOf(directions);
		// All table are allowed
		this.allowTables = null;
	}

	/**
	 * <p>
	 * Constructor for AbstractSynchroInterceptor.
	 * </p>
	 * 
	 * @param tableIncludes
	 *            a {@link java.util.Set} object.
	 * @param directions
	 *            a {@link fr.ifremer.adagio.synchro.service.SynchroDirection} object.
	 */
	public AbstractSynchroInterceptor(Set<String> tableIncludes, SynchroDirection... directions) {
		Preconditions.checkArgument(tableIncludes != null && tableIncludes.size() > 0);
		Preconditions.checkArgument(directions != null && directions.length > 0);

		this.allowTables = ImmutableSet.copyOf(tableIncludes);

		// Fill allowed directions
		this.allowDirections = ImmutableSet.copyOf(directions);
	}

	/**
	 * <p>
	 * Constructor for AbstractSynchroInterceptor.
	 * </p>
	 * 
	 * @param tableIncludes
	 *            a {@link java.util.Set} object.
	 */
	public AbstractSynchroInterceptor(Set<String> tableIncludes) {
		Preconditions.checkArgument(tableIncludes != null && tableIncludes.size() > 0);

		this.allowTables = ImmutableSet.copyOf(tableIncludes);

		// All direction are allowed
		this.allowDirections = null;
	}

	/** {@inheritDoc} */
	@Override
	public SynchroInterceptorBase clone() {
		AbstractSynchroInterceptor result = (AbstractSynchroInterceptor) super.clone();
		result.meta = this.meta;
		result.allowDirections = allowDirections;
		result.allowTables = allowTables;
		return result;
	}

	/** {@inheritDoc} */
	@Override
	public final boolean apply(SynchroDatabaseConfiguration config) {
		boolean result = getConfigClass().isInstance(config)
				/* super is need to store the config into a field */
				&& super.apply(config);

		if (result) {
			// Init the interceptor
			init((T) config);
		}
		return result;
	}

	/**
	 * <p>
	 * getConfigClass.
	 * </p>
	 * 
	 * @return a {@link java.lang.Class} object.
	 */
	protected abstract Class<T> getConfigClass();

	/** {@inheritDoc} */
	@Override
	public final boolean apply(SynchroDatabaseMetadata meta, TableMetadata table) {
		Preconditions.checkNotNull(meta);
		if (this.meta != null) {
			Preconditions.checkState(this.meta == meta);
		}

		// Store meta and configuration
		else {
			this.meta = meta;
		}

		return (allowDirections == null || allowDirections.contains(getConfig().getDirection()))
				&& (allowTables == null || allowTables.contains(table.getName()))
				&& doApply(meta, table);
	}

	/**
	 * <p>
	 * doApply.
	 * </p>
	 * 
	 * @param meta
	 *            a {@link fr.ifremer.common.synchro.meta.SynchroDatabaseMetadata} object.
	 * @param table
	 *            a {@link org.hibernate.tool.hbm2ddl.TableMetadata} object.
	 * @return a boolean.
	 */
	public abstract boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table);

	/* -- delegate methods -- */

	/**
	 * <p>
	 * getConfig.
	 * </p>
	 * 
	 * @return a T object.
	 */
	public T getConfig() {
		return (T) getDefaultDatabaseConfiguration();
	}

	/* -- internal methods -- */

	/**
	 * <p>
	 * init.
	 * </p>
	 * 
	 * @param config
	 *            a T object.
	 */
	protected void init(T config) {
		// could be override
	}

	/**
	 * <p>
	 * checkAndGetPersonId.
	 * </p>
	 * 
	 * @return a int.
	 */
	protected int checkAndGetPersonId() {
		int personId = getConfig().getPersonId();
		Preconditions
				.checkNotNull(
						personId,
						String.format(
								"Could not retrieve person id (in database configuration). %s need a not null personId.",
								getClass().getSimpleName()
								));
		return personId;
	}

	/**
	 * <p>
	 * checkAndGetPersonSessionId.
	 * </p>
	 * 
	 * @return a int.
	 */
	protected int checkAndGetPersonSessionId() {
		Integer personSessionId = getConfig().getPersonSessionId();
		Preconditions
				.checkNotNull(
						personSessionId,
						String.format(
								"Could not retrieve person session (no personSessionId in context). %s need a not null personSessionId.",
								getClass().getSimpleName()));

		return personSessionId;

	}

	/**
	 * <p>
	 * isInDirections.
	 * </p>
	 * 
	 * @param allowDirections
	 *            a {@link fr.ifremer.adagio.synchro.service.SynchroDirection} object.
	 * @return a boolean.
	 */
	protected boolean isInDirections(SynchroDirection... allowDirections) {
		return Arrays.asList(allowDirections).contains(getConfig().getDirection());
	}
}
