package fr.ifremer.adagio.synchro.service;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.sql.Timestamp;
import java.util.Arrays;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.cfg.AvailableSettings;
import org.hibernate.cfg.Environment;
import org.hibernate.dialect.Dialect;
import org.hibernate.internal.util.config.ConfigurationHelper;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.Sets;

import fr.ifremer.adagio.synchro.config.SynchroConfiguration;

public class SynchroDatabaseConfiguration {

	/**
	 * is database the targetted database ?
	 */
	public static String IS_TARGET_DATABASE = "synchro.database.isTarget";

	/**
	 * is database in readonly ?
	 */
	public static String READ_ONLY = "synchro.database.readonly";

	/**
	 * Enable or not load full metadata.
	 */
	public static String ENABLE_FULL_METADATA = "synchro.database.enableQueryMetadata";

	/**
	 * Is database a mirror database.
	 * <p/>
	 * If true, ID will be kept, if false, sequence will be used to generate new ID
	 */
	public static String IS_MIRROR_DATABASE = "synchro.database.isMirror";

	/**
	 * Should queries keep where clause, when getting data (or count() ) by FKs ?
	 * <p/>
	 * Should be 'true' for synchronization on referential tables, and 'false' for data tables
	 */
	public static String KEEP_WHERE_CLAUSE_ON_QUERIES_BY_FKS = "synchro.database.keepWhereClauseOnQueriesByFks";

	/**
	 * Column names to excludes : TABLE1.MY_COLUMN,TABLE2.MY_COLUMN;%.MY_COLUMN
	 */
	public static String COLUMN_EXCLUDES = "synchro.database.column.excludes";

	/**
	 * Tables names to includes
	 */
	public static String TABLE_INCLUDES = "synchro.database.table.includes";

	/**
	 * System timestamp
	 */
	public static String SYSTEM_TIMESTAMP = "synchro.database.systimestamp";

	/**
	 * ID column name
	 */
	public static String COLUMN_ID = "synchro.database.column.id";

	/**
	 * UPDATE_DATE column name
	 */
	public static String COLUMN_UPDATE_DATE = "synchro.database.column.updateDate";

	/**
	 * sequence suffix
	 */
	public static String SEQUENCE_SUFFIX = "synchro.database.sequence.suffix";

	/**
	 * max sql name length
	 */
	public static String MAX_SQL_NAME_LENGTH = "synchro.database.maxSqlNameLength";

	/* -- inherited from hibernate settings: -- */

	/**
	 * Names the Hibernate {@literal SQL} {@link org.hibernate.dialect.Dialect} class
	 */
	public static String DIALECT = Environment.DIALECT;

	/**
	 * Names the {@literal JDBC} driver class
	 */
	public static String DRIVER = Environment.DRIVER;

	/**
	 * Names the {@literal JDBC} connection url.
	 */
	public static String URL = Environment.URL;

	/**
	 * Names the connection user. This might mean one of 2 things in out-of-the-box Hibernate
	 * {@link org.hibernate.engine.jdbc.connections.spi.ConnectionProvider}:
	 * <ul>
	 * <li>The username used to pass along to creating the JDBC connection</li>
	 * <li>The username used to obtain a JDBC connection from a data source</li>
	 * </ul>
	 */
	public static String USER = Environment.USER;

	/**
	 * Names the connection password. See usage discussion on {@link #USER}
	 */
	public static String PASS = Environment.PASS;

	private Properties properties;

	private Dialect dialect;

	private String columnUpdateDate;

	private String columnId;

	private String sequenceSuffix;

	private int maxSqlNameLength;

	public SynchroDatabaseConfiguration(SynchroConfiguration config, Properties settings, boolean isTarget) {
		super();
		this.properties = settings;
		setIsTarget(isTarget);
		setReadOnly(!isTarget);

		// Force to refresh some cached values
		reloadCachedFields(config);
	}

	public String getProperty(String propertyName) {
		return properties.getProperty(propertyName);
	}

	public Properties getProperties() {
		return properties;
	}

	public SynchroDatabaseConfiguration setProperties(Properties properties) {
		boolean savedIsTarget = isTarget();
		this.properties = properties;
		setIsTarget(savedIsTarget); // restore previous value

		// Refresh some cached value
		reloadCachedFields(SynchroConfiguration.getInstance());

		return this;
	}

	private void reloadCachedFields(SynchroConfiguration config) {
		this.dialect = Dialect.getDialect(properties);
		this.columnId = ConfigurationHelper.getString(COLUMN_ID, properties, config.getColumnId());
		this.columnUpdateDate = ConfigurationHelper.getString(COLUMN_UPDATE_DATE, properties, config.getColumnUpdateDate());
		this.sequenceSuffix = ConfigurationHelper.getString(SEQUENCE_SUFFIX, properties, config.getSequenceSuffix());
		this.maxSqlNameLength = ConfigurationHelper.getInt(MAX_SQL_NAME_LENGTH, properties, config.getMaxSqlNameLength());
	}

	public String getJdbcSchema() {
		return ConfigurationHelper.getString(AvailableSettings.DEFAULT_SCHEMA, properties);
	}

	public String getJdbcCatalog() {
		return ConfigurationHelper.getString(AvailableSettings.DEFAULT_CATALOG, properties);
	}

	public String getJdbcUser() {
		return ConfigurationHelper.getString(AvailableSettings.USER, properties);
	}

	/**
	 * Set a property value by name
	 * 
	 * @param propertyName
	 *            The name of the property to set
	 * @param value
	 *            The new property value
	 * 
	 * @return this for method chaining
	 */
	public SynchroDatabaseConfiguration setProperty(String propertyName, String value) {
		properties.setProperty(propertyName, value);
		return this;
	}

	public String getColumnId() {
		return columnId;
	}

	public SynchroDatabaseConfiguration setColumnId(String columnName) {
		this.columnId = columnName;
		setProperty(COLUMN_ID, columnName);
		return this;
	}

	public String getColumnUpdateDate() {
		return columnUpdateDate;
	}

	public SynchroDatabaseConfiguration setColumnUpdateDate(String columnName) {
		this.columnUpdateDate = columnName;
		setProperty(COLUMN_UPDATE_DATE, columnName);
		return this;
	}

	public String getSequenceSuffix() {
		return sequenceSuffix;
	}

	public SynchroDatabaseConfiguration setSequenceSuffix(String sequenceSuffix) {
		this.sequenceSuffix = sequenceSuffix;
		setProperty(SEQUENCE_SUFFIX, sequenceSuffix);
		return this;
	}

	public int getMaxSqlNameLength() {
		return maxSqlNameLength;
	}

	public SynchroDatabaseConfiguration setMaxSqlNameLength(int maxSqlNameLength) {
		this.maxSqlNameLength = maxSqlNameLength;
		setProperty(MAX_SQL_NAME_LENGTH, Integer.toString(maxSqlNameLength));
		return this;
	}

	public String getUrl() {
		return getProperty(URL);
	}

	public String getUser() {
		return getProperty(USER);
	}

	public String getPassword() {
		return getProperty(PASS);
	}

	public Dialect getDialect() {
		return dialect;
	}

	public boolean isTarget() {
		return ConfigurationHelper.getBoolean(IS_TARGET_DATABASE, getProperties(), false);
	}

	public boolean isSource() {
		return !isTarget();
	}

	public SynchroDatabaseConfiguration setIsTarget(boolean isTarget) {
		setProperty(IS_TARGET_DATABASE, Boolean.toString(isTarget));
		return this;
	}

	public SynchroDatabaseConfiguration setReadOnly(boolean readonly) {
		setProperty(READ_ONLY, Boolean.toString(readonly));
		return this;
	}

	public boolean isReadOnly() {
		return ConfigurationHelper.getBoolean(READ_ONLY, getProperties(), false);
	}

	public void setSystemTimestamp(Timestamp systimestamp) {
		if (systimestamp == null) {
			properties.remove(SYSTEM_TIMESTAMP);
			return;
		}
		long timeInNanos = (systimestamp.getTime() / 1000) * 1000000000 + systimestamp.getNanos();
		setProperty(SYSTEM_TIMESTAMP, Long.toString(timeInNanos));
	}

	public Timestamp getSystemTimestamp() {
		Long timeInNanos = ConfigurationHelper.getLong(SYSTEM_TIMESTAMP, getProperties(), -1);
		if (timeInNanos == -1) {
			return null;
		}
		long time = (timeInNanos / 1000000000) * 1000;
		int nanos = (int) (timeInNanos % 1000000000);
		Timestamp result = new Timestamp(time);
		result.setNanos(nanos);
		return result;
	}

	public boolean isFullMetadataEnable() {
		return ConfigurationHelper.getBoolean(ENABLE_FULL_METADATA, getProperties(), isTarget());
	}

	public void setFullMetadataEnable(boolean enable) {
		setProperty(ENABLE_FULL_METADATA, Boolean.toString(enable));
	}

	public boolean isMirrorDatabase() {
		return ConfigurationHelper.getBoolean(IS_MIRROR_DATABASE, getProperties(), isTarget());
	}

	public void setIsMirrorDatabase(boolean isMirror) {
		setProperty(IS_MIRROR_DATABASE, Boolean.toString(isMirror));
	}

	public boolean isKeepWhereClauseOnQueriesByFks() {
		return ConfigurationHelper.getBoolean(KEEP_WHERE_CLAUSE_ON_QUERIES_BY_FKS, getProperties(), true);
	}

	public void setKeepWhereClauseOnQueriesByFks(boolean keepWhereClauseOnQueriesByFks) {
		setProperty(KEEP_WHERE_CLAUSE_ON_QUERIES_BY_FKS, Boolean.toString(keepWhereClauseOnQueriesByFks));
	}

	public String getColumnExcludes() {
		return ConfigurationHelper.getString(COLUMN_EXCLUDES, getProperties(), null);
	}

	public Set<String> getColumnExcludesAsSet() {
		String columnExcludesProperty = getColumnExcludes();
		if (StringUtils.isBlank(columnExcludesProperty)) {
			return null;
		}
		Set<String> columnExcludes = Sets.newHashSet(Arrays.asList(columnExcludesProperty.split(",")));
		return columnExcludes;
	}

	public SynchroDatabaseConfiguration addColumnExclude(String tableName, String columnName) {
		Preconditions.checkNotNull(getProperties());
		Preconditions.checkArgument(StringUtils.isNotBlank(tableName));
		Preconditions.checkArgument(StringUtils.isNotBlank(columnName));

		String excludesProperty = getProperty(COLUMN_EXCLUDES);
		Set<String> excludes = Sets.newHashSet();
		if (excludesProperty != null) {
			excludes.addAll(Arrays.asList(excludesProperty.split(",")));
		}

		excludes.add(tableName.toLowerCase() + "." + columnName.toLowerCase());

		// Store in properties
		excludesProperty = Joiner.on(',').join(excludes);
		setProperty(COLUMN_EXCLUDES, excludesProperty);

		return this;
	}

	public SynchroDatabaseConfiguration addColumnExclude(String columnName) {
		return addColumnExclude("%", columnName);
	}

	public SynchroDatabaseConfiguration addColumnExcludes(Set<String> columnExcludes) {
		Preconditions.checkNotNull(getProperties());
		Preconditions.checkArgument(CollectionUtils.isNotEmpty(columnExcludes));

		String excludesProperty = getProperty(COLUMN_EXCLUDES);
		Set<String> excludes = Sets.newHashSet();
		if (excludesProperty != null) {
			excludes.addAll(Splitter.on(',').trimResults().splitToList(excludesProperty));
		}
		for (String exclude : columnExcludes) {
			excludes.add(exclude.toLowerCase());
		}

		// Store in properties
		excludesProperty = Joiner.on(',').join(excludes);
		setProperty(COLUMN_EXCLUDES, excludesProperty);

		return this;
	}

	public SynchroDatabaseConfiguration removeColumnExclude(String columnName) {
		return removeColumnExclude("%", columnName);
	}

	public SynchroDatabaseConfiguration removeColumnExclude(String tableName, String columnName) {
		Preconditions.checkNotNull(getProperties());
		Preconditions.checkArgument(StringUtils.isNotBlank(tableName));
		Preconditions.checkArgument(StringUtils.isNotBlank(columnName));

		String excludesProperty = getProperty(COLUMN_EXCLUDES);
		Set<String> excludes = Sets.newHashSet();
		if (excludesProperty != null) {
			excludes.addAll(Arrays.asList(excludesProperty.split(",")));
		}

		excludes.remove(tableName.toLowerCase() + "." + columnName.toLowerCase());

		// Store in properties
		excludesProperty = Joiner.on(',').join(excludes);
		setProperty(COLUMN_EXCLUDES, excludesProperty);

		return this;
	}

	protected boolean getPropertyAsBoolean(String key, boolean defaultValue) {
		String property = getProperty(key);
		if (StringUtils.isBlank(property)) {
			return defaultValue;
		}
		return "true".equalsIgnoreCase(property);
	}

	public String toString() {
		StringBuilder sb = new StringBuilder();
		if (isTarget()) {
			sb.append("Target ");
		}
		else {
			sb.append("Source ");
		}
		sb.append("database configuration:");
		sb.append("\n    JDBC URL: ").append(getUrl());
		sb.append("\n    JDBC User: ").append(getUser());
		sb.append("\n    readOnly: ").append(isReadOnly());
		sb.append("\n    enableFullMetadata: ").append(isFullMetadataEnable());
		sb.append("\n    is mirror: ").append(isMirrorDatabase());

		// Column excludes
		String columnExcludes = getColumnExcludes();
		sb.append("\n    excludes columns: ");
		if (StringUtils.isNotBlank(columnExcludes)) {
			sb.append(columnExcludes);
		}

		return sb.toString();
	}
}
