package fr.ifremer.adagio.synchro.meta;

/*
 * #%L
 * Tutti :: Persistence
 * $Id: ReferentialSynchroTableMetadata.java 1573 2014-02-04 16:41:40Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/trunk/tutti-persistence/src/main/java/fr/ifremer/adagio/core/service/technical/synchro/ReferentialSynchroTableMetadata.java $
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import static org.nuiton.i18n.I18n.t;

import java.lang.reflect.Field;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.LockMode;
import org.hibernate.dialect.Dialect;
import org.hibernate.internal.util.StringHelper;
import org.hibernate.mapping.ForeignKey;
import org.hibernate.tool.hbm2ddl.ColumnMetadata;
import org.hibernate.tool.hbm2ddl.ForeignKeyMetadata;
import org.hibernate.tool.hbm2ddl.IndexMetadata;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.SubscriberExceptionContext;
import com.google.common.eventbus.SubscriberExceptionHandler;

import fr.ifremer.adagio.synchro.SynchroTechnicalException;
import fr.ifremer.adagio.synchro.dao.Daos;
import fr.ifremer.adagio.synchro.intercept.SynchroInterceptor;
import fr.ifremer.adagio.synchro.meta.event.CreateQueryEvent;
import fr.ifremer.adagio.synchro.meta.event.LoadJoinEvent;
import fr.ifremer.adagio.synchro.meta.event.LoadPkEvent;
import fr.ifremer.adagio.synchro.meta.event.LoadTableEvent;
import fr.ifremer.adagio.synchro.query.SynchroQueryBuilder;
import fr.ifremer.adagio.synchro.query.SynchroQueryName;
import fr.ifremer.adagio.synchro.service.SynchroDatabaseConfiguration;

/**
 * Overrides of the {@link TableMetadata} with some improvements:
 * <ul>
 * <li>Obtains number of columns via {@link #getColumnsCount()}</li>
 * <li>Obtains all columns names available via {@link #getColumnNames()}</li>
 * <li>Obtains number of columns in the select query, via {@link #getSelectColumnsCount()}</li>
 * <li>Obtains all columns names available in the select query via {@link #getSelectColumnNames()}</li>
 * <li>Obtains primary key column names via {@link #getPkNames()}</li>
 * </ul>
 * <p/>
 * And others methods used to synchronize referentials:
 * <ul>
 * <li>Obtains query to update a row of the table (column names order is the one introduced by method
 * {@link #getColumnNames()}: {@link #getUpdateQuery()}</li>
 * <li>Obtains query to insert a row in the table (column names order is the one introduced by method
 * {@link #getColumnNames()}: {@link #getInsertQuery()}</li>
 * </ul>
 * Created on 1/14/14.
 * 
 * @author Tony Chemit <chemit@codelutin.com>
 * @author Benoit Lavenier <benoit.lavenier@e-is.pro>
 * @since 3.0
 */
public class SynchroTableMetadata {
	private static final Log log = LogFactory.getLog(SynchroTableMetadata.class);

	public enum DuplicateKeyStrategy {
		REPLACE,
		REJECT,
		WARN
	}

	public static final String PK_SEPARATOR = "~~";

	public static final String UPDATE_DATE_BINDPARAM = "updateDate";

	private static Field delegateTableMetadataColumnsField = null;

	@SuppressWarnings({ "unchecked", "rawtypes" })
	public static Map<String, ColumnMetadata> getColumns(TableMetadata delegate) {
		try {
			if (delegateTableMetadataColumnsField == null) {
				delegateTableMetadataColumnsField = TableMetadata.class.getDeclaredField("columns");
				delegateTableMetadataColumnsField.setAccessible(true);
			}
			return Maps.<String, ColumnMetadata> newLinkedHashMap((Map) delegateTableMetadataColumnsField.get(delegate));
		} catch (Exception e) {
			throw new SynchroTechnicalException(e.getMessage(), e);
		}
	}

	/**
	 * Serialize into a String a list of PK value. Usefull when more than one PK column in a table
	 * 
	 * @param pkList
	 * @return
	 */
	public static String toPkStr(List<Object> pkList) {
		StringBuilder sb = new StringBuilder();
		for (Object pk : pkList) {
			sb.append(PK_SEPARATOR).append(pk);
		}
		return sb.substring(PK_SEPARATOR.length());
	}

	/**
	 * Serialize into a String a list of PK value. Usefull when more than one PK column in a table
	 * 
	 * @param pkList
	 * @return
	 */
	public static String toPkStr(Object[] pkList) {
		StringBuilder sb = new StringBuilder();
		for (Object pk : pkList) {
			sb.append(PK_SEPARATOR).append(pk);
		}
		return sb.substring(PK_SEPARATOR.length());
	}

	public static List<Object> fromPkStr(String pkStr) {
		List<Object> pkList = Lists.newArrayList();
		String[] split = pkStr.split(PK_SEPARATOR);
		for (String s : split) {
			if ("null".equals(s)) {
				s = null;
			}
			pkList.add(s);
		}
		return pkList;
	}

	public static List<List<Object>> fromPksStr(Set<String> pksStr) {
		List<List<Object>> result = Lists.transform(ImmutableList.copyOf(pksStr),
				new Function<String, List<Object>>() {
					@Override
					public List<Object> apply(String pkStr) {
						return SynchroTableMetadata.fromPkStr(pkStr);
					}
				});
		return result;
	}

	protected final String selectPksStrQuery;

	protected final String selectPksQuery;

	protected final String selectMaxUpdateDateQuery;

	protected final String selectUpdateDateByPkQuery;

	protected final String countAllQuery;

	protected final TableMetadata delegate;

	protected final Map<String, SynchroColumnMetadata> columns;

	protected List<SynchroJoinMetadata> childJoins;

	protected List<SynchroJoinMetadata> parentJoins;

	protected List<SynchroJoinMetadata> joins;

	protected boolean hasJoins;

	protected boolean hasChildJoins;

	protected final List<String> columnNames;

	protected final List<String> selectColumnNames;

	protected final List<String> insertColumnNames;

	protected final int columnCount;

	protected final int selectColumnCount;

	protected final Set<String> pkNames;

	protected final int[] selectPkIndexs;

	protected String insertQuery;

	protected String updateQuery;

	protected final int updateDateIndexInSelectQuery;

	protected final boolean withUpdateDateColumn;

	protected final boolean withIdColumn;

	protected boolean isRoot;

	protected final String countQuery;

	protected final String countFromUpdateDateQuery;

	protected final String selectFromUpdateDateQuery;

	protected final String selectAllQuery;

	protected final String selectByPkQuery;

	protected final String deleteByPkQuery;

	protected final Map<String, List<String>> uniqueConstraints;

	protected final Map<String, DuplicateKeyStrategy> duplicateKeyStrategies;

	protected Map<String, String> selectPksByIndexQueries;

	protected String sequenceName;

	protected String selectSequenceNextValString;

	protected String sequenceNextValString;

	protected List<SynchroInterceptor> interceptors;

	protected boolean hasUniqueConstraints;

	protected LockMode lockModeOnUpdate;

	protected boolean isBuild;

	// This field is reset when build() is called
	protected SynchroDatabaseMetadata dbMeta;

	// This field is reset when build() is called
	protected Dialect dialect;

	// This field is reset when build() is called
	protected EventBus eventBus;

	protected SynchroTableMetadata(
			SynchroDatabaseMetadata dbMeta,
			TableMetadata delegate,
			Dialect dialect,
			List<Object> listeners,
			String tableName,
			Predicate<SynchroColumnMetadata> columnFilter,
			boolean enableQueries) {

		Preconditions.checkNotNull(delegate);
		Preconditions.checkNotNull(dbMeta);

		this.isBuild = false;
		this.dbMeta = dbMeta;
		this.dialect = dialect;
		this.delegate = delegate;
		this.eventBus = initEventBus(enableQueries, listeners, delegate.getName());
		this.interceptors = initInterceptors(listeners);

		SynchroDatabaseConfiguration config = dbMeta.getConfiguration();

		try {
			this.columns = initColumns(dialect, tableName, columnFilter);
			this.columnNames = initColumnNames(columns);
			this.columnCount = columnNames.size();
			this.pkNames = initPrimaryKeys(dbMeta);
			Preconditions.checkNotNull(pkNames);
			this.uniqueConstraints = Maps.newLinkedHashMap();
			this.duplicateKeyStrategies = Maps.newHashMap();
			initUniqueConstraints(dbMeta);
			this.hasUniqueConstraints = MapUtils.isNotEmpty(this.uniqueConstraints);
			this.withIdColumn = columnNames.contains(config.getColumnId());
			this.withUpdateDateColumn = columnNames.contains(config.getColumnUpdateDate());
			this.sequenceName = initSequenceName(dbMeta);
		} catch (Exception e) {
			throw new SynchroTechnicalException(t("adagio.synchro.meta.table.instanciation.error", delegate.getName()), e);
		}

		if (enableQueries) {
			this.selectSequenceNextValString = createSelectSequenceNextValString(dialect, this.sequenceName);
			this.sequenceNextValString = createSequenceNextValString(dialect, this.sequenceName);
			this.countAllQuery = createAllCountQuery();
			this.countQuery = createCountQuery();
			this.countFromUpdateDateQuery = createCountFromUpdateDate(config);
			this.selectMaxUpdateDateQuery = createSelectMaxUpdateDateQuery(config);
			this.selectUpdateDateByPkQuery = createSelectUpdateDateByPkQuery(config);
			this.selectAllQuery = createSelectAllQuery();
			this.selectFromUpdateDateQuery = createSelectFromUpdateDateQuery(config);
			this.selectByPkQuery = createSelectByPkQuery();
			this.selectPksStrQuery = createSelectPksStrQuery(config);
			this.selectPksQuery = createSelectPksQuery(config);
			this.insertQuery = createInsertQuery();
			this.updateQuery = createUpdateQuery();
			this.selectPksByIndexQueries = createSelectPksByIndexQueries(config);

			this.selectColumnNames = initColumnNamesFromQuery(this.selectAllQuery);
			this.insertColumnNames = initColumnNamesFromQuery(this.insertQuery);
			this.selectPkIndexs = createSelectPkIndex(this.selectColumnNames);
			Preconditions.checkArgument(this.selectPkIndexs.length == pkNames.size(),
					"Could not found all table PKs in the select query. This is require.");
			this.selectColumnCount = selectColumnNames.size();

			this.updateDateIndexInSelectQuery = selectColumnNames.indexOf(config.getColumnUpdateDate());
			this.deleteByPkQuery = createDeleteByPk();
		}
		else {
			this.selectSequenceNextValString = null;
			this.sequenceNextValString = null;
			this.countAllQuery = null;
			this.countQuery = null;
			this.countFromUpdateDateQuery = null;
			this.selectMaxUpdateDateQuery = null;
			this.selectUpdateDateByPkQuery = null;
			this.selectAllQuery = null;
			this.selectFromUpdateDateQuery = null;
			this.selectByPkQuery = null;
			this.selectPksStrQuery = null;
			this.selectPksQuery = null;
			this.insertQuery = null;
			this.updateQuery = null;
			this.selectPksByIndexQueries = null;
			this.selectColumnNames = Lists.newArrayList(); // empty list, to avoid NullPointerException on
															// getSelectColumnIndex()
			this.insertColumnNames = Lists.newArrayList();
			this.selectColumnCount = 0;
			this.updateDateIndexInSelectQuery = -1;
			this.selectPkIndexs = null;
			this.deleteByPkQuery = null;
		}

		// Default value for joins (will be override in initJoins())
		this.joins = Lists.newArrayList();
		this.hasJoins = false;
		this.parentJoins = null;
		this.hasChildJoins = false;
		this.lockModeOnUpdate = LockMode.NONE;

		// Default values (could be override using interceptor)
		this.isRoot = false;
	}

	// for tests purposes
	SynchroTableMetadata() {
		isBuild = false;
		eventBus = null;
		delegate = null;
		columns = null;
		columnNames = null;
		selectSequenceNextValString = null;
		sequenceNextValString = null;
		joins = null;
		pkNames = null;
		selectPkIndexs = null;
		updateDateIndexInSelectQuery = -1;
		withUpdateDateColumn = false;
		withIdColumn = false;
		isRoot = false;
		insertQuery = null;
		updateQuery = null;
		countAllQuery = null;
		countQuery = null;
		countFromUpdateDateQuery = null;
		selectPksStrQuery = null;
		selectPksQuery = null;
		selectMaxUpdateDateQuery = null;
		selectUpdateDateByPkQuery = null;
		selectFromUpdateDateQuery = null;
		sequenceName = null;
		selectAllQuery = null;
		selectByPkQuery = null;
		columnCount = 0;
		selectPksByIndexQueries = null;
		uniqueConstraints = null;
		duplicateKeyStrategies = null;
		selectColumnNames = null;
		insertColumnNames = null;
		selectColumnCount = 0;
		lockModeOnUpdate = LockMode.NONE;
		deleteByPkQuery = null;
	}

	/**
	 * Initialize Join metadata. this need to have already loaded all tables.
	 * 
	 * @param dbMeta
	 *            the Database metadata, with some preloaded tables inside
	 */
	public void initJoins(SynchroDatabaseMetadata dbMeta) {
		try {
			this.joins = initJoins(getCatalog(), getSchema(), getName(), dbMeta, this.columns);

			// Update fields on joins
			this.hasJoins = !joins.isEmpty();
			this.childJoins = initChildJoins(joins);
			this.parentJoins = initParentJoins(joins);
			this.hasChildJoins = !childJoins.isEmpty();

		} catch (Exception e) {
			throw new SynchroTechnicalException(t("adagio.synchro.meta.table.instanciation.error", delegate.getName()), e);
		}
	}

	public Set<String> getPkNames() {
		return pkNames;
	}

	public boolean isWithUpdateDateColumn() {
		return withUpdateDateColumn;
	}

	public boolean isWithIdColumn() {
		return withIdColumn;
	}

	public boolean isRoot() {
		return isRoot;
	}

	public void setRoot(boolean isRoot) {
		this.isRoot = isRoot;
	}

	public int getColumnsCount() {
		return columnCount;
	}

	public Set<String> getColumnNames() {
		return ImmutableSet.copyOf(columnNames);
	}

	public int getSelectColumnsCount() {
		return selectColumnCount;
	}

	public Set<String> getSelectColumnNames() {
		return ImmutableSet.copyOf(selectColumnNames);
	}

	public String getColumnName(int columnIndex) {
		return columnNames.get(columnIndex);
	}

	public int getColumnIndex(String name) {
		return columnNames.indexOf(name.toLowerCase());
	}

	public int getSelectColumnIndex(String name) {
		return selectColumnNames.indexOf(name.toLowerCase());
	}

	public int getInsertColumnIndex(String name) {
		return insertColumnNames.indexOf(name.toLowerCase());
	}

	public TableMetadata getDelegate() {
		return delegate;
	}

	public String getName() {
		return delegate.getName();
	}

	public ForeignKeyMetadata getForeignKeyMetadata(ForeignKey fk) {
		return delegate.getForeignKeyMetadata(fk);
	}

	public SynchroColumnMetadata getColumnMetadata(String columnName) {
		return columns.get(StringHelper.toLowerCase(columnName));
	}

	public String getSchema() {
		return delegate.getSchema();
	}

	public String getCatalog() {
		return delegate.getCatalog();
	}

	public ForeignKeyMetadata getForeignKeyMetadata(String keyName) {
		return delegate.getForeignKeyMetadata(keyName);
	}

	public IndexMetadata getIndexMetadata(String indexName) {
		return delegate.getIndexMetadata(indexName);
	}

	public SynchroColumnMetadata getColumn(String columnName) {
		return columns.get(columnName);
	}

	public List<SynchroJoinMetadata> getJoins() {
		return joins;
	}

	public List<SynchroJoinMetadata> getChildJoins() {
		return childJoins;
	}

	public List<SynchroJoinMetadata> getParentJoins() {
		return parentJoins;
	}

	public boolean hasJoins() {
		return hasJoins;
	}

	public boolean hasChildJoins() {
		return hasChildJoins;
	}

	public LockMode getLockModeOnUpdate() {
		return lockModeOnUpdate;
	}

	public void setLockOnUpdate(LockMode lockModeOnUpdate) {
		this.lockModeOnUpdate = lockModeOnUpdate;
	}

	public void setSequenceName(String sequenceName) {
		this.sequenceName = sequenceName;
	}

	public List<SynchroInterceptor> getInterceptors() {
		return this.interceptors;
	}

	public String getTableLogPrefix() {
		return "[" + getName() + "]";
	}

	@Override
	public String toString() {
		return delegate.toString();
	}

	// ------------------------------------------------------------------------//
	// -- queries methods --//
	// ------------------------------------------------------------------------//

	public String getSequenceName() {
		return sequenceName;
	}

	public String getInsertQuery() {
		return insertQuery;
	}

	public String getUpdateQuery() {
		return updateQuery;
	}

	/**
	 * Obtains a SQL with one column output : a concatenation of all PK.
	 * 
	 * @return a SQL select, with only one result column
	 */
	public String getSelectPksStrQuery() {
		return selectPksStrQuery;
	}

	public String getSelectPksQuery() {
		return selectPksQuery;
	}

	public String getSelectMaxUpdateDateQuery() {
		return selectMaxUpdateDateQuery;
	}

	public String getSelectUpdateDateByPkQuery() {
		return selectUpdateDateByPkQuery;
	}

	public String getSelectDataQueryFromPk() {
		return selectByPkQuery;
	}

	public String getSelectAllQuery() {
		return selectAllQuery;
	}

	public String getSelectUpdatedDataQuery() {
		return selectFromUpdateDateQuery;
	}

	public String getDeleteByPkQuery() {
		return deleteByPkQuery;
	}

	public String getSequenceNextValString() {
		return sequenceNextValString;
	}

	public String getSelectSequenceNextValString() {
		return selectSequenceNextValString;
	}

	/**
	 * Count all rows (no where clause)
	 * 
	 * @return
	 */
	public String getCountAllQuery() {
		return countAllQuery;
	}

	/**
	 * Count rows to synchronize (could have where clause).
	 * <p/>
	 * Could be override using listener on SynchroQueryName.count
	 * 
	 * @return
	 */
	public String getCountQuery() {
		return countQuery;
	}

	public String getCountUpdatedDataQuery() {
		if (countFromUpdateDateQuery == null) {
			return null;
		}
		return countFromUpdateDateQuery;
	}

	// ------------------------------------------------------------------------//
	// -- PK methods --//
	// ------------------------------------------------------------------------//

	public int[] getSelectPkIndexs() {
		return selectPkIndexs;
	}

	public boolean isSimpleKey() {
		return selectPkIndexs.length == 1;
	}

	public boolean isSelectPrimaryKeysAsStringQueryEnable() {
		return selectPksStrQuery != null;
	}

	public Timestamp getUpdateDate(ResultSet incomingData) throws SQLException {
		return incomingData.getTimestamp(updateDateIndexInSelectQuery + 1);
	}

	public Timestamp getUpdateDate(Object[] incomingData) throws SQLException {
		return (Timestamp) incomingData[updateDateIndexInSelectQuery];
	}

	/**
	 * Add a unique constraint. <br/>
	 * The given name must not exists in the constraint list. If you need to replace a existing constraint,
	 * use the method {@link putUniqueConstraint()} instead.
	 * 
	 * @param indexName
	 *            the unique constraint name
	 * @param columnNames
	 * @param duplicateKeyStrategy
	 */
	public void addUniqueConstraint(String indexName, List<String> columnNames, DuplicateKeyStrategy duplicateKeyStrategy) {
		Preconditions.checkNotNull(indexName);
		Preconditions.checkArgument(!uniqueConstraints.containsKey(indexName), "Duplicate unique constraints name");
		Preconditions.checkNotNull(CollectionUtils.isNotEmpty(columnNames));

		if (log.isDebugEnabled()) {
			log.debug(String.format("[%s] Add unique constraint: %s", getName(), indexName));
		}

		List<String> columnNamesLowerCase = Lists.newArrayListWithCapacity(columnNames.size());
		for (String columnName : columnNames) {
			columnNamesLowerCase.add(columnName.toLowerCase());
		}

		uniqueConstraints.put(indexName, columnNamesLowerCase);
		duplicateKeyStrategies.put(indexName, duplicateKeyStrategy);
		hasUniqueConstraints = true;
	}

	/**
	 * Add (or replace) a unique constraint
	 * 
	 * @param indexName
	 *            the unique constraint name
	 * @param columnNames
	 * @param duplicateKeyStrategy
	 */
	public void putUniqueConstraint(String indexName, List<String> columnNames, DuplicateKeyStrategy duplicateKeyStrategy) {
		Preconditions.checkNotNull(indexName);
		Preconditions.checkNotNull(CollectionUtils.isNotEmpty(columnNames));

		if (log.isDebugEnabled()) {
			log.debug(String.format("[%s] Add unique constraint: %s", getName(), indexName));
		}

		List<String> columnNamesLowerCase = Lists.newArrayListWithCapacity(columnNames.size());
		for (String columnName : columnNames) {
			columnNamesLowerCase.add(columnName.toLowerCase());
		}

		uniqueConstraints.put(indexName, columnNamesLowerCase);
		duplicateKeyStrategies.put(indexName, duplicateKeyStrategy);
		hasUniqueConstraints = true;
	}

	public Map<String, List<String>> getUniqueConstraints() {
		checkBuild();
		return this.uniqueConstraints;
	}

	public boolean hasUniqueConstraint(String constraintName) {
		return this.uniqueConstraints.containsKey(constraintName);
	}

	public boolean hasUniqueConstraints() {
		return this.hasUniqueConstraints;
	}

	public void removeUniqueConstraints(String indexName) {
		checkBuild();
		this.uniqueConstraints.remove(indexName);
		this.duplicateKeyStrategies.remove(indexName);
	}

	public Map<String, DuplicateKeyStrategy> getDuplicatKeyStrategies() {
		checkBuild();
		return this.duplicateKeyStrategies;
	}

	public String getSelectPkByIndex(String indexName) {
		checkBuild();
		return MapUtils.getString(this.selectPksByIndexQueries, indexName);
	}

	/**
	 * Return queries, to count if a pK is used by other tables.<br/>
	 * The SQL queries returned has params (<code>:pk1</code> to <code>:pkN<code>) that need to be bind.
	 * 
	 * @param dbMeta
	 *            the metadata of the target database (This is required because the SynchroTableMetadata could have be
	 *            build on the soure database).
	 * @return
	 * @throws SQLException
	 */
	public Map<String, String> getCountPkReferenceQueries(DatabaseMetaData dbMeta) throws SQLException {
		checkBuild();
		Preconditions.checkNotNull(dbMeta);

		if (CollectionUtils.isEmpty(pkNames)) {
			return null;
		}

		Map<String, String> result;
		if (pkNames.size() == 1) {
			result = getCountPkReferenceQueriesUniquePk(dbMeta);
		}
		else {
			result = getCountPkReferenceQueriesCompositePk(dbMeta);
		}

		return result;
	}

	// ------------------------------------------------------------------------//
	// -- Protected methods --//
	// ------------------------------------------------------------------------//

	protected Map<String, SynchroColumnMetadata> initColumns(
			Dialect dialect,
			String tableName,
			Predicate<SynchroColumnMetadata> columnFilter
			) throws NoSuchFieldException, SecurityException, IllegalArgumentException, IllegalAccessException {

		Map<String, ColumnMetadata> delegateColumns = getColumns(delegate);
		Map<String, SynchroColumnMetadata> columns = Maps.newHashMap();
		// Dialect dialect = dbMeta.getDialect();
		for (String columnName : delegateColumns.keySet()) {
			ColumnMetadata delegateColumn = delegateColumns.get(columnName);
			SynchroColumnMetadata column = new SynchroColumnMetadata(
					tableName,
					delegateColumn,
					dialect
					);
			boolean skipColumn = columnFilter != null && !columnFilter.apply(column);
			if (!skipColumn) {
				columns.put(columnName, column);
			}
		}

		return columns;
	}

	protected List<String> initColumnNames(Map<String, SynchroColumnMetadata> columns) {
		List<String> result = Lists.newArrayListWithExpectedSize(columns.size());
		for (String name : columns.keySet()) {
			result.add(name.toLowerCase());
		}
		return result;
	}

	protected List<String> initColumnNamesFromQuery(String aQuery) {
		return SynchroQueryBuilder.newBuilder(aQuery).getColumnNames();
	}

	protected Set<String> initPrimaryKeys(SynchroDatabaseMetadata dbMeta) throws SQLException {

		Set<String> result = Sets.newLinkedHashSet();
		ResultSet rs = dbMeta.getPrimaryKeys(getCatalog(), getSchema(), getName());
		try {

			while (rs.next()) {
				result.add(rs.getString("COLUMN_NAME").toLowerCase());
			}
			// Fire event to interceptors
			result = fireOnPkLoad(result);
			return ImmutableSet.copyOf(result);
		} finally {
			Daos.closeSilently(rs);
		}
	}

	protected void initUniqueConstraints(SynchroDatabaseMetadata dbMeta) throws SQLException {
		Preconditions.checkNotNull(this.uniqueConstraints);

		ResultSet rs = null;
		try {
			rs = dbMeta.getIndexInfo(getCatalog(), getSchema(), getName(),
					true /* unique */,
					true /* allow info to be approximated (do not re-compute statistics) */
					);

			List<String> columnNames = Lists.newArrayList();
			String lastIndexName = null;
			while (rs.next()) {

				String indexName = rs.getString("INDEX_NAME");
				if (indexName != null) {
					indexName = indexName.toLowerCase();
					// If index changed, add the previous index to the result
					if (lastIndexName != null && indexName.equals(lastIndexName) == false) {
						addUniqueConstraintsIfNotPK(
								indexName,
								columnNames,
								// Always reject, when constraints is define on database level
								DuplicateKeyStrategy.REJECT);
						// Reset loop var
						lastIndexName = indexName;
						columnNames = Lists.newArrayList();
					}

					String columName = rs.getString("COLUMN_NAME").toLowerCase();
					columnNames.add(columName);
				}
			}

			// Add the last index to the result
			if (lastIndexName != null) {
				addUniqueConstraintsIfNotPK(
						lastIndexName,
						columnNames,
						// Always reject, when constraints is define on database level
						DuplicateKeyStrategy.REJECT);
			}

		} finally {
			Daos.closeSilently(rs);
		}
	}

	protected List<SynchroInterceptor> initInterceptors(List<Object> listeners) {
		List<SynchroInterceptor> result = Lists.newArrayList();
		if (listeners != null) {
			for (Object listener : listeners) {
				if (listener instanceof SynchroInterceptor) {
					result.add((SynchroInterceptor) listener);
				}
			}
		}
		return result;
	}

	protected int[] createSelectPkIndex(List<String> selectColumnNames) {

		int[] result = new int[pkNames.size()];

		int pkI = 0;
		for (String pkName : pkNames) {
			String pkColumnName = pkName.toLowerCase();

			int i = 1;

			int index = -1;
			for (String columnName : selectColumnNames) {
				if (pkColumnName.equals(columnName)) {
					index = i;
				} else {
					i++;
				}
			}
			result[pkI++] = index;
		}
		return result;
	}

	protected List<SynchroJoinMetadata> initJoins(
			String catalog,
			String schema,
			String tableName,
			SynchroDatabaseMetadata dbMeta,
			Map<String, SynchroColumnMetadata> columns
			) throws SQLException {

		List<SynchroJoinMetadata> result = Lists.newArrayList();

		// Exported keys (primary keys referenced by another table)
		ResultSet rs = null;
		try {
			rs = dbMeta.getExportedKeys(catalog, schema, tableName);
			while (rs.next()) {
				String columnName = rs.getString("PKCOLUMN_NAME").toLowerCase();
				if (columns.containsKey(columnName)) {
					SynchroJoinMetadata join = new SynchroJoinMetadata(rs, this, dbMeta);

					// Fire event to interceptors
					fireOnJoinLoad(join);

					if (join.isValid()) {
						result.add(join);
					}
				}
			}
		} finally {
			Daos.closeSilently(rs);
		}

		// Imported keys (foreign keys that references another table)
		try {
			rs = dbMeta.getImportedKeys(catalog, schema, tableName);
			while (rs.next()) {
				String columnName = rs.getString("FKCOLUMN_NAME").toLowerCase();
				if (columns.containsKey(columnName)) {
					SynchroJoinMetadata join = new SynchroJoinMetadata(rs, this, dbMeta);

					// Fire event to interceptors
					fireOnJoinLoad(join);

					if (join.isValid()) {
						result.add(join);

						SynchroColumnMetadata column = columns.get(columnName);
						column.setParentJoin(join);
					}
				}
			}
		} finally {
			Daos.closeSilently(rs);
		}

		return result;
	}

	protected List<SynchroJoinMetadata> initChildJoins(
			List<SynchroJoinMetadata> joins
			) throws SQLException {

		List<SynchroJoinMetadata> result = Lists.newArrayListWithExpectedSize(joins.size());

		for (SynchroJoinMetadata join : joins) {
			if (join.isChild()) {
				result.add(join);
			}
		}
		return result;
	}

	protected List<SynchroJoinMetadata> initParentJoins(
			List<SynchroJoinMetadata> joins
			) throws SQLException {

		List<SynchroJoinMetadata> result = Lists.newArrayListWithExpectedSize(joins.size());

		for (SynchroJoinMetadata join : joins) {
			if (!join.isChild()) {
				result.add(join);
			}
		}
		return result;
	}

	protected String createInsertQuery() {
		if (CollectionUtils.isEmpty(columnNames)) {
			return null;
		}

		StringBuilder queryParams = new StringBuilder();
		StringBuilder valueParams = new StringBuilder();

		for (String columnName : columnNames) {
			queryParams.append(", ").append(columnName);
			valueParams.append(", ?");
		}

		String sql = String.format("INSERT INTO %s (%s) VALUES (%s)",
				getName(),
				queryParams.substring(2),
				valueParams.substring(2));

		return fireOnCreateQuery(SynchroQueryName.insert, sql);
	}

	protected String createUpdateQuery() {
		// Could not update, because no PK found
		if (CollectionUtils.isEmpty(pkNames)) {
			return null;
		}
		if (CollectionUtils.isEmpty(columnNames)) {
			return null;
		}

		StringBuilder updateParams = new StringBuilder();

		for (String columnName : columnNames) {
			updateParams.append(", ").append(columnName);
			updateParams.append(" = ?");
		}

		String sql = String.format("UPDATE %s SET %s WHERE %s",
				getName(),
				updateParams.substring(2),
				createPkWhereClause());
		return fireOnCreateQuery(SynchroQueryName.update, sql);
	}

	protected String createDeleteByPk() {
		String sql = String.format("DELETE FROM %s WHERE %s",
				getName(),
				createPkWhereClause()
				);
		return fireOnCreateQuery(SynchroQueryName.deleteByPk, sql);
	}

	protected String createSelectByPkQuery() {
		// Could not update, because no PK found
		if (CollectionUtils.isEmpty(pkNames)) {
			return null;
		}

		String sql = String.format("SELECT %s FROM %s t WHERE %s",
				createSelectParams("t"),
				getName(),
				createPkWhereClause("t"));

		return fireOnCreateQuery(SynchroQueryName.selectByPk, sql);
	}

	protected String createSelectPksStrQuery(SynchroDatabaseConfiguration config) {
		// Could not update, because no PK found
		if (CollectionUtils.isEmpty(pkNames)) {
			return null;
		}

		String prefix = " || '" + PK_SEPARATOR + "' || ";
		StringBuilder pkParams = new StringBuilder();

		boolean allowUniqueOutputColumn = true;

		for (String columnName : pkNames) {

			// If some date conversion => not database independent
			SynchroColumnMetadata column = columns.get(columnName);

			// For date/timestamp : make sure the convertion to char give the same result
			// -> add a cast to Timestamp
			if (column.getTypeCode() == Types.TIMESTAMP
					|| column.getTypeCode() == Types.DATE) {
				allowUniqueOutputColumn = false;
			}
			pkParams.append(prefix).append("t.").append(columnName);
		}

		if (!allowUniqueOutputColumn) {
			return null;
		}

		// Append updateDate is present
		if (withUpdateDateColumn) {
			pkParams.append(", ").append(config.getColumnUpdateDate());
		}

		String sql = String.format("SELECT %s FROM %s t",
				pkParams.substring(prefix.length()),
				getName());

		return fireOnCreateQuery(SynchroQueryName.selectPksStr, sql);
	}

	protected String createSelectPksQuery(SynchroDatabaseConfiguration config) {
		// Could not update, because no PK found
		if (CollectionUtils.isEmpty(pkNames)) {
			return null;
		}

		StringBuilder pkParams = new StringBuilder();

		for (String columnName : pkNames) {
			pkParams.append(", t.").append(columnName);
		}

		// Append updateDate is present
		if (withUpdateDateColumn) {
			pkParams.append(", t.").append(config.getColumnUpdateDate());
		}

		String sql = String.format("SELECT %s FROM %s t",
				pkParams.substring(2),
				getName());

		return fireOnCreateQuery(SynchroQueryName.selectPks, sql);
	}

	protected Map<String, String> createSelectPksByIndexQueries(SynchroDatabaseConfiguration config) {
		// Could not update, because no PK found
		if (MapUtils.isEmpty(uniqueConstraints)) {
			return null;
		}
		// Could not update, because no PK found
		if (CollectionUtils.isEmpty(pkNames)) {
			return null;
		}
		Map<String, String> result = Maps.newHashMap();

		// Create the select clause
		StringBuilder pkParams = new StringBuilder();
		for (String columnName : pkNames) {
			pkParams.append(", t.").append(columnName);
		}

		// Append updateDate is present
		if (withUpdateDateColumn) {
			pkParams.append(", t.").append(config.getColumnUpdateDate());
		}

		// For each unique constraints
		for (Entry<String, List<String>> entry : uniqueConstraints.entrySet()) {

			// Create the where clause
			StringBuilder whereParams = new StringBuilder();
			for (String columnName : entry.getValue()) {
				whereParams.append(" AND t.").append(columnName).append(" = ?");
			}

			String sql = String.format("SELECT %s FROM %s t WHERE %s",
					pkParams.substring(2),
					getName(),
					whereParams.substring(5));

			sql = fireOnCreateQuery(SynchroQueryName.selectPksByIndex, sql);

			if (StringUtils.isNotBlank(sql)) {
				result.put(entry.getKey(), sql);
			}
		}

		return result;
	}

	protected String createSelectAllQuery() {
		String sql = String.format("SELECT %s FROM %s t",
				createSelectParams("t"),
				getName()
				);

		return fireOnCreateQuery(SynchroQueryName.select, sql);
	}

	protected String createSelectMaxUpdateDateQuery(SynchroDatabaseConfiguration config) {
		if (!withUpdateDateColumn) {
			return null;
		}
		String sql = String.format("SELECT max(t.%s) FROM %s t",
				config.getColumnUpdateDate(),
				getName());

		return fireOnCreateQuery(SynchroQueryName.selectMaxUpdateDate, sql);
	}

	protected String createSelectUpdateDateByPkQuery(SynchroDatabaseConfiguration config) {
		if (!withUpdateDateColumn) {
			return null;
		}
		String sql = String.format("SELECT t.%s FROM %s t WHERE %s",
				config.getColumnUpdateDate(),
				getName(),
				createPkWhereClause("t"));

		return fireOnCreateQuery(SynchroQueryName.selectUpdateDateByPk, sql);
	}

	protected String createSelectFromUpdateDateQuery(final SynchroDatabaseConfiguration config) {
		if (!withUpdateDateColumn) {
			return null;
		}
		String sql = String.format("SELECT %s FROM %s t%s",
				createSelectParams("t"),
				getName(),
				createWithUpdateDateWhereClause(config, "t"));

		return fireOnCreateQuery(SynchroQueryName.selectFromUpdateDate, sql);
	}

	protected String createAllCountQuery() {
		String sql = String.format("SELECT count(*) FROM %s t",
				getName());

		// DO NOT send override query event (not where clause should be add)
		// return sqlfireOnCreateQuery(SynchroQueryName.countAll, sql);

		return sql;
	}

	protected String createCountQuery() {
		String sql = String.format("SELECT count(*) FROM %s t",
				getName());

		return fireOnCreateQuery(SynchroQueryName.count, sql);
	}

	protected String createCountFromUpdateDate(final SynchroDatabaseConfiguration config) {
		String sql = String.format("SELECT count(*) FROM %s t%s",
				getName(),
				createWithUpdateDateWhereClause(config, "t"));

		return fireOnCreateQuery(SynchroQueryName.countFromUpdateDate, sql);
	}

	public String createPkWhereClause() {
		return createPkWhereClause(null);
	}

	public String createPkWhereClause(String tableAlias) {
		Preconditions.checkArgument(CollectionUtils.isNotEmpty(pkNames));
		StringBuilder pkParams = new StringBuilder();
		String prefix = tableAlias != null ? tableAlias + "." : "";

		for (String columnName : pkNames) {
			pkParams.append(" AND ")
					.append(prefix)
					.append(columnName)
					.append(" = ?");
		}

		return pkParams.substring(5);
	}

	protected String createWithUpdateDateWhereClause(final SynchroDatabaseConfiguration config) {
		return createWithUpdateDateWhereClause(config, null);
	}

	protected String createWithUpdateDateWhereClause(final SynchroDatabaseConfiguration config, String tableAlias) {
		String whereClause;

		if (isWithUpdateDateColumn()) {
			String columnUpdateDate = config.getColumnUpdateDate();
			String prefix = tableAlias != null ? tableAlias + "." : "";
			// add a filter
			whereClause = String.format(
					" WHERE (%s%s IS NULL OR %s%s > :%s)",
					prefix, columnUpdateDate,
					prefix, columnUpdateDate,
					UPDATE_DATE_BINDPARAM);
		} else {
			whereClause = "";
		}
		return whereClause;
	}

	protected String createSelectParams() {
		return createSelectParams(null);
	}

	protected String createSelectParams(String tableAlias) {
		return createSelectParams(this.columnNames, tableAlias);
	}

	protected String createSelectParams(List<String> columnNames, String tableAlias) {
		Preconditions.checkArgument(CollectionUtils.isNotEmpty(columnNames),
				String.format("No column found for table: %s", delegate.getName()));

		StringBuilder queryParams = new StringBuilder();

		for (String columnName : columnNames) {
			queryParams.append(", ");
			if (tableAlias != null) {
				queryParams.append(tableAlias)
						.append(".");
			}
			queryParams.append(columnName);
		}

		return queryParams.substring(2);
	}

	protected String initSequenceName(SynchroDatabaseMetadata dbMeta) {
		final Set<String> availableSequences = dbMeta.getSequences();
		final SynchroDatabaseConfiguration config = dbMeta.getConfiguration();
		final String sequenceSuffix = config.getSequenceSuffix();
		final int maxSqlNameLength = config.getMaxSqlNameLength();

		final String tableName = getName().toLowerCase();
		String sequenceName;

		// Compute the max size of
		final int maxLength = maxSqlNameLength - sequenceSuffix.length();
		if (maxLength > -0) {
			sequenceName = SynchroMetadataUtils.ensureMaximumNameLength(tableName, maxLength) + sequenceSuffix;
			if (availableSequences.contains(sequenceName.toLowerCase())) {
				return sequenceName;
			}
		}

		// If not found (with length limit), try without length limit
		sequenceName = tableName + sequenceSuffix;
		if (availableSequences.contains(sequenceName.toLowerCase())) {
			return sequenceName;
		}

		// sequence not found
		return null;
	}

	protected String createSelectSequenceNextValString(Dialect dialect, String sequenceName) {
		if (StringUtils.isBlank(sequenceName)) {
			return null;
		}
		return dialect.getSelectSequenceNextValString(sequenceName);
	}

	protected String createSequenceNextValString(Dialect dialect, String sequenceName) {
		if (StringUtils.isBlank(sequenceName)) {
			return null;
		}
		return dialect.getSequenceNextValString(sequenceName);
	}

	protected String fireOnCreateQuery(SynchroQueryName queryName, String sql) {
		if (eventBus == null) {
			return sql;
		}

		// Dispatch event to listeners
		CreateQueryEvent e = new CreateQueryEvent(this, queryName, sql);
		eventBus.post(e);

		// Return the updated sql query
		return e.sql;
	}

	protected SynchroJoinMetadata fireOnJoinLoad(SynchroJoinMetadata join) {
		if (eventBus == null || !join.isValid()) {
			return join;
		}

		// Dispatch event to listeners
		LoadJoinEvent e = new LoadJoinEvent(this, join);
		eventBus.post(e);

		// Return the updated join
		return e.join;
	}

	protected Set<String> fireOnPkLoad(Set<String> pk) {
		if (eventBus == null) {
			return pk;
		}

		// Dispatch event to listeners
		LoadPkEvent e = new LoadPkEvent(this, pk);
		eventBus.post(e);

		// Return the updated join
		return e.pk;
	}

	public SynchroTableMetadata build() {
		if (isBuild) {
			return this;
		}

		// Dispatch load event to listeners
		if (eventBus != null) {
			LoadTableEvent e = new LoadTableEvent(dbMeta, this);
			eventBus.post(e);
		}

		try {
			// Compute/Refresh some fields (after event dispatch)
			this.selectPksByIndexQueries = createSelectPksByIndexQueries(dbMeta.getConfiguration());
			this.selectSequenceNextValString = createSelectSequenceNextValString(dialect, this.sequenceName);
			this.sequenceNextValString = createSequenceNextValString(dialect, this.sequenceName);

		} catch (Exception e) {
			throw new SynchroTechnicalException(t("adagio.synchro.meta.table.instanciation.error", delegate.getName()), e);
		}

		// Clean fields unused after build
		this.eventBus = null;
		this.dbMeta = null;
		this.dialect = null;

		if (log.isTraceEnabled()) {
			logQueriesMappings();
		}

		isBuild = true;
		return this;
	}

	public void logQueriesMappings() {
		if (selectAllQuery == null || !log.isDebugEnabled()) {
			return;
		}
		List<String> selectColumnNames = SynchroQueryBuilder.newBuilder(this.selectAllQuery).getColumnNames();
		SynchroQueryBuilder insertQuery = SynchroQueryBuilder.newBuilder(this.insertQuery);
		SynchroQueryBuilder updateQuery = SynchroQueryBuilder.newBuilder(this.updateQuery);
		List<String> insertBindColumnNames = Lists.newArrayList();
		for (String columnName : insertQuery.getColumnNames()) {
			if ("?".equals(insertQuery.getColumnValue(columnName))) {
				insertBindColumnNames.add(columnName);
			}
		}
		List<String> updateBindColumnNames = Lists.newArrayList();
		for (String columnName : updateQuery.getColumnNames()) {
			if ("?".equals(updateQuery.getColumnValue(columnName))) {
				updateBindColumnNames.add(columnName);
			}
		}

		StringBuilder insertBuffer = new StringBuilder();
		insertBuffer.append(String.format("[%s] Insert mapping:", getName()));
		StringBuilder updateBuffer = new StringBuilder();
		updateBuffer.append(String.format("[%s] Update mapping:", getName()));
		int i = 0;
		int j = 0;
		for (String selectColumnName : selectColumnNames) {
			String insertColumnName = "N/A";
			if (i < insertBindColumnNames.size()) {
				insertColumnName = insertBindColumnNames.get(i++);
			}
			String updateColumnName = "N/A";
			if (j < updateBindColumnNames.size()) {
				updateColumnName = updateBindColumnNames.get(j++);
			}
			insertBuffer.append("\n\t").append(selectColumnName).append(" -> ").append(insertColumnName);
			updateBuffer.append("\n\t").append(selectColumnName).append(" -> ").append(updateColumnName);
		}
		if (insertBindColumnNames.size() > i) {
			for (int k = i; k < insertBindColumnNames.size(); k++) {
				insertBuffer.append("\n\tN/A  -> ").append(insertBindColumnNames.get(k));
			}
		}
		if (updateBindColumnNames.size() > j) {
			for (int k = j; k < updateBindColumnNames.size(); k++) {
				updateBuffer.append("\n\tN/A  -> ").append(updateBindColumnNames.get(k));
			}
		}

		log.trace(insertBuffer.toString());
		log.trace(updateBuffer.toString());
	}

	protected EventBus initEventBus(boolean enableEvent, List<Object> listeners, String tableName) {
		if (CollectionUtils.isEmpty(listeners) || !enableEvent) {
			return null;
		}
		EventBus eventBus = new EventBus(new SubscriberExceptionHandler() {
			@Override
			public void handleException(Throwable exception, SubscriberExceptionContext context) {
				throw new SynchroTechnicalException(exception);
			}
		});

		if (log.isDebugEnabled()) {
			log.debug(String.format("[%s] Active listeners: ", tableName));
		}

		for (Object listener : listeners) {
			eventBus.register(listener);
			if (log.isDebugEnabled()) {
				log.debug(String.format("[%s]  - %s", tableName, listener.getClass().getName()));
			}
		}
		return eventBus;
	}

	protected void addUniqueConstraintsIfNotPK(String indexName, List<String> columnNames, DuplicateKeyStrategy duplicateKeyStrategy) {
		Preconditions.checkArgument(!uniqueConstraints.containsKey(indexName), "Duplicate unique constraints name");
		Preconditions.checkNotNull(CollectionUtils.isNotEmpty(columnNames));

		// Do not add unique constraints if equals to PK constraints
		if (CollectionUtils.isEqualCollection(pkNames, columnNames)) {
			return;
		}
		if (log.isDebugEnabled()) {
			log.debug(String.format("[%s] Add unique constraint: %s", getName(), indexName));
		}

		uniqueConstraints.put(indexName, columnNames);
		duplicateKeyStrategies.put(indexName, duplicateKeyStrategy);
		hasUniqueConstraints = true;
	}

	protected void checkBuild() {
		Preconditions.checkState(isBuild, "table not build. Please call build() first.");
	}

	protected Map<String, String> getCountPkReferenceQueriesUniquePk(DatabaseMetaData dbMeta) throws SQLException {
		String pkColumnName = pkNames.iterator().next();

		// If a PK is exported (used by another table): store table and columns in a multi map
		Multimap<String, String[]> columnMappingsByTable = ArrayListMultimap.create();
		ResultSet rs = null;
		try {
			rs = dbMeta.getExportedKeys(getCatalog(), getSchema(), getName());
			while (rs.next()) {
				String columnName = rs.getString("PKCOLUMN_NAME").toLowerCase();

				if (pkColumnName.equals(columnName)) {
					String fkTableName = rs.getString("FKTABLE_NAME").toUpperCase();
					String fkColumnName = rs.getString("FKCOLUMN_NAME").toLowerCase();
					columnMappingsByTable.put(fkTableName, new String[] { columnName, fkColumnName });
				}
			}
		} finally {
			Daos.closeSilently(rs);
		}

		// if PK exported: stop
		if (columnMappingsByTable.isEmpty()) {
			return null;
		}

		Map<String, String> result = Maps.newHashMap();
		for (String fkTableName : columnMappingsByTable.keySet()) {
			// Compute the where clause
			StringBuilder whereClause = new StringBuilder();
			Collection<String[]> columnMappings = columnMappingsByTable.get(fkTableName);

			// iterate on PK to keep PKs order in binding parameters
			for (String[] columnMapping : columnMappings) {
				String fkColumnName = columnMapping[1];
				// If more than one reference, need or OR (when 2 referencee one a table - mantis #23134)
				whereClause.append(" OR ")
						.append(fkColumnName)
						.append("=:pk1");
			}

			String sql = String.format("select count(*) from %s where %s",
					fkTableName,
					whereClause.substring(4)
					);
			result.put(fkTableName, sql);
		}

		return result;
	}

	protected Map<String, String> getCountPkReferenceQueriesCompositePk(DatabaseMetaData dbMeta) throws SQLException {
		Set<String> pkColumnNames = getPkNames();

		// If a PK is exported (used by another table): store table and columns in a multi map
		Multimap<String, String[]> columnMappingsByTable = ArrayListMultimap.create();
		ResultSet rs = null;
		try {
			rs = dbMeta.getExportedKeys(getCatalog(), getSchema(), getName());
			while (rs.next()) {
				String columnName = rs.getString("PKCOLUMN_NAME").toLowerCase();

				if (pkColumnNames.contains(columnName)) {
					String fkTableName = rs.getString("FKTABLE_NAME").toUpperCase();
					String fkColumnName = rs.getString("FKCOLUMN_NAME").toLowerCase();
					columnMappingsByTable.put(fkTableName, new String[] { columnName, fkColumnName });
				}
			}
		} finally {
			Daos.closeSilently(rs);
		}

		// if no exported PK: stop
		if (columnMappingsByTable.isEmpty()) {
			return null;
		}

		Map<String, String> result = Maps.newHashMap();
		for (String fkTableName : columnMappingsByTable.keySet()) {
			// Compute the where clause
			StringBuilder whereClause = new StringBuilder();
			Collection<String[]> columnMappings = columnMappingsByTable.get(fkTableName);

			// iterate on PK to keep PKs order in binding parameters
			int pkIndex = 1;
			for (String pkColumnName : pkColumnNames) {
				String pkParamName = ":pk" + pkIndex;
				for (String[] columnMapping : columnMappings) {
					if (columnMapping[0].equals(pkColumnName)) {
						String fkColumnName = columnMapping[1];
						whereClause.append(" AND ")
								.append(fkColumnName)
								.append("=")
								.append(pkParamName);
					}
					else {
						// a fake condition, to be able to bind on each pk column (see SynchroTableDaoImpl)
						whereClause.append(" AND ")
								.append(pkParamName)
								.append(" is not null");
					}
				}
				pkIndex++;
			}

			String sql = String.format("select count(*) from %s where %s",
					fkTableName,
					whereClause.substring(5)
					);
			result.put(fkTableName, sql);
		}

		return result;
	}

}
