package fr.ifremer.adagio.synchro.meta;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.sql.ResultSet;
import java.sql.SQLException;

import org.hibernate.internal.CoreMessageLogger;
import org.jboss.logging.Logger;

/**
 * Store metadata on a link between two tables.
 * <ul>
 * <li>Obtains owner table name {@link #getTableName()};</li>
 * <li>Obtains linked column (for FK only) {@link #getParentJoin()} as a join metadata: used to synchronize data in
 * {@link DataSynchroService};</li>
 * </ul>
 * 
 * @author Benoit Lavenier <benoit.lavenier@e-is.pro>
 * @since 3.5.2
 * 
 */
public class SynchroJoinMetadata {

	private static final CoreMessageLogger LOG = Logger.getMessageLogger(CoreMessageLogger.class, SynchroJoinMetadata.class.getName());

	protected SynchroTableMetadata sourceTable;

	protected SynchroColumnMetadata sourceColumn;

	protected SynchroColumnMetadata targetColumn;

	protected SynchroTableMetadata targetTable;

	protected boolean isValid = false;

	protected boolean isChild = false;

	protected boolean isSourceFk = false;

	public SynchroJoinMetadata(
			ResultSet rs,
			SynchroTableMetadata table,
			SynchroDatabaseMetadata meta) throws SQLException {

		// String pkCatalog = rs.getString("PKTABLE_CAT");
		// String pkSchema = rs.getString("PKTABLE_SCHEM");
		String pkTableName = rs.getString("PKTABLE_NAME").toLowerCase();
		String pkColumnName = rs.getString("PKCOLUMN_NAME").toLowerCase();
		String fkTableName = rs.getString("FKTABLE_NAME").toLowerCase();
		String fkColumnName = rs.getString("FKCOLUMN_NAME").toLowerCase();

		// Load PK metadata
		SynchroTableMetadata pkTable = meta.getLoadedTable(pkTableName);
		SynchroColumnMetadata pkColumn = null;
		if (pkTable == null) {
			if (LOG.isTraceEnabled()) {
				LOG.trace(String.format("[%s] Ignore join to %s: table not found",
						table.getName(), pkTableName.toUpperCase()));
			}
			return;
		}
		pkColumn = pkTable.getColumn(pkColumnName);
		if (pkColumn == null) {
			if (LOG.isTraceEnabled()) {
				LOG.trace(String.format("[%s] Ignore join to %s: column %s not found",
						table.getName(), pkTableName.toUpperCase(), pkColumnName.toUpperCase()));
			}
			return;
		}

		// Load FK metadata
		SynchroTableMetadata fkTable = meta.getLoadedTable(fkTableName);
		SynchroColumnMetadata fkColumn = null;
		if (fkTable != null) {
			fkColumn = fkTable.getColumn(fkColumnName);
		}
		if (fkTable == null) {
			// Ignoring the join, because referenced table not bound
			if (LOG.isTraceEnabled()) {
				LOG.trace(String.format("[%s] Ignore join from %s: table not found",
						table.getName(), fkTableName.toUpperCase()));
			}
			return;
		}
		if (fkTable == null || fkColumn == null) {
			// Ignoring the join, because referenced column not bound
			if (LOG.isTraceEnabled()) {
				LOG.trace(String.format("[%s] Ignore join from %s: column %s not found",
						table.getName(), fkTableName.toUpperCase(), fkColumnName.toUpperCase()));
			}
			return;
		}

		if (pkTable == table) {
			this.sourceTable = pkTable;
			this.sourceColumn = pkColumn;

			this.targetTable = fkTable;
			this.targetColumn = fkColumn;

			this.isChild = true;
			this.isSourceFk = false;
		}
		else {
			this.sourceTable = fkTable;
			this.sourceColumn = fkColumn;

			this.targetTable = pkTable;
			this.targetColumn = pkColumn;

			this.isChild = false;
			this.isSourceFk = true;
		}
		isValid = true;
	}

	public SynchroJoinMetadata(
			SynchroTableMetadata table,
			SynchroTableMetadata pkTable,
			SynchroColumnMetadata pkColumn,
			SynchroTableMetadata fkTable,
			SynchroColumnMetadata fkColumn) {

		if (pkTable == table) {
			this.sourceTable = pkTable;
			this.sourceColumn = pkColumn;

			this.targetTable = fkTable;
			this.targetColumn = fkColumn;

			this.isChild = true;
			this.isSourceFk = false;
		}
		else {
			this.sourceTable = fkTable;
			this.sourceColumn = fkColumn;

			this.targetTable = pkTable;
			this.targetColumn = pkColumn;

			this.isChild = false;
			this.isSourceFk = true;
		}
		isValid = true;
	}

	public boolean isChild() {
		return isChild;
	}

	public int hashCode() {
		return sourceColumn.hashCode() + targetColumn.hashCode();
	}

	public boolean equals(Object obj) {
		if (obj instanceof SynchroJoinMetadata) {
			SynchroJoinMetadata otherJoin = (SynchroJoinMetadata) obj;
			return sourceColumn.equals(otherJoin.sourceColumn) && targetColumn.equals(otherJoin.targetColumn);
		}
		return this.equals(obj);
	}

	public SynchroTableMetadata getTargetTable() {
		return targetTable;
	}

	public SynchroColumnMetadata getTargetColumn() {
		return targetColumn;
	}

	public SynchroTableMetadata getSourceTable() {
		return sourceTable;
	}

	public SynchroColumnMetadata getSourceColumn() {
		return sourceColumn;
	}

	public boolean isValid() {
		return isValid;
	}

	public void setIsValid(boolean isValid) {
		this.isValid = isValid;
	}

	public String toString() {
		return String.format("JoinMetadata(%s.%s = %s.%s)",
				sourceTable.getName(), sourceColumn.getName(),
				targetTable.getName(), targetColumn.getName());
	}

	/**
	 * @return true if the source side = the FK side. false if source = PK
	 */
	public boolean isSourceFk() {
		return isSourceFk;
	}

	/**
	 * @return true if the current table has the targeted PK. false if has the FK
	 */
	public boolean isTargetPk() {
		return !isSourceFk;
	}

	public SynchroTableMetadata getPkTable() {
		return !isSourceFk ? sourceTable : targetTable;
	}

	public SynchroColumnMetadata getPkColumn() {
		return !isSourceFk ? sourceColumn : targetColumn;
	}

	public SynchroTableMetadata getFkTable() {
		return isSourceFk ? sourceTable : targetTable;
	}

	public SynchroColumnMetadata getFkColumn() {
		return isSourceFk ? sourceColumn : targetColumn;
	}
}
