package fr.ifremer.adagio.synchro.intercept;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.io.IOException;
import java.sql.SQLException;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.hibernate.tool.hbm2ddl.ColumnMetadata;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import fr.ifremer.adagio.synchro.SynchroTechnicalException;
import fr.ifremer.adagio.synchro.dao.SynchroTableDao;
import fr.ifremer.adagio.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata;
import fr.ifremer.adagio.synchro.service.SynchroDatabaseConfiguration;

public class SynchroInterceptorBase implements SynchroInterceptor {

	private SynchroInterceptor next;

	private boolean enableOnRead = false;

	private boolean enableOnWrite = false;

	private SynchroDatabaseConfiguration config;

	public SynchroInterceptorBase() {
		this.next = null;
	}

	@Override
	public boolean apply(SynchroDatabaseConfiguration config) {
		this.config = config;
		return true;
	}

	public SynchroInterceptorBase(SynchroInterceptor next) {
		this.next = next;
	}

	@Override
	public SynchroInterceptorBase clone() {

		try {
			SynchroInterceptorBase newInstance = this.getClass().newInstance();
			newInstance.config = this.config;
			return newInstance;
		} catch (InstantiationException e) {
			throw new SynchroTechnicalException(String.format(
					"Could not clone the interceptor class %s. Make sure method clone() has been implemented.", this.getClass().getName()));
		} catch (IllegalAccessException e) {
			throw new SynchroTechnicalException(String.format(
					"Could not clone the interceptor class %s. Make sure method clone() has been implemented.", this.getClass().getName()));
		}
	}

	public boolean apply(SynchroDatabaseMetadata meta, TableMetadata table) {
		return false;
	}

	@Override
	public void setNext(SynchroInterceptor next) {
		this.next = next;
	}

	public SynchroInterceptor getNext() {
		return this.next;
	}

	@Override
	public final void close() throws IOException {
		doClose();

		if (next != null) {
			next.close();
		}
	}

	@Override
	public final void onRead(Object[] data, SynchroTableDao sourceDao, SynchroTableDao targetDao) throws SQLException {
		doOnRead(data, sourceDao, targetDao);

		if (next != null) {
			next.onRead(data, sourceDao, targetDao);
		}
	}

	@Override
	public final void onWrite(Object[] data, List<Object> pk, SynchroTableDao sourceDao, SynchroTableDao targetDao, SynchroOperationRepository buffer)
			throws SQLException {
		doOnWrite(data, pk, sourceDao, targetDao, buffer);

		if (next != null) {
			next.onWrite(data, pk, sourceDao, targetDao, buffer);
		}
	}

	@Override
	public final void onDelete(List<Object> pk, SynchroTableDao sourceDao, SynchroTableDao targetDao, SynchroOperationRepository buffer)
			throws SQLException {
		doOnDelete(pk, sourceDao, targetDao, buffer);

		if (next != null) {
			next.onDelete(pk, sourceDao, targetDao, buffer);
		}
	}

	@Override
	public final void onDetach(List<Object> pk, SynchroTableDao sourceDao, SynchroTableDao targetDao, SynchroOperationRepository buffer)
			throws SQLException {
		doOnDetach(pk, sourceDao, targetDao, buffer);

		if (next != null) {
			next.onDetach(pk, sourceDao, targetDao, buffer);
		}
	}

	@Override
	public boolean enableOnRead() {
		return enableOnRead;
	}

	@Override
	public boolean enableOnWrite() {
		return enableOnWrite;
	}

	public void setEnableOnRead(boolean enableOnRead) {
		this.enableOnRead = enableOnRead;
	}

	public void setEnableOnWrite(boolean enableOnWrite) {
		this.enableOnWrite = enableOnWrite;
	}

	/**
	 * Return the database configuration (without cast).<br/>
	 * Subclasses could define another getter method, with cast
	 * 
	 * @return
	 */
	protected SynchroDatabaseConfiguration getDefaultDatabaseConfiguration() {
		return config;
	}

	/* -- protected method -- */

	protected void doClose() throws IOException {
	}

	protected void doOnRead(Object[] data, SynchroTableDao sourceDao, SynchroTableDao targetDao) throws SQLException {
	}

	protected void doOnWrite(Object[] data, List<Object> pk, SynchroTableDao sourceDao, SynchroTableDao targetDao, SynchroOperationRepository buffer)
			throws SQLException {
	}

	protected void doOnDelete(List<Object> pk, SynchroTableDao sourceDao, SynchroTableDao targetDao, SynchroOperationRepository buffer)
			throws SQLException {
	}

	protected void doOnDetach(List<Object> pk, SynchroTableDao sourceDao, SynchroTableDao targetDao, SynchroOperationRepository buffer)
			throws SQLException {
	}

	protected boolean hasColumns(TableMetadata table, String... columnNames) {

		Map<String, ColumnMetadata> delegateColumns = SynchroTableMetadata.getColumns(table);
		for (String columnName : columnNames) {
			if (!delegateColumns.containsKey(columnName.toLowerCase())) {
				return false;
			}
		}
		return true;
	}

	protected boolean hasColumns(SynchroTableMetadata table, String... columnNames) {
		Set<String> delegateColumns = table.getColumnNames();
		for (String columnName : columnNames) {
			if (!delegateColumns.contains(columnName.toLowerCase())) {
				return false;
			}
		}
		return true;
	}

}
