package fr.ifremer.adagio.synchro.config;

/*
 * #%L
 * SIH-Adagio :: Core for Allegro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import static org.nuiton.i18n.I18n.t;

import java.io.File;
import java.util.Locale;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.config.ApplicationConfig;
import org.nuiton.config.ApplicationConfigHelper;
import org.nuiton.config.ApplicationConfigProvider;
import org.nuiton.config.ArgumentsParserException;

import com.google.common.base.Charsets;
import com.google.common.collect.ImmutableSet;

import fr.ifremer.adagio.synchro.SynchroTechnicalException;
import fr.ifremer.adagio.synchro.dao.Daos;

public class SynchroConfiguration {
	/** Logger. */
	private static final Log log = LogFactory.getLog(SynchroConfiguration.class);

	/**
	 * Delegate application config.
	 */
	protected final ApplicationConfig applicationConfig;

	private static SynchroConfiguration instance;

	public static SynchroConfiguration getInstance() {
		return instance;
	}

	public static void setInstance(SynchroConfiguration instance) {
		SynchroConfiguration.instance = instance;
	}

	protected final String[] optionKeyToNotSave;

	protected File configFile;

	public SynchroConfiguration(ApplicationConfig applicationConfig) {
		super();
		this.applicationConfig = applicationConfig;
		this.optionKeyToNotSave = null;
	}

	public SynchroConfiguration(String file, String... args) {
		super();
		this.applicationConfig = new ApplicationConfig();
		this.applicationConfig.setEncoding(Charsets.UTF_8.name());
		this.applicationConfig.setConfigFileName(file);

		// get all config providers
		Set<ApplicationConfigProvider> providers =
				ApplicationConfigHelper.getProviders(null,
						null,
						null,
						true);

		// load all default options
		ApplicationConfigHelper.loadAllDefaultOption(applicationConfig,
				providers);

		// Load actions
		for (ApplicationConfigProvider provider : providers) {
			applicationConfig.loadActions(provider.getActions());
		}

		// Load module actions (not in provider)
		applicationConfig.loadActions(SynchroConfigurationAction.values());

		applicationConfig.addAlias("-u", "--option", SynchroConfigurationOption.JDBC_USERNAME.getKey());
		applicationConfig.addAlias("--user", "--option", SynchroConfigurationOption.JDBC_USERNAME.getKey());
		applicationConfig.addAlias("-p", "--option", SynchroConfigurationOption.JDBC_PASSWORD.getKey());
		applicationConfig.addAlias("--password", "--option", SynchroConfigurationOption.JDBC_PASSWORD.getKey());
		applicationConfig.addAlias("-db", "--option", SynchroConfigurationOption.JDBC_URL.getKey());
		applicationConfig.addAlias("--database", "--option", SynchroConfigurationOption.JDBC_URL.getKey());

		applicationConfig.addAlias("-iu", "--option", SynchroConfigurationOption.IMPORT_JDBC_USERNAME.getKey());
		applicationConfig.addAlias("--import-user", "--option", SynchroConfigurationOption.IMPORT_JDBC_USERNAME.getKey());
		applicationConfig.addAlias("-ip", "--option", SynchroConfigurationOption.IMPORT_JDBC_PASSWORD.getKey());
		applicationConfig.addAlias("--import-password", "--option", SynchroConfigurationOption.IMPORT_JDBC_PASSWORD.getKey());
		applicationConfig.addAlias("-idb", "--option", SynchroConfigurationOption.IMPORT_JDBC_URL.getKey());
		applicationConfig.addAlias("--import-database", "--option", SynchroConfigurationOption.IMPORT_JDBC_URL.getKey());

		// Server options
		applicationConfig.addAlias("--port", "--option", SynchroConfigurationOption.SERVER_PORT.getKey());
		applicationConfig.addAlias("--keystore", "--option", SynchroConfigurationOption.SERVER_SSL_KEYSTORE.getKey());

		// get all transient and final option keys
		Set<String> optionToSkip =
				ApplicationConfigHelper.getTransientOptionKeys(providers);

		if (log.isDebugEnabled()) {
			log.debug("Option that won't be saved: " + optionToSkip);
		}
		optionKeyToNotSave = optionToSkip.toArray(new String[optionToSkip.size()]);

		try {
			applicationConfig.parse(args);

		} catch (ArgumentsParserException e) {
			throw new SynchroTechnicalException(t("adagio.config.parse.error"), e);
		}

		// TODO Review this, this is very dirty to do this...
		File allegroBasedir = applicationConfig.getOptionAsFile(
				SynchroConfigurationOption.BASEDIR.getKey());

		if (allegroBasedir == null) {
			allegroBasedir = new File("");
		}
		if (!allegroBasedir.isAbsolute()) {
			allegroBasedir = new File(allegroBasedir.getAbsolutePath());
		}
		if (allegroBasedir.getName().equals("..")) {
			allegroBasedir = allegroBasedir.getParentFile().getParentFile();
		}
		if (allegroBasedir.getName().equals(".")) {
			allegroBasedir = allegroBasedir.getParentFile();
		}
		if (log.isInfoEnabled()) {
			log.info("Application basedir: " + allegroBasedir);
		}
		applicationConfig.setOption(
				SynchroConfigurationOption.BASEDIR.getKey(),
				allegroBasedir.getAbsolutePath());
	}

	public File getConfigFile() {
		if (configFile == null) {
			File dir = getBasedir();
			if (dir == null || !dir.exists()) {
				dir = new File(applicationConfig.getUserConfigDirectory());
			}
			configFile = new File(dir, applicationConfig.getConfigFileName());
		}
		return configFile;
	}

	/** @return {@link SynchroConfigurationOption#BASEDIR} value */
	public File getBasedir() {
		File result = applicationConfig.getOptionAsFile(SynchroConfigurationOption.BASEDIR.getKey());
		return result;
	}

	/** @return {@link SynchroConfigurationOption#DATA_DIRECTORY} value */
	public File getDataDirectory() {
		File result = applicationConfig.getOptionAsFile(SynchroConfigurationOption.DATA_DIRECTORY.getKey());
		return result;
	}

	public ApplicationConfig getApplicationConfig() {
		return applicationConfig;
	}

	public File getDbDirectory() {
		return applicationConfig.getOptionAsFile(SynchroConfigurationOption.DB_DIRECTORY.getKey());
	}

	public void setDbDirectory(File dbDirectory) {
		applicationConfig.setOption(SynchroConfigurationOption.DB_DIRECTORY.getKey(), dbDirectory.getPath());
	}

	public File getDbAttachmentDirectory() {
		return applicationConfig.getOptionAsFile(SynchroConfigurationOption.DB_ATTACHMENT_DIRECTORY.getKey());
	}

	public File getCacheDirectory() {
		return applicationConfig.getOptionAsFile(SynchroConfigurationOption.DB_CACHE_DIRECTORY.getKey());
	}

	public File getDbBackupDirectory() {
		return applicationConfig.getOptionAsFile(SynchroConfigurationOption.DB_BACKUP_DIRECTORY.getKey());
	}

	public String getHibernateDialect() {
		return applicationConfig.getOption(SynchroConfigurationOption.HIBERNATE_DIALECT.getKey());
	}

	public String getJdbcDriver() {
		return applicationConfig.getOption(SynchroConfigurationOption.JDBC_DRIVER.getKey());
	}

	public String getJdbcURL() {
		return applicationConfig.getOption(SynchroConfigurationOption.JDBC_URL.getKey());
	}

	public String getDbName() {
		return applicationConfig.getOption(SynchroConfigurationOption.DB_NAME.getKey());
	}

	public String getDbValidationQuery() {
		return applicationConfig.getOption(SynchroConfigurationOption.DB_VALIDATION_QUERY.getKey());
	}

	public String getJdbcUsername() {
		return applicationConfig.getOption(SynchroConfigurationOption.JDBC_USERNAME.getKey());
	}

	public String getJdbcPassword() {
		return applicationConfig.getOption(SynchroConfigurationOption.JDBC_PASSWORD.getKey());
	}

	public File getI18nDirectory() {
		return applicationConfig.getOptionAsFile(
				SynchroConfigurationOption.I18N_DIRECTORY.getKey());
	}

	public Locale getI18nLocale() {
		return applicationConfig.getOptionAsLocale(
				SynchroConfigurationOption.I18N_LOCALE.getKey());
	}

	public void setI18nLocale(Locale locale) {
		applicationConfig.setOption(SynchroConfigurationOption.I18N_LOCALE.getKey(), locale.toString());
	}

	public String getImportJdbcURL() {
		return applicationConfig.getOption(SynchroConfigurationOption.IMPORT_JDBC_URL.getKey());
	}

	public String getImportJdbcUsername() {
		return applicationConfig.getOption(SynchroConfigurationOption.IMPORT_JDBC_USERNAME.getKey());
	}

	public String getImportJdbcPassword() {
		return applicationConfig.getOption(SynchroConfigurationOption.IMPORT_JDBC_PASSWORD.getKey());
	}

	public String getImportJdbcSchema() {
		return applicationConfig.getOption(SynchroConfigurationOption.IMPORT_JDBC_SCHEMA.getKey());
	}

	public String getImportJdbcCatalog() {
		return applicationConfig.getOption(SynchroConfigurationOption.IMPORT_JDBC_CATALOG.getKey());
	}

	public String getImportJdbcDriver() {
		return applicationConfig.getOption(SynchroConfigurationOption.IMPORT_JDBC_DRIVER.getKey());
	}

	public int getImportJdbcBatchSize() {
		// If batch disable, force to batch-size=1
		if (!isImportJdbcBatchEnable()) {
			return 1;
		}

		int importJdbcBatchSize = applicationConfig.getOptionAsInt(SynchroConfigurationOption.IMPORT_JDBC_BATCH_SIZE.getKey());

		// If negative value, use the default value
		if (importJdbcBatchSize <= 0) {
			importJdbcBatchSize = Integer.parseInt(SynchroConfigurationOption.IMPORT_JDBC_BATCH_SIZE.getDefaultValue());
			log.warn(String.format("Invalid batch size [%s] on configuration property [%s]. Will use default value [%s]",
					importJdbcBatchSize,
					SynchroConfigurationOption.IMPORT_JDBC_BATCH_SIZE.getKey(),
					importJdbcBatchSize
					));
		}
		return importJdbcBatchSize;
	}

	public int getImportJdbcFetchSize() {
		return applicationConfig.getOptionAsInt(SynchroConfigurationOption.IMPORT_JDBC_FETCH_SIZE.getKey());
	}

	public boolean isImportJdbcBatchEnable() {
		return applicationConfig.getOptionAsBoolean(SynchroConfigurationOption.IMPORT_JDBC_BATCH_ENABLE.getKey());
	}

	public String getImportHibernateDialect() {
		return applicationConfig.getOption(SynchroConfigurationOption.IMPORT_HIBERNATE_DIALECT.getKey());
	}

	public String getSynchronizationStatusSynchronized() {
		return applicationConfig.getOption(SynchroConfigurationOption.SYNCHRONIZATION_STATUS_SYNCHRONIZED.getKey());
	}

	public String getTempQueryParemeterUserIdColumn() {
		return applicationConfig.getOption(SynchroConfigurationOption.TEMP_QUERY_PARAMETER_USER_ID.getKey());
	}

	public String getSynchronizationStatusReadyToSynchronize() {
		return applicationConfig.getOption(SynchroConfigurationOption.SYNCHRONIZATION_STATUS_READY_TO_SYNCHRONIZE.getKey());
	}

	public String getSynchronizationStatusDeleted() {
		return applicationConfig.getOption(SynchroConfigurationOption.SYNCHRONIZATION_STATUS_DELETED.getKey());
	}

	public String getSynchronizationStatusDirty() {
		return applicationConfig.getOption(SynchroConfigurationOption.SYNCHRONIZATION_STATUS_DIRTY.getKey());
	}

	public int getServerPort() {
		return applicationConfig.getOptionAsInt(SynchroConfigurationOption.SERVER_PORT.getKey());
	}

	public File getServerSslKeyStore() {
		return applicationConfig.getOptionAsFile(SynchroConfigurationOption.SERVER_SSL_KEYSTORE.getKey());
	}

	public Properties getConnectionProperties() {
		return Daos.getConnectionProperties(
				getJdbcURL(),
				getJdbcUsername(),
				getJdbcPassword(),
				null,
				getHibernateDialect(),
				getJdbcDriver());
	}

	public Properties getImportConnectionProperties() {
		return Daos.getConnectionProperties(
				getImportJdbcURL(),
				getImportJdbcUsername(),
				getImportJdbcPassword(),
				getImportJdbcSchema(),
				getImportHibernateDialect(),
				getImportJdbcDriver());
	}

	public Set<String> getImportDataTablesIncludes() {
		String dataTablesStr = applicationConfig.getOption(SynchroConfigurationOption.IMPORT_TABLES_DATA_INCLUDES.getKey());
		Set<String> result = null;
		if (StringUtils.isNotBlank(dataTablesStr)) {
			result = ImmutableSet.<String> builder()
					.add(dataTablesStr.toUpperCase().split("\\s*,\\s*"))
					.build();
		}
		return result;
	}

	public Set<String> getImportReferentialTablesIncludes() {
		String referentialTablesStr = applicationConfig.getOption(SynchroConfigurationOption.IMPORT_TABLES_REFERENTIAL_INCLUDES.getKey());
		Set<String> result = null;
		if (StringUtils.isNotBlank(referentialTablesStr)) {
			result = ImmutableSet.<String> builder().add(referentialTablesStr.split("\\s*,\\s*")).build();
		}
		return result;
	}

	public String getColumnId() {
		return applicationConfig.getOption(SynchroConfigurationOption.DB_COLUMN_ID.getKey());
	}

	public String getColumnUpdateDate() {
		return applicationConfig.getOption(SynchroConfigurationOption.DB_COLUMN_UPDATE_DATE.getKey());
	}

	public String getSequenceSuffix() {
		return applicationConfig.getOption(SynchroConfigurationOption.DB_SEQUENCE_SUFFIX.getKey());
	}

	public int getMaxSqlNameLength() {
		return applicationConfig.getOptionAsInt(SynchroConfigurationOption.DB_MAX_SQL_NAME_LENGTH.getKey());
	}

}
