package fr.ifremer.adagio.synchro.service.data;

/*
 * #%L
 * Tutti :: Persistence
 * $Id: DataSynchroServiceImpl.java 1573 2014-02-04 16:41:40Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/trunk/tutti-persistence/src/main/java/fr/ifremer/adagio/core/service/technical/synchro/DataSynchroServiceImpl.java $
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import static org.nuiton.i18n.I18n.t;

import java.io.File;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.sql.DataSource;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.cfg.Configuration;
import org.hibernate.cfg.Environment;
import org.nuiton.util.TimeLog;
import org.springframework.jdbc.datasource.DataSourceUtils;

import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.Lists;

import fr.ifremer.adagio.synchro.config.SynchroConfiguration;
import fr.ifremer.adagio.synchro.dao.DaoUtils;
import fr.ifremer.adagio.synchro.dao.SynchroTableDao;
import fr.ifremer.adagio.synchro.dao.SynchroTableDaoImpl;
import fr.ifremer.adagio.synchro.meta.SynchroColumnMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroJoinMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroMetadataUtils;
import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata.TableInsertStrategy;
import fr.ifremer.adagio.synchro.service.SynchroBaseService;
import fr.ifremer.adagio.synchro.service.SynchroContext;
import fr.ifremer.adagio.synchro.service.SynchroSchemaValidationException;
import fr.ifremer.adagio.synchro.service.SynchroResult;
import fr.ifremer.adagio.synchro.service.SynchroServiceUtils;
import fr.ifremer.adagio.synchro.service.SynchroTableOperationBuffer;
import fr.ifremer.adagio.synchro.type.ProgressionModel;

/**
 * Created on 1/14/14.
 * 
 * @author Benoit Lavenier <benoit.lavenier@e-is.pro>
 * @since 3.5.2
 */
public class DataSynchroServiceImpl extends SynchroBaseService implements DataSynchroService {

	/** Logger. */
	private static final Log log =
			LogFactory.getLog(DataSynchroServiceImpl.class);

	private static final TimeLog TIME =
			new TimeLog(DataSynchroServiceImpl.class);

	public DataSynchroServiceImpl(DataSource dataSource, SynchroConfiguration config) {
		super(dataSource, config);
	}

	public DataSynchroServiceImpl() {
		super();
	}

	@Override
	public SynchroContext createSynchroContext(File sourceDbDirectory) {

		String dbName = config.getDbName();
		Properties targetConnectionProperties = config.getConnectionProperties();

		Properties sourceConnectionProperties = new Properties(targetConnectionProperties);
		sourceConnectionProperties.setProperty(Environment.URL,
				DaoUtils.getJdbcUrl(sourceDbDirectory, dbName));

		Set<String> tableToIncludes = config.getImportDataTablesIncludes();

		SynchroContext context = SynchroContext.newContext(
				tableToIncludes,
				sourceConnectionProperties,
				targetConnectionProperties,
				new SynchroResult());
		return context;
	}

	@Override
	public SynchroContext createSynchroContext(Properties sourceConnectionProperties) {

		Properties targetConnectionProperties = config.getConnectionProperties();

		Set<String> tableToIncludes = config.getImportDataTablesIncludes();

		SynchroContext context = SynchroContext.newContext(
				tableToIncludes,
				sourceConnectionProperties,
				targetConnectionProperties,
				new SynchroResult());
		return context;
	}

	@Override
	public void prepare(SynchroContext synchroContext) {

		Preconditions.checkNotNull(synchroContext);

		Properties sourceConnectionProperties = synchroContext.getSourceConnectionProperties();
		Preconditions.checkNotNull(sourceConnectionProperties);

		Properties targetConnectionProperties = synchroContext.getTargetConnectionProperties();
		Preconditions.checkNotNull(targetConnectionProperties);

		Set<String> tableNames = synchroContext.getTableNames();
		Predicate<String> tableFilter = synchroContext.getTableFilter();
		if (CollectionUtils.isEmpty(tableNames) && tableFilter == null) {
			log.info(t("adagio.persistence.synchronizeData.prepare.noTableFilter"));
		}

		SynchroResult result = synchroContext.getResult();
		Preconditions.checkNotNull(result);

		result.setLocalUrl(getUrl(targetConnectionProperties));
		result.setRemoteUrl(getUrl(sourceConnectionProperties));

		Connection targetConnection = null;
		Connection sourceConnection = null;
		try {

			ProgressionModel progressionModel = result.getProgressionModel();
			progressionModel.setMessage(t("adagio.persistence.synchronizeData.prepare.step1"));

			// create target connection
			targetConnection = createConnection(targetConnectionProperties);

			progressionModel.setMessage(t("adagio.persistence.synchronizeData.prepare.step2"));

			// create source Connection
			sourceConnection = createConnection(sourceConnectionProperties);

			// load metas
			SynchroDatabaseMetadata targetMeta =
					SynchroDatabaseMetadata.loadDatabaseMetadata(
							targetConnection,
							getTargetDialect(synchroContext),
							getConfiguration(targetConnectionProperties),
							synchroContext,
							tableNames,
							tableFilter,
							null, /* no column filter */
							true /* load join metadata */);

			SynchroDatabaseMetadata sourceMeta =
					SynchroDatabaseMetadata.loadDatabaseMetadata(
							sourceConnection,
							getSourceDialect(synchroContext),
							getConfiguration(sourceConnectionProperties),
							synchroContext,
							tableNames,
							tableFilter,
							null, /* no column filter */
							true /* load join metadata */);

			progressionModel.setMessage(t("adagio.persistence.synchronizeData.prepare.step3"));

			// check schema
			try {
				SynchroServiceUtils.checkSchemas(sourceMeta, targetMeta, true, false, result);
			} catch (SynchroSchemaValidationException e) {
				log.error(e.getMessage());
				result.setError(e);
				return;
			}

			if (result.isSuccess()) {

				// prepare model (compute update date, count rows to update,...)
				Set<String> rootTableNames = targetMeta.getLoadedRootTableNames();
				if (rootTableNames.size() == 0 && log.isWarnEnabled()) {
					log.warn(t("adagio.persistence.synchronizeData.prepare.noRootTable"));
				}

				for (String tableName : rootTableNames) {

					long t0 = TimeLog.getTime();

					progressionModel.setMessage(t("adagio.persistence.synchronizeData.prepare.step4", tableName));

					SynchroTableMetadata sourceTable = sourceMeta.getTable(tableName);
					SynchroTableMetadata targetTable = targetMeta.getTable(tableName);

					if (log.isDebugEnabled()) {
						log.debug("Prepare table: " + tableName);
					}
					prepareRootTable(
							sourceTable,
							targetTable,
							sourceConnection,
							targetConnection,
							result);

					TIME.log(t0, "prepare table " + tableName);
				}

				long totalRows = result.getTotalRows();
				if (log.isInfoEnabled()) {
					log.info("Total rows to update: " + totalRows);
				}
				targetConnection.rollback();
			}
		} catch (SQLException e) {
			try {
				if (targetConnection != null) {
					targetConnection.rollback();
				}
			} catch (SQLException e1) {

				// ignore the rolback error
			}
			result.setError(e);
		} finally {
			releaseConnection(sourceConnection);
			releaseConnection(targetConnection);
		}
	}

	@Override
	public void synchronize(SynchroContext synchroContext) {

		Preconditions.checkNotNull(synchroContext);

		Properties sourceConnectionProperties = synchroContext.getSourceConnectionProperties();
		Preconditions.checkNotNull(sourceConnectionProperties);

		Properties targetConnectionProperties = synchroContext.getTargetConnectionProperties();
		Preconditions.checkNotNull(targetConnectionProperties);

		Set<String> tableNames = synchroContext.getTableNames();
		Predicate<String> tableFilter = synchroContext.getTableFilter();

		SynchroResult result = synchroContext.getResult();
		Preconditions.checkNotNull(result);

		Connection targetConnection = null;
		Connection sourceConnection = null;
		try {

			// create source Connection
			sourceConnection = createConnection(sourceConnectionProperties);

			// create target connection
			targetConnection = createConnection(targetConnectionProperties);

			// Create column filter (exclude missing optional column)
			Predicate<SynchroColumnMetadata> columnFilter = null;
			if (!result.getMissingOptionalColumnNameMaps().isEmpty()) {
				columnFilter = SynchroMetadataUtils.newExcludeColumnPredicate(result.getMissingOptionalColumnNameMaps());
			}

			// load metas
			SynchroDatabaseMetadata dbMetas =
					SynchroDatabaseMetadata.loadDatabaseMetadata(
							targetConnection,
							getTargetDialect(synchroContext),
							getConfiguration(targetConnectionProperties),
							synchroContext,
							tableNames,
							tableFilter,
							columnFilter,
							true /* load join metadata */
							);

			// set total in progression model
			ProgressionModel progressionModel = result.getProgressionModel();
			progressionModel.setTotal(result.getTotalRows());

			List<SynchroTableOperationBuffer> pendingOperations = Lists.newArrayList();

			// Add each root table to operation
			for (String tableName : dbMetas.getLoadedRootTableNames()) {
				SynchroTableOperationBuffer tableOperationBuffer = new SynchroTableOperationBuffer(tableName);
				pendingOperations.add(tableOperationBuffer);
			}

			while (CollectionUtils.isNotEmpty(pendingOperations)) {
				List<SynchroTableOperationBuffer> pendingOperationsCopy = Lists.newArrayList(pendingOperations);
				pendingOperations.clear();
				for (SynchroTableOperationBuffer pendingOperation : pendingOperationsCopy) {

					// Execute synchronization operation
					List<SynchroTableOperationBuffer> newPendingOperations = synchronizeOperation(
							pendingOperation,
							dbMetas,
							sourceConnection,
							targetConnection,
							synchroContext,
							result);

					// Add new operation if not empty
					if (CollectionUtils.isNotEmpty(newPendingOperations)) {
						pendingOperations.addAll(newPendingOperations);
					}
				}
			}

			if (log.isInfoEnabled()) {
				long totalInserts = result.getTotalInserts();
				long totalUpdates = result.getTotalUpdates();
				log.info("Total rows to treat: " + result.getTotalRows());
				log.info("Total rows inserted: " + totalInserts);
				log.info("Total rows  updated: " + totalUpdates);
				log.info("Total rows  treated: " + (totalInserts + totalUpdates));
			}

			progressionModel.setMessage(t("adagio.persistence.synchronizeData.synchronize.step2"));

			targetConnection.commit();

		} catch (SQLException e) {
			try {
				if (targetConnection != null) {
					targetConnection.rollback();
				}
			} catch (SQLException e1) {

				// ignore the rollback error
			}
			result.setError(e);
		} catch (Exception e) {
			try {
				if (targetConnection != null) {
					targetConnection.rollback();
				}
			} catch (SQLException e1) {

				// ignore the rollback error
			}
			result.setError(e);
		} finally {
			releaseConnection(sourceConnection);
			releaseConnection(targetConnection);
		}
	}

	protected List<SynchroTableOperationBuffer> synchronizeOperation(
			SynchroTableOperationBuffer synchroOperation,
			SynchroDatabaseMetadata dbMetas,
			Connection sourceConnection,
			Connection targetConnection,
			SynchroContext context,
			SynchroResult result
			) throws SQLException {

		boolean hasChildToUpdate = MapUtils.isNotEmpty(synchroOperation.getChildToUpdate());

		boolean hasChildToUpdateComplex = MapUtils.isNotEmpty(synchroOperation.getChildToUpdateComplexMap());

		boolean hasMissingUpdates = MapUtils.isNotEmpty(synchroOperation.getMissingUpdates());

		List<SynchroTableOperationBuffer> pendingOperations = Lists.newArrayList();

		ProgressionModel progressionModel = result.getProgressionModel();

		// Root table
		if (!hasChildToUpdate
				&& !hasChildToUpdateComplex
				&& !hasMissingUpdates) {
			String tableName = synchroOperation.getTableName();
			SynchroTableMetadata table = dbMetas.getTable(tableName);
			long t0 = TimeLog.getTime();

			progressionModel.setMessage(t("adagio.persistence.synchronizeData.synchronize.step1", tableName));

			if (log.isInfoEnabled()) {
				log.info("Synchronize root table: " + tableName);
			}
			long countToUpdate = result.getNbRows(tableName);

			if (countToUpdate > 0) {

				synchronizeRootTable(
						table,
						sourceConnection,
						targetConnection,
						context,
						result,
						pendingOperations);

				TIME.log(t0, "synchronize table " + tableName);

			}
		}

		// Synchronize childs tables
		else if (hasChildToUpdate) {

			String parentTableName = synchroOperation.getTableName();
			Map<String, Map<String, List<Object>>> childToUpdateMap = synchroOperation.getChildToUpdate();

			for (String tableName : childToUpdateMap.keySet()) {
				Map<String, List<Object>> childToUpdate = childToUpdateMap.get(tableName);

				SynchroTableMetadata table = dbMetas.getTable(tableName);

				long t0 = TimeLog.getTime();

				if (log.isInfoEnabled()) {
					log.info(String.format("Synchronize child table: %s (child of %s)", tableName, parentTableName));
				}

				for (String columnName : childToUpdate.keySet()) {
					List<Object> columnValues = childToUpdate.get(columnName);

					synchronizeChildTable(
							table,
							columnName,
							columnValues,
							sourceConnection,
							targetConnection,
							context,
							result,
							pendingOperations);

				}
				TIME.log(t0, "synchronize child table " + tableName);
			}

			// Clean operation
			synchroOperation.getChildToUpdate().clear();

		}

		// missing update :
		else if (hasMissingUpdates) {

			String tableName = synchroOperation.getTableName();
			SynchroTableMetadata table = dbMetas.getTable(tableName);

			long t0 = TimeLog.getTime();

			if (log.isInfoEnabled()) {
				log.info("Update missing references on table: " + tableName);
			}
			progressionModel.setMessage(t("adagio.persistence.synchronizeData.synchronize.step3", tableName));

			updateMissingUpdates(table, synchroOperation.getMissingUpdates(), targetConnection, context, result, pendingOperations);

			TIME.log(t0, "update missing reference on table " + tableName);

			// Clean operation
			synchroOperation.getMissingUpdates().clear();
		}

		// If operation not empty, add it to pendings
		if (!synchroOperation.isEmpty()) {
			pendingOperations.add(synchroOperation);
		}

		return pendingOperations;
	}

	protected void prepareRootTable(
			SynchroTableMetadata sourceTable,
			SynchroTableMetadata targetTable,
			Connection sourceConnection,
			Connection targetConnection,
			SynchroResult result) throws SQLException {

		String tablePrefix = sourceTable.getTableLogPrefix();
		String tableName = sourceTable.getName();

		SynchroTableDao targetDao = new SynchroTableDaoImpl(targetTable.getDatabaseMetadata().getDialect(),
				targetConnection, targetTable, false);
		SynchroTableDao sourceDao = new SynchroTableDaoImpl(sourceTable.getDatabaseMetadata().getDialect(),
				sourceConnection, sourceTable, false);

		try {
			// get last updateDate used by target db
			Timestamp updateDate = targetDao.getLastUpdateDate();

			if (updateDate != null) {

				// just inscrements of 1 milisecond to not having same
				updateDate = new Timestamp(DateUtils.setMilliseconds(updateDate, 0).getTime());
				updateDate = new Timestamp(DateUtils.addSeconds(updateDate, 1).getTime());
			}

			long countToUpdate = sourceDao.countDataToUpdate(updateDate);

			if (log.isInfoEnabled()) {
				log.info(String.format("%s nb rows to update: %s", tablePrefix, countToUpdate));
			}

			result.setUpdateDate(tableName, updateDate);
			result.addRows(tableName, (int) countToUpdate);
		} finally {
			IOUtils.closeQuietly(targetDao);
			IOUtils.closeQuietly(sourceDao);
		}
	}

	protected void synchronizeRootTable(
			SynchroTableMetadata table,
			Connection sourceConnection,
			Connection targetConnection,
			SynchroContext context,
			SynchroResult result,
			List<SynchroTableOperationBuffer> pendingOperations) throws SQLException {

		String tableName = table.getName();

		result.getProgressionModel().setMessage(t("adagio.persistence.synchronizeData.synchronizeTable", tableName));

		SynchroTableDao sourceDao = new SynchroTableDaoImpl(getSourceDialect(context), sourceConnection, table, false);
		SynchroTableDao targetDao = new SynchroTableDaoImpl(getTargetDialect(context), targetConnection, table, true);

		// get last updateDate used by target db
		Date updateDate = result.getUpdateDate(tableName);

		ResultSet dataToUpdate = null;
		try {
			// get data to update from source db
			dataToUpdate = sourceDao.getDataToUpdate(updateDate);

			updateTableUsingRemoteId(
					targetDao,
					dataToUpdate,
					result,
					pendingOperations);
		} finally {
			DaoUtils.closeSilently(dataToUpdate);

			IOUtils.closeQuietly(targetDao);
			IOUtils.closeQuietly(sourceDao);
		}

	}

	protected void synchronizeChildTable(
			SynchroTableMetadata table,
			String columnName,
			List<Object> columnValues,
			Connection sourceConnection,
			Connection targetConnection,
			SynchroContext context,
			SynchroResult result,
			List<SynchroTableOperationBuffer> pendingOperations) throws SQLException {

		String tableName = table.getName();

		result.getProgressionModel().setMessage(t("adagio.persistence.synchronizeData.synchronizeTable", tableName));

		SynchroTableDao sourceDao = new SynchroTableDaoImpl(getDialect(context.getSourceConnectionProperties()), sourceConnection, table, false);
		SynchroTableDao targetDao = new SynchroTableDaoImpl(getDialect(context.getTargetConnectionProperties()), targetConnection, table, true);

		SynchroTableOperationBuffer pendingOperation = new SynchroTableOperationBuffer(tableName);
		targetDao.setPendingOperationBuffer(pendingOperation);

		try {
			ResultSet dataToUpdate = sourceDao.getDataByColumn(columnName, columnValues);

			try {
				// Table with a REMOTE_ID (and ID)
				if (table.isWithRemoteIdColumn()) {

					// small table update strategy
					updateTableUsingRemoteId(
							targetDao,
							dataToUpdate,
							result,
							pendingOperations);
				}

				// Association tables, ...
				else {
					updateTableNoRemoteId(
							targetDao,
							dataToUpdate,
							result,
							pendingOperations);
				}

				if (!pendingOperation.isEmpty()) {
					pendingOperations.add(pendingOperation);
				}

			} finally {
				DaoUtils.closeSilently(dataToUpdate);
			}

		} finally {
			IOUtils.closeQuietly(targetDao);
			IOUtils.closeQuietly(sourceDao);
		}
	}

	protected void synchronizeChildTable(
			SynchroTableMetadata table,
			Set<String> columnNames,
			Set<List<Object>> columnValues,
			Connection sourceConnection,
			Connection targetConnection,
			SynchroContext context,
			SynchroResult result,
			List<SynchroTableOperationBuffer> pendingOperations) throws SQLException {

		String tableName = table.getName();

		result.getProgressionModel().setMessage(t("adagio.persistence.synchronizeData.synchronizeTable", tableName));

		SynchroTableDao sourceDao = new SynchroTableDaoImpl(getSourceDialect(context), sourceConnection, table, false);
		SynchroTableDao targetDao = new SynchroTableDaoImpl(getTargetDialect(context), targetConnection, table, true);

		SynchroTableOperationBuffer pendingOperation = new SynchroTableOperationBuffer(tableName);
		targetDao.setPendingOperationBuffer(pendingOperation);

		try {
			ResultSet dataToUpdate = sourceDao.getDataByColumns(columnNames, columnValues);

			try {
				// Table with a REMOTE_ID (and ID)
				if (table.isWithRemoteIdColumn()) {

					// small table update strategy
					updateTableUsingRemoteId(
							targetDao,
							dataToUpdate,
							result,
							pendingOperations);
				}

				// Association tables, ...
				else {
					updateTableNoRemoteId(
							targetDao,
							dataToUpdate,
							result,
							pendingOperations);
				}

				if (!pendingOperation.isEmpty()) {
					pendingOperations.add(pendingOperation);
				}

			} finally {
				DaoUtils.closeSilently(dataToUpdate);
			}

		} finally {
			IOUtils.closeQuietly(targetDao);
			IOUtils.closeQuietly(sourceDao);
		}
	}

	protected void updateMissingUpdates(
			SynchroTableMetadata table,
			Map<String, Map<String, Object>> missingUpdates,
			Connection targetConnection,
			SynchroContext context,
			SynchroResult result,
			List<SynchroTableOperationBuffer> pendingOperations
			) throws SQLException {

		String tableName = table.getName();
		String tablePrefix = table.getTableLogPrefix();

		result.getProgressionModel().setMessage(t("adagio.persistence.synchronizeData.updateMissingReference", tableName));

		SynchroTableDao targetDao = new SynchroTableDaoImpl(getTargetDialect(context), targetConnection, table, true);

		try {
			for (String columnName : missingUpdates.keySet()) {
				Map<String, Object> valuesByPkStr = missingUpdates.get(columnName);
				targetDao.executeColumnUpdates(columnName, valuesByPkStr);
			}
		} finally {
			IOUtils.closeQuietly(targetDao);
		}

		int updateCount = targetDao.getUpdateCount();

		if (log.isInfoEnabled()) {
			log.info(String.format("%s done: (updates: %s)", tablePrefix, updateCount));
		}
	}

	/**
	 * To update the content of the given {@code table} on the target db,
	 * from the given {@code incomingData} of the source db.
	 * <p/>
	 * The algorithm use remote_id : for each row of the {@code incomingData}, if exists on target table, then do an
	 * update, otherwise do a insert.
	 * <p/>
	 * 
	 * @param synchroContext
	 *            Synchronization context
	 * @param targetDao
	 *            connection on the target db
	 * @param incomingData
	 *            data to update from the source db
	 * @param result
	 *            where to store operation results
	 * @throws SQLException
	 *             if any sql errors
	 */
	protected void updateTableUsingRemoteId(
			SynchroTableDao targetDao,
			ResultSet incomingData,
			SynchroResult result,
			List<SynchroTableOperationBuffer> pendingOperations) throws SQLException {

		SynchroTableMetadata table = targetDao.getTable();
		Preconditions.checkArgument(table.isWithRemoteIdColumn());
		boolean enableGeneratedIdFirst = table.getInsertStrategy() == TableInsertStrategy.GENERATE_ID_FIRST;

		String tableName = table.getName();
		String tablePrefix = table.getTableLogPrefix() + " - " + result.getNbRows(tableName);

		// get existing ids in the target db
		Map<Integer, Integer> existingRemoteIdsMap = targetDao.getExistingRemoteIdsMap();
		if (log.isDebugEnabled()) {
			log.debug(tablePrefix + " existing rows: " + existingRemoteIdsMap.size());
		}

		result.addTableName(tableName);

		int countR = 0;

		boolean hasChildTables = table.hasChildJoins();
		List<Object> updatedRemoteIds = null;
		if (hasChildTables) {
			updatedRemoteIds = Lists.newArrayList();
		}

		while (incomingData.next()) {

			Integer remoteId = table.getId(incomingData);
			Integer localId = existingRemoteIdsMap.get(remoteId);
			boolean doUpdate = localId != null;

			if (doUpdate) {
				List<Object> pk = Lists.<Object> newArrayList(localId);
				targetDao.executeUpdate(pk, incomingData);

			} else {
				if (enableGeneratedIdFirst) {
					localId = targetDao.executeInsertAndReturnId(incomingData);
					if (hasChildTables) {
						updatedRemoteIds.add(remoteId);
					}
				}
				else {
					targetDao.executeInsert(incomingData);
				}
			}

			countR++;

			reportProgress(result, targetDao, countR, tablePrefix);
		}

		targetDao.flush();

		int insertCount = targetDao.getInsertCount();
		int updateCount = targetDao.getUpdateCount();

		result.addInserts(tableName, insertCount);
		result.addUpdates(tableName, updateCount);

		// Add child table to pending operation buffer
		if (hasChildTables && CollectionUtils.isNotEmpty(updatedRemoteIds)) {
			SynchroTableOperationBuffer pendingOperation = new SynchroTableOperationBuffer(tableName);

			for (SynchroJoinMetadata join : table.getChildJoins()) {
				SynchroTableMetadata childTable = join.getTargetTable();
				SynchroColumnMetadata childTableColumn = join.getTargetColumn();

				pendingOperation.addChildsToUpdate(childTable.getName(), childTableColumn.getName(), updatedRemoteIds);
			}
			pendingOperations.add(pendingOperation);
		}

		if (log.isInfoEnabled()) {
			log.info(String.format("%s done: %s (inserts: %s, updates: %s)", tablePrefix, insertCount + updateCount, insertCount, updateCount));
		}

		if (log.isDebugEnabled()) {
			log.debug(String.format("%s INSERT count: %s", tablePrefix, insertCount));
			log.debug(String.format("%s UPDATE count: %s", tablePrefix, updateCount));
		}

		result.getProgressionModel().increments(countR % 1000);
	}

	/**
	 * To update the content of the given {@code table} on the target db,
	 * from the given {@code incomingData} of the source db.
	 * The algorithm use a standard update strategy, using primary key.
	 * 
	 * @param synchroContext
	 * @param targetDao
	 * @param incomingData
	 * @param result
	 * @throws SQLException
	 */
	protected void updateTableNoRemoteId(
			SynchroTableDao targetDao,
			ResultSet incomingData,
			SynchroResult result,
			List<SynchroTableOperationBuffer> pendingOperations) throws SQLException {
		SynchroTableMetadata table = targetDao.getTable();
		Preconditions.checkArgument(!table.isWithRemoteIdColumn());

		String tableName = table.getName();
		String tablePrefix = table.getTableLogPrefix() + " - " + result.getNbRows(tableName);

		// get existing ids in the target db
		Set<String> existingIds = targetDao.getExistingPrimaryKeys();
		if (log.isDebugEnabled()) {
			log.debug(tablePrefix + " existing rows: " + existingIds.size());
		}

		result.addTableName(tableName);

		int countR = 0;

		boolean hasChildTables = table.hasChildJoins();
		List<List<Object>> updatedPks = null;
		if (hasChildTables) {
			updatedPks = Lists.newArrayList();
		}

		while (incomingData.next()) {
			List<Object> pk = targetDao.getPk(incomingData);
			String pkStr = table.toPkStr(pk);

			boolean doUpdate = existingIds.contains(pkStr);

			if (doUpdate) {

				targetDao.executeUpdate(pk, incomingData);

			} else {

				targetDao.executeInsert(incomingData);
			}

			if (hasChildTables) {
				updatedPks.add(pk);
			}

			countR++;

			reportProgress(result, targetDao, countR, tablePrefix);
		}

		targetDao.flush();

		// Put in context (to be used by child join tables)
		if (hasChildTables && !updatedPks.isEmpty()) {
			if (hasChildTables && CollectionUtils.isNotEmpty(updatedPks)) {
				for (SynchroJoinMetadata join : table.getChildJoins()) {
					SynchroTableMetadata childTable = join.getTargetTable();
					SynchroColumnMetadata childTableColumn = join.getTargetColumn();

					// synchroOperation.addChildToUpdate(childTable.getName(), childTableColumn.getName(), updatedPks);
					// pendingOperations.add(synchroOperation)
				}
			}
		}

		int insertCount = targetDao.getInsertCount();
		int updateCount = targetDao.getUpdateCount();

		result.addInserts(tableName, insertCount);
		result.addUpdates(tableName, updateCount);

		if (log.isInfoEnabled()) {
			log.info(String.format("%s done: %s (inserts: %s, updates: %s)", tablePrefix, insertCount + updateCount, insertCount, updateCount));
		}

		if (log.isDebugEnabled()) {
			log.debug(String.format("%s INSERT count: %s", tablePrefix, insertCount));
			log.debug(String.format("%s UPDATE count: %s", tablePrefix, updateCount));
		}

		result.getProgressionModel().increments(countR % 1000);
	}

	Connection createConnection(Properties connectionProperties) throws SQLException {
		return createConnection(
				connectionProperties.getProperty(Environment.URL),
				connectionProperties.getProperty(Environment.USER),
				connectionProperties.getProperty(Environment.PASS));
	}

	String getUrl(Properties connectionProperties) {
		return connectionProperties.getProperty(Environment.URL);
	}

	Configuration getConfiguration(Properties connectionProperties) {
		return new Configuration().setProperties(connectionProperties);
	}

	Connection createConnection(String jdbcUrl,
			String user,
			String password) throws SQLException {
		Preconditions.checkArgument(StringUtils.isNotBlank(jdbcUrl));

		// If same URL as datasource, use the dataSource
		if (jdbcUrl.equals(config.getJdbcURL()) && this.dataSource != null) {
			return DataSourceUtils.getConnection(this.dataSource);
		}

		Connection connection = DriverManager.getConnection(jdbcUrl,
				user,
				password);
		connection.setAutoCommit(false);
		return connection;
	}

	void releaseConnection(Connection connection) {
		DaoUtils.closeSilently(connection);
	}

	protected Properties getRemoteProperties(File dbDirectory) {
		Properties sourceConnectionProperties = new Properties();
		SynchroConfiguration config = SynchroConfiguration.getInstance();

		String jdbcUrl = DaoUtils.getJdbcUrl(dbDirectory,
				config.getDbName());

		DaoUtils.fillConnectionProperties(sourceConnectionProperties,
				jdbcUrl,
				config.getJdbcUsername(),
				config.getJdbcPassword());
		return sourceConnectionProperties;
	}
	//
	// protected void synchronizeChildTables(
	// SynchroTableMetadata parentTable,
	// SynchroPendingOperationBuffer parentBuffer,
	// Connection sourceConnection,
	// Connection targetConnection,
	// SynchroResult result,
	// List<SynchroPendingOperationBuffer> pendingOperationBuffers,
	// boolean enableLogCount) throws SQLException {
	//
	// Preconditions.checkNotNull(parentTable);
	//
	// List<SynchroTableMetadata> updatedTables = Lists.newArrayList();
	// List<SynchroPendingOperationBuffer> updatedTablesBuffers = Lists.newArrayList();
	//
	// for (SynchroJoinMetadata join : parentTable.getChildJoins()) {
	// long t0 = TimeLog.getTime();
	//
	// SynchroTableMetadata table = join.getTargetTable();
	// String tableName = table.getName();
	// if (log.isInfoEnabled()) {
	// log.info(String.format("Synchronize table: %s (as child of %s)", tableName, parentTable.getName()));
	// }
	//
	// SynchroPendingOperationBuffer pendingOperationBuffer = new SynchroPendingOperationBuffer(tableName);
	//
	// // Retrieve the table to update, from the join
	// String joinColumnName = join.getTargetColumn().getName();
	//
	// synchronizeChildTable(
	// table,
	// joinColumnName,
	// parentIds,
	// sourceConnection,
	// targetConnection,
	// result,
	// pendingOperationBuffer);
	//
	// TIME.log(t0, "synchronize table " + tableName);
	//
	// // Store updated remote ids, in order to update childs
	// boolean needToUpdateChild = MapUtils.isNotEmpty(pendingOperationBuffer.getRemoteIdsMap());
	// if (needToUpdateChild) {
	// updatedTables.add(table);
	// updatedTablesBuffers.add(pendingOperationBuffer);
	// }
	//
	// // Store buffer into the global list, if missing updates exists
	// boolean hasMissingRemoteIds = MapUtils.isNotEmpty(pendingOperationBuffer.getMissingRemoteIds());
	// if (hasMissingRemoteIds) {
	// pendingOperationBuffers.add(pendingOperationBuffer);
	// }
	// }
	//
	// // Recursive call, for each child of the processed child tables
	// for (int i = 0; i < updatedTables.size(); i++) {
	// SynchroTableMetadata table = updatedTables.get(i);
	// SynchroPendingOperationBuffer tableBuffer = updatedTablesBuffers.get(i);
	//
	// Set<Integer> updatedRemoteIds = tableBuffer.getRemoteIdsMap().keySet();
	// synchronizeChildTablesWithFk(table, updatedRemoteIds, context, sourceConnection, targetConnection, result,
	// pendingOperationBuffers, false);
	//
	// tableBuffer.getRemoteIdsMap().clear();
	// }
	//
	// }

	// protected void synchronizeChildTablesWithFk(
	// SynchroTableMetadata parentTable,
	// Set<Integer> parentIds,
	// SynchroContext context,
	// Connection sourceConnection,
	// Connection targetConnection,
	// SynchroResult result,
	// List<SynchroTableOperationBuffer> pendingOperations,
	// boolean enableLogCount) throws SQLException {
	//
	// Preconditions.checkNotNull(parentTable);
	// Preconditions.checkNotNull(parentIds);
	// Preconditions.checkArgument(!parentIds.isEmpty());
	//
	// List<SynchroTableMetadata> updatedTables = Lists.newArrayList();
	// List<SynchroPendingOperationBuffer> updatedTablesBuffers = Lists.newArrayList();
	//
	// for (SynchroJoinMetadata join : parentTable.getChildJoins()) {
	// long t0 = TimeLog.getTime();
	//
	// SynchroTableMetadata table = join.getTargetTable();
	// String tableName = table.getName();
	// if (log.isInfoEnabled()) {
	// log.info(String.format("Synchronize table: %s (as child of %s)", tableName, parentTable.getName()));
	// }
	//
	// SynchroPendingOperationBuffer pendingOperationBuffer = new SynchroPendingOperationBuffer(tableName);
	//
	// // Retrieve the table to update, from the join
	// String joinColumnName = join.getTargetColumn().getName();
	//
	// synchronizeChildTable(
	// table,
	// joinColumnName,
	// parentIds,
	// sourceConnection,
	// targetConnection,
	// result,
	// pendingOperationBuffer);
	//
	// TIME.log(t0, "synchronize table " + tableName);
	//
	// // Store updated remote ids, in order to update childs
	// boolean needToUpdateChild = MapUtils.isNotEmpty(pendingOperationBuffer.getRemoteIdsMap());
	// if (needToUpdateChild) {
	// updatedTables.add(table);
	// updatedTablesBuffers.add(pendingOperationBuffer);
	// }
	//
	// // Store buffer into the global list, if missing updates exists
	// boolean hasMissingRemoteIds = MapUtils.isNotEmpty(pendingOperationBuffer.getMissingRemoteIds());
	// if (hasMissingRemoteIds) {
	// pendingOperationBuffers.add(pendingOperationBuffer);
	// }
	// }
	//
	// // Recursive call, for each child of the processed child tables
	// for (int i = 0; i < updatedTables.size(); i++) {
	// SynchroTableMetadata table = updatedTables.get(i);
	// SynchroTableOperationBuffer tableBuffer = updatedTablesBuffers.get(i);
	//
	// Set<Integer> updatedRemoteIds = tableBuffer.getRemoteIdsMap().keySet();
	// synchronizeChildTablesWithFk(table, updatedRemoteIds, context, sourceConnection, targetConnection, result,
	// pendingOperationBuffers, false);
	//
	// tableBuffer.getRemoteIdsMap().clear();
	// }
	//
	// }

}
