package fr.ifremer.adagio.synchro.service;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.util.Map;
import java.util.Properties;
import java.util.Set;

import com.google.common.base.Predicate;
import com.google.common.collect.Maps;

import fr.ifremer.adagio.synchro.meta.SynchroMetadataUtils;

/**
 * A context class, need for referential and data synchronization
 * Created on 5/05/14.
 * 
 * @author Benoit Lavenier <benoit.lavenier@e-is.pro>
 * @since 3.5.3
 */
public class SynchroContext {

	protected Properties sourceConnectionProperties;

	protected Properties targetConnectionProperties;

	protected SynchroResult result;

	protected Predicate<String> tableFilter;

	protected Set<String> tableNames;

	protected Map<String, SynchroTableOperationBuffer> tableContextMap;

	/**
	 * Create a new synchro context
	 * 
	 * @param tableNames
	 *            Table set to includes. If null, all tables will be processed by table filter
	 * @param tableFilter
	 *            Use to filter tables to process. If null, given tables names will be processed. See
	 *            {@link SynchroMetadataUtils} for predicate examples with includes/excludes.
	 * @param sourceConnectionProperties
	 *            Connection properties for source database
	 * @param targetConnectionProperties
	 *            Connection properties for target database
	 * @param securityContext
	 *            A security context (use for data synchronization: see {@link SynchroInterceptor})
	 * @param result
	 *            Store the synchronization result
	 * @return a new synchro context
	 * @see SynchroInterceptor
	 */
	public static SynchroContext newContext(
			Set<String> tableNames,
			Predicate<String> tableFilter,
			Properties sourceConnectionProperties,
			Properties targetConnectionProperties,
			SynchroResult result) {
		SynchroContext context = new SynchroContext();
		context.setTableNames(tableNames);
		context.setTableFilter(tableFilter);
		context.setSourceConnectionProperties(sourceConnectionProperties);
		context.setTargetConnectionProperties(targetConnectionProperties);
		context.setResult(result);
		return context;
	}

	/**
	 * Create a new synchro context
	 * 
	 * @param tableNames
	 *            Table set to includes. If null, all tables will be retrieve.
	 * @param sourceConnectionProperties
	 *            Connection properties for source database
	 * @param targetConnectionProperties
	 *            Connection properties for target database
	 * @param securityContext
	 *            A security context (use for data synchronisation)
	 * @param result
	 *            Store the synchronization result
	 * @return a new synchro context
	 */
	public static SynchroContext newContext(
			Set<String> tableNames,
			Properties sourceConnectionProperties,
			Properties targetConnectionProperties,
			SynchroResult result) {

		return newContext(
				tableNames,
				null,
				sourceConnectionProperties,
				targetConnectionProperties,
				result);
	}

	public SynchroContext() {
		this.tableContextMap = Maps.newHashMap();
	}

	public SynchroResult getResult() {
		return result;
	}

	public void setResult(SynchroResult result) {
		this.result = result;
	}

	public void setSourceConnectionProperties(Properties sourceConnectionProperties) {
		this.sourceConnectionProperties = sourceConnectionProperties;
	}

	public void setTargetConnectionProperties(Properties targetConnectionProperties) {
		this.targetConnectionProperties = targetConnectionProperties;
	}

	public void setTableFilter(Predicate<String> tableFilter) {
		this.tableFilter = tableFilter;
	}

	public void setTableNames(Set<String> tableNames) {
		this.tableNames = tableNames;
	}

	public Properties getSourceConnectionProperties() {
		return sourceConnectionProperties;
	}

	public Properties getTargetConnectionProperties() {
		return targetConnectionProperties;
	}

	public Predicate<String> getTableFilter() {
		return tableFilter;
	}

	public Set<String> getTableNames() {
		return this.tableNames;
	}

	public SynchroTableOperationBuffer getTableContext(String tableName) {
		return tableContextMap.get(tableName);
	}

	public void addTableContext(String tableName, SynchroTableOperationBuffer tableContext) {
		tableContextMap.put(tableName, tableContext);
	}

	public void removeTableContext(String tableName) {
		tableContextMap.remove(tableName);
	}

}
