package fr.ifremer.adagio.synchro.intercept.internal;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import com.google.common.base.Preconditions;

import fr.ifremer.adagio.synchro.SynchroTechnicalException;
import fr.ifremer.adagio.synchro.config.SynchroConfiguration;
import fr.ifremer.adagio.synchro.dao.DaoUtils;
import fr.ifremer.adagio.synchro.dao.SynchroTableDao;
import fr.ifremer.adagio.synchro.intercept.SynchroInterceptor;
import fr.ifremer.adagio.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.adagio.synchro.intercept.SynchroWriteBuffer;
import fr.ifremer.adagio.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroJoinMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata;

public class RemoteIdWriteInterceptor extends SynchroInterceptorBase {

	private Set<String> dataTableIncludes;

	public RemoteIdWriteInterceptor() {
		super();
		dataTableIncludes = SynchroConfiguration.getInstance().getImportDataTablesIncludes();
	}

	public RemoteIdWriteInterceptor(SynchroInterceptor next) {
		super(next);
		dataTableIncludes = SynchroConfiguration.getInstance().getImportDataTablesIncludes();
	}

	@Override
	public boolean apply(SynchroDatabaseMetadata meta, TableMetadata table) {
		if (CollectionUtils.isEmpty(dataTableIncludes)) {
			return false;
		}

		boolean isDataTable = dataTableIncludes.contains(table.getName());

		return isDataTable;
	}

	@Override
	protected void doJoinLoad(SynchroTableMetadata table, SynchroJoinMetadata join) {
		if (join.needRemoteIdInterceptor()) {
			String pkTableName = join.getTargetTable().getName().toLowerCase();
			String fkColumnName = join.getSourceColumn().getName().toLowerCase();
			int fkColumnIndex = join.getSourceTable().getColumnIndex(fkColumnName);

			WriteInterceptor remoteIdInterceptor = new WriteInterceptor(
					pkTableName,
					fkColumnName,
					fkColumnIndex,
					join.getSourceColumn().isNullable());

			table.getInterceptors().add(remoteIdInterceptor);
		}
	}

	private class WriteInterceptor extends SynchroInterceptorBase {

		private final String tableName;

		private final String columnName;

		private final int columnIndex;

		private final boolean isNullable;

		private final String selectIdFromRemoteIdQuery;

		private PreparedStatement selectIdFromRemoteIdStatement = null;

		public WriteInterceptor(String tableName, String columnName, int columnIndex, boolean isNullable) {
			super();
			Preconditions.checkArgument(columnIndex >= 0);
			this.tableName = tableName;
			this.columnName = columnName;
			this.columnIndex = columnIndex;
			this.isNullable = isNullable;
			this.selectIdFromRemoteIdQuery = initSelectIdFromRemoteIdQuery(tableName);
		}

		@Override
		public SynchroInterceptorBase clone() {
			return new WriteInterceptor(
					tableName,
					columnName,
					columnIndex,
					isNullable);
		}

		@Override
		public boolean enableOnWrite() {
			return true;
		}

		protected void doOnWrite(Object[] data, String pkStr, SynchroTableDao dao, SynchroWriteBuffer buffer) throws SQLException {
			Integer remoteId = (Integer) data[columnIndex];
			if (remoteId == null) {
				return;
			}

			Integer localId = getIdFromRemoteId(dao.getConnection(), remoteId);
			if (localId == null) {
				// Mandatory column, could not continu
				if (!isNullable) {
					throw new SynchroTechnicalException(String.format("Could not retrieve ID corresponding to REMOTE_ID=%s, for table %s",
							remoteId, tableName));
				}

				// Optional column : add to pending changes (will be processed later)
				buffer.addMissingColumnUpdate(columnName, pkStr, remoteId);
				data[columnIndex] = null;
			}
			else {
				data[columnIndex] = localId;
			}
		}

		protected Integer getIdFromRemoteId(Connection connection, Integer remoteId) throws SQLException {
			if (selectIdFromRemoteIdStatement == null) {
				selectIdFromRemoteIdStatement = connection.prepareStatement(selectIdFromRemoteIdQuery);
			}
			selectIdFromRemoteIdStatement.setInt(1, remoteId);
			ResultSet resultSet = selectIdFromRemoteIdStatement.executeQuery();
			if (!resultSet.next()) {
				return null;
			}
			Integer result = resultSet.getInt(1);

			return result;
		}

		@Override
		protected void doClose() throws IOException {
			super.doClose();

			// Close statement
			DaoUtils.closeSilently(selectIdFromRemoteIdStatement);
			selectIdFromRemoteIdStatement = null;
		}

		protected String initSelectIdFromRemoteIdQuery(String tableName) {
			return String.format("SELECT %s FROM %s where %s=?",
					SynchroTableMetadata.COLUMN_ID,
					tableName,
					SynchroTableMetadata.COLUMN_REMOTE_ID
					);
		}

	}

}
