package fr.ifremer.adagio.synchro.intercept.internal;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.sql.SQLException;
import java.sql.Types;
import java.util.List;
import java.util.Map;

import oracle.sql.TIMESTAMP;

import org.hibernate.dialect.Dialect;
import org.hibernate.tool.hbm2ddl.ColumnMetadata;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import fr.ifremer.adagio.synchro.dao.SynchroTableDao;
import fr.ifremer.adagio.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.adagio.synchro.intercept.SynchroWriteBuffer;
import fr.ifremer.adagio.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata;
import fr.ifremer.adagio.synchro.service.SynchroContext;

/**
 * Workaround need for Oracle timestamp serialization
 * If ojdbc not found in classpath, do not try to cast to TIMESTAMP
 * 
 * @author Benoit Lavenier <benoit.lavenier@e-is.pro>
 * @since
 * 
 */
public class OracleInterceptor extends SynchroInterceptorBase {

	private Map<TableMetadata, int[]> columnIndexesByTable = Maps.newHashMap();

	private SynchroTableDao lastDao = null;

	private int[] lastIndexes = null;

	@Override
	public boolean apply(SynchroDatabaseMetadata meta, TableMetadata table) {
		boolean isOracleDialect = meta.getDialect().getClass().getName().contains("Oracle");
		if (!isOracleDialect) {
			return false;
		}

		Map<String, ColumnMetadata> columns = SynchroTableMetadata.getColumns(table);
		List<Integer> columnIndexesList = Lists.newArrayList();
		int index = 0;
		for (ColumnMetadata column : columns.values()) {
			if (column.getTypeCode() == Types.DATE
					|| column.getTypeCode() == Types.TIMESTAMP) {
				columnIndexesList.add(index);
			}
			index++;
		}

		if (columnIndexesList.isEmpty()) {
			return false;
		}

		int[] columnIndexes = getColumnIndexes(table);
		if (columnIndexes == null) {
			return false;
		}

		columnIndexesByTable.put(table, columnIndexes);

		return true;
	}

	@Override
	public boolean enableOnRead() {
		return true;
	}

	@Override
	protected String doCreateSelectQuery(SynchroTableMetadata table, String queryName, String sql) {
		return sql;
	}

	@Override
	protected void doOnRead(Object[] data, SynchroTableDao dao) throws SQLException {
		if (dao != lastDao) {
			lastIndexes = getColumnMapIndexes(dao);
			lastDao = dao;
		}
		for (int index : lastIndexes) {
			Object value = data[index];
			if (value != null && value instanceof TIMESTAMP) {
				data[index] = ((TIMESTAMP) value).timestampValue();
			}
		}
	}

	@Override
	public boolean enableOnWrite() {
		return true;
	}

	@Override
	protected void doOnWrite(Object[] data, String pkStr, SynchroTableDao dao, SynchroWriteBuffer buffer) throws SQLException {
		if (dao != lastDao) {
			lastIndexes = getColumnMapIndexes(dao);
			lastDao = dao;
		}
		for (int index : lastIndexes) {
			Object value = data[index];
			if (value != null && value instanceof TIMESTAMP) {
				data[index] = ((TIMESTAMP) value).timestampValue();
			}
		}
	}

	protected int[] getColumnMapIndexes(SynchroTableDao dao) {
		TableMetadata table = dao.getTable().getDelegate();
		int[] columnIndexes = columnIndexesByTable.get(table);
		if (columnIndexes == null) {
			columnIndexes = getColumnIndexes(table);
			columnIndexesByTable.put(table, columnIndexes);
		}
		return columnIndexes;
	}

	protected int[] getColumnIndexes(TableMetadata table) {
		Map<String, ColumnMetadata> columns = SynchroTableMetadata.getColumns(table);
		if (columns == null) {
			return null;
		}

		List<Integer> columnIndexesList = Lists.newArrayList();
		int index = 0;
		for (ColumnMetadata column : columns.values()) {
			if (column.getTypeCode() == Types.DATE
					|| column.getTypeCode() == Types.TIMESTAMP) {
				columnIndexesList.add(index);
			}
			index++;
		}

		if (columnIndexesList.isEmpty()) {
			return null;
		}

		int[] columnIndexes = new int[columnIndexesList.size()];
		index = 0;
		for (Integer columnIndex : columnIndexesList) {
			columnIndexes[index] = columnIndex;
		}

		return columnIndexes;
	}
}
