package fr.ifremer.adagio.synchro.intercept;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.util.Collection;
import java.util.List;
import java.util.ServiceLoader;

import org.hibernate.tool.hbm2ddl.TableMetadata;

import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.google.common.collect.Lists;

import fr.ifremer.adagio.synchro.SynchroTechnicalException;
import fr.ifremer.adagio.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.adagio.synchro.service.SynchroContext;

public class SynchroInterceptorUtils {

	private SynchroInterceptorUtils() {
		// helper class
	}

	public static <T, U extends SynchroInterceptor> List<U> load(Class<U> clazz, SynchroContext context) {
		List<U> result = Lists.newArrayList();

		ServiceLoader<U> loader = ServiceLoader.load(clazz);
		for (U interceptor : loader) {
			// Set the current context
			interceptor.setContext(context);

			result.add(interceptor);
		}
		return result;
	}

	public static <T, U extends SynchroInterceptor> Collection<U> filter(List<U> interceptors, final SynchroDatabaseMetadata meta,
			final TableMetadata table) {
		// Retrieve overrides to apply on this table
		Collection<U> result = Collections2.filter(
				interceptors,
				new Predicate<SynchroInterceptor>() {
					public boolean apply(SynchroInterceptor interceptor) {
						return interceptor.apply(meta, table);
					}
				});
		return result;
	}

	public static <T, U extends SynchroInterceptor> U chain(List<U> interceptors,
			final SynchroDatabaseMetadata meta,
			final TableMetadata table,
			Class<? extends SynchroInterceptorChain> chainClazz) {
		Collection<U> filteredInterceptors = filter(interceptors, meta, table);

		try {
			return (U) createChain(filteredInterceptors, chainClazz);
		} catch (InstantiationException e) {
			throw new SynchroTechnicalException("Could not instantiate interceptor : " + e.getMessage(), e);
		} catch (IllegalAccessException e) {
			throw new SynchroTechnicalException("Could not instantiate interceptor : " + e.getMessage(), e);
		}
	}

	public static <T, U extends SynchroInterceptor, V extends SynchroInterceptorChain> U chain(Collection<U> interceptors,
			Class<V> chainClazz) {
		try {
			return (U) createChain(interceptors, chainClazz);
		} catch (InstantiationException e) {
			throw new SynchroTechnicalException("Could not instantiate interceptor : " + e.getMessage(), e);
		} catch (IllegalAccessException e) {
			throw new SynchroTechnicalException("Could not instantiate interceptor : " + e.getMessage(), e);
		}
	}

	protected static <T, U extends SynchroInterceptor> SynchroInterceptorChain createChain(Collection<U> interceptors,
			Class<? extends SynchroInterceptorChain> chainClazz) throws InstantiationException, IllegalAccessException {
		SynchroInterceptorChain result = null;
		SynchroInterceptorChain previous = null;
		for (U interceptor : interceptors) {
			SynchroInterceptorChain newChain = null;
			if (interceptor instanceof SynchroInterceptorChain) {
				newChain = (SynchroInterceptorChain) interceptor;
			}
			else {
				newChain = encapsulate(interceptor, chainClazz);
			}
			if (result == null) {
				result = newChain;
			}
			else if (previous != null) {
				previous.setNext((U) newChain);
			}
			previous = newChain;
		}
		return result;
	}

	protected static <T, U extends SynchroInterceptor> SynchroInterceptorChain encapsulate(U delegate,
			Class<? extends SynchroInterceptorChain> chainClass) throws InstantiationException, IllegalAccessException {
		SynchroInterceptorChain chain = (SynchroInterceptorChain) chainClass.newInstance();
		chain.setNext(delegate);
		return chain;
	}
}
