package fr.ifremer.adagio.synchro.dao;

/*
 * #%L
 * Tutti :: Persistence API
 * $Id: TuttiEntities.java 1578 2014-02-07 15:31:18Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/trunk/tutti-persistence/src/main/java/fr/ifremer/tutti/persistence/entities/TuttiEntities.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.io.File;
import java.math.BigDecimal;
import java.math.MathContext;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.Properties;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.JDBCException;
import org.hibernate.Session;
import org.hibernate.cfg.AvailableSettings;
import org.hibernate.cfg.Configuration;
import org.hibernate.cfg.Environment;
import org.hibernate.dialect.Dialect;
import org.hibernate.exception.GenericJDBCException;
import org.hibernate.exception.spi.SQLExceptionConversionDelegate;
import org.hibernate.exception.spi.SQLExceptionConverter;

import com.google.common.base.Preconditions;

import fr.ifremer.adagio.synchro.SynchroTechnicalException;

/**
 * Usefull method around DAO and entities.
 * 
 * @author Benoit Lavenier <benoit.lavenier@e-is.pro>
 * @since 3.5.4
 */
public class DaoUtils {
	/** Logger. */
	private static final Log log = LogFactory.getLog(DaoUtils.class);

	private final static String JDBC_URL_PREFIX_HSQLDB = "jdbc:hsqldb:file:";

	protected DaoUtils() {
		// helper class does not instantiate
	}

	/**
	 * Create a new hibernate configuration, with all hbm.xml files
	 * for the schema need for app
	 * 
	 * @return the hibernate Configuration
	 */
	public static Properties getConnectionProperties(String jdbcUrl, String username, String password, String schema, String dialect, String driver) {

		// Building a new configuration
		Properties p = new Properties();

		// Set driver
		p.setProperty(Environment.DRIVER, driver);

		// Set hibernate dialect
		p.setProperty(Environment.DIALECT, dialect);

		// To be able to retrieve connection
		p.setProperty(Environment.URL, jdbcUrl);
		p.setProperty(Environment.USER, username);
		p.setProperty(Environment.PASS, password);

		if (StringUtils.isNotBlank(schema)) {
			p.setProperty(Environment.DEFAULT_SCHEMA, schema);
		}

		// Try with synonyms enable
		p.setProperty(AvailableSettings.ENABLE_SYNONYMS, "true");

		// Pour tester avec le metadata generic (normalement plus long pour Oracle)
		// cfg.setProperty("hibernatetool.metadatadialect", "org.hibernate.cfg.rveng.dialect.JDBCMetaDataDialect");
		if (jdbcUrl.startsWith("jdbc:oracle")) {
			p.setProperty("hibernatetool.metadatadialect", "org.hibernate.cfg.rveng.dialect.OracleMetaDataDialect");
		}

		return p;
	}

	/**
	 * Override default Hibernate 'org.hibernate.tool.hbm2ddl.DatabaseMetadata', because of th use of deprecated method
	 * buildSqlExceptionConverter()
	 * (see https://hibernate.atlassian.net/browse/HHH-9131)
	 * 
	 * @return
	 */
	public static SQLExceptionConverter newSQLExceptionConverter(final Dialect dialect) {
		// Build a valid sql converter
		return new SQLExceptionConverter() {
			private static final long serialVersionUID = 5458961195167573495L;

			SQLExceptionConversionDelegate delegate = dialect.buildSQLExceptionConversionDelegate();;

			@Override
			public JDBCException convert(SQLException sqlException, String message, String sql) {
				JDBCException exception = delegate.convert(sqlException, message, sql);
				if (exception != null) {
					return exception;
				}
				return new GenericJDBCException(message, sqlException, sql);
			}
		};
	}

	public static void shutdownDatabase(Properties connectionProperties) throws SQLException {
		Connection conn = DaoUtils.createConnection(connectionProperties);
		try {
			shutdownDatabase(conn);
		} finally {
			closeSilently(conn);
		}
	}

	public static void closeSilently(Statement statement) {
		try {
			if (statement != null && !statement.isClosed()) {
				statement.close();
			}
		} catch (AbstractMethodError e) {
			try {
				statement.close();
			} catch (SQLException e1) {
			}
			if (log.isDebugEnabled()) {
				log.debug("Fix this linkage error, damned hsqlsb 1.8.0.7:(");
			}
		} catch (IllegalAccessError e) {
			if (log.isDebugEnabled()) {
				log.debug("Fix this IllegalAccessError error, damned hsqlsb 1.8.0.7:(");
			}
		} catch (Exception e) {
			if (log.isErrorEnabled()) {
				log.error("Could not close statement, but do not care", e);
			}
		}
	}

	public static void closeSilently(Connection connection) {
		try {
			if (connection != null && !connection.isClosed()) {
				connection.close();
			}
		} catch (Exception e) {
			try {
				connection.close();
			} catch (SQLException e1) {
			}
			if (log.isErrorEnabled()) {
				log.error("Could not close connection, but do not care", e);
			}
		}
	}

	public static void closeSilently(ResultSet statement) {
		try {
			if (statement != null && !statement.isClosed()) {

				statement.close();
			}
		} catch (AbstractMethodError e) {
			try {
				statement.close();
			} catch (SQLException e1) {
			}
			if (log.isDebugEnabled()) {
				log.debug("Fix this linkage error, damned hsqlsb 1.8.0.7:(");
			}
		} catch (IllegalAccessError e) {
			if (log.isDebugEnabled()) {
				log.debug("Fix this IllegalAccessError error, damned hsqlsb 1.8.0.7:(");
			}
		} catch (Exception e) {
			if (log.isErrorEnabled()) {
				log.error("Could not close statement, but do not care", e);
			}
		}
	}

	public static void closeSilently(Session session) {
		try {
			if (session != null && session.isOpen()) {

				session.close();
			}
		} catch (Exception e) {
			if (log.isErrorEnabled()) {
				log.error("Could not close session, but do not care", e);
			}
		}
	}

	public static Connection createConnection(Properties connectionProperties) throws SQLException {
		return createConnection(
				connectionProperties.getProperty(Environment.URL),
				connectionProperties.getProperty(Environment.USER),
				connectionProperties.getProperty(Environment.PASS));
	}

	public static Connection createConnection(String jdbcUrl,
			String user,
			String password) throws SQLException {
		Connection connection = DriverManager.getConnection(jdbcUrl,
				user,
				password);
		connection.setAutoCommit(false);
		return connection;
	}

	public static void fillConnectionProperties(Properties p,
			String url,
			String username,
			String password) {
		p.put(Environment.URL, url);
		p.put(Environment.USER, username);
		p.put(Environment.PASS, password);
	}

	public static String getJdbcUrl(File directory, String dbName) {
		String jdbcUrl = JDBC_URL_PREFIX_HSQLDB + directory.getAbsolutePath() + "/" + dbName;
		jdbcUrl = jdbcUrl.replaceAll("\\\\", "/");
		return jdbcUrl;
	}

	public static boolean isFileDatabase(String jdbcUrl) {
		Preconditions.checkNotNull(jdbcUrl);
		return jdbcUrl.startsWith(JDBC_URL_PREFIX_HSQLDB);
	}

	/**
	 * Check if connection properties are valid. Try to open a SQL connection, then close it.
	 * If no error occur, the connection is valid.
	 * 
	 * @param jdbcDriver
	 * @param jdbcUrl
	 * @param user
	 * @param password
	 * @return
	 */
	public static boolean isValidConnectionProperties(Properties connectionProperties) {
		return isValidConnectionProperties(
				connectionProperties.getProperty(Environment.DRIVER),
				connectionProperties.getProperty(Environment.URL),
				connectionProperties.getProperty(Environment.USER),
				connectionProperties.getProperty(Environment.PASS));
	}

	/**
	 * Check if connection properties are valid. Try to open a SQL connection, then close it.
	 * If no error occur, the connection is valid.
	 * 
	 * @param jdbcDriver
	 * @param jdbcUrl
	 * @param user
	 * @param password
	 * @return
	 */
	public static boolean isValidConnectionProperties(
			String jdbcDriver,
			String jdbcUrl,
			String user,
			String password) {
		String driverClassName = jdbcDriver;
		try {
			Class<?> driverClass = Class.forName(driverClassName);
			DriverManager.registerDriver((java.sql.Driver) driverClass.newInstance());
		} catch (Exception e) {
			log.error("Could not load JDBC Driver: " + e.getMessage(), e);
			return false;
		}

		Connection connection = null;
		try {
			connection = createConnection(
					jdbcUrl,
					user,
					password);
			return true;
		} catch (SQLException e) {
			log.error("Could not connect to database: " + e.getMessage().trim());
		} finally {
			DaoUtils.closeSilently(connection);
		}
		return false;
	}

	public static String getUrl(Properties connectionProperties) {
		return connectionProperties.getProperty(Environment.URL);
	}

	public static Dialect getDialect(Properties connectionProperties) {
		return Dialect.getDialect(connectionProperties);
	}

	public static Configuration getConfiguration(Properties connectionProperties) {
		return new Configuration().setProperties(connectionProperties);
	}

	public static void shutdownDatabase(Connection connection) {
		try {
			String jdbcUrl = connection.getMetaData().getURL();
			if (jdbcUrl.startsWith(JDBC_URL_PREFIX_HSQLDB)) {
				sqlUpdate(connection, "SHUTDOWN");
			}
		} catch (SQLException e) {
			e.printStackTrace();
		}

	}

	public static int sqlUpdate(Connection connection, String sql) {
		Statement stmt = null;
		try {
			stmt = connection.createStatement();
		} catch (SQLException ex) {
			closeSilently(stmt);
			throw new SynchroTechnicalException("Could not open database connection", ex);
		}

		// Log using a special logger
		if (log.isDebugEnabled()) {
			log.debug(sql);
		}

		try {
			return stmt.executeUpdate(sql);
		} catch (SQLException ex) {
			throw new SynchroTechnicalException("Could not execute query: " + sql, ex);
		} finally {
			closeSilently(stmt);
		}
	}

}
