package fr.ifremer.adagio.synchro.action;

/*
 * #%L
 * SIH-Adagio :: Shared
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.sql.SQLException;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.I18n;

import fr.ifremer.adagio.synchro.SynchroTechnicalException;
import fr.ifremer.adagio.synchro.config.SynchroConfiguration;
import fr.ifremer.adagio.synchro.dao.DaoUtils;
import fr.ifremer.adagio.synchro.service.SynchroContext;
import fr.ifremer.adagio.synchro.service.SynchroResult;
import fr.ifremer.adagio.synchro.service.referential.ReferentialSynchroServiceImpl;

public class SynchroImportReferentialAction {
	/* Logger */
	private static final Log log = LogFactory.getLog(SynchroImportReferentialAction.class);

	public void run() {
		SynchroConfiguration config = SynchroConfiguration.getInstance();

		// Get connections properties :
		Properties sourceConnectionProperties = config.getImportConnectionProperties();
		Properties targetConnectionProperties = config.getConnectionProperties();

		// Log target connection
		if (log.isInfoEnabled()) {
			log.info("Starting database referential importation...");
		}

		// Check connections
		if (!checkConnection(config, sourceConnectionProperties, targetConnectionProperties)) {
			return;
		}

		// Create the result
		SynchroResult result = new SynchroResult();

		// Get tables to includes from configuration
		Set<String> tableToIncludes = config.getImportReferentialTablesIncludes();

		// Create the context
		SynchroContext context = SynchroContext.newContext(
				tableToIncludes,
				sourceConnectionProperties,
				targetConnectionProperties,
				result);

		// Prepare synchro
		ReferentialSynchroServiceImpl service = new ReferentialSynchroServiceImpl();
		service.prepare(context);

		if (!result.isSuccess()) {
			throw new SynchroTechnicalException(I18n.t("adagio.synchro.prepare.error"),
					result.getError());
		}

		// Run the synchro
		service.synchronize(context);

		if (!result.isSuccess()) {
			throw new SynchroTechnicalException(I18n.t("adagio.synchro.synchro.error"),
					result.getError());
		}

		// Shutdown database at end
		try {
			DaoUtils.shutdownDatabase(targetConnectionProperties);
		} catch (SQLException e) {
			throw new SynchroTechnicalException(I18n.t("adagio.synchro.shutdown.error"),
					result.getError());
		}
	}

	protected boolean checkConnection(
			SynchroConfiguration config,
			Properties sourceConnectionProperties,
			Properties targetConnectionProperties) {

		// Log target connection
		if (log.isInfoEnabled()) {
			log.info("Connecting to target database...");
			log.info(String.format(" Database directory: %s", config.getDbDirectory()));
			log.info(String.format(" JDBC Driver: %s", config.getJdbcDriver()));
			log.info(String.format(" JDBC URL: %s", config.getJdbcURL()));
			log.info(String.format(" JDBC Username: %s", config.getJdbcUsername()));
		}

		// Check target connection
		boolean isValidConnection = DaoUtils.isValidConnectionProperties(targetConnectionProperties);
		if (!isValidConnection) {
			log.error("Connection error: could not connect to target database.");
			return false;
		}

		// Log source connection
		if (log.isInfoEnabled()) {
			log.info("Connecting to source database...");
			log.info(String.format(" JDBC Driver: %s", config.getImportJdbcDriver()));
			log.info(String.format(" JDBC URL: %s", config.getImportJdbcURL()));
			log.info(String.format(" JDBC Username: %s", config.getImportJdbcUsername()));
		}

		// Check source connection
		String sourceJdbcUrl = DaoUtils.getUrl(sourceConnectionProperties);
		if (!sourceJdbcUrl.startsWith("jdbc:oracle:")) {
			// Source database = an oracle connection (see file synchro-test-write.properties)
			log.warn("Source database is not a Oracle database. Make sure your configuration file is correct.");
		}
		isValidConnection = DaoUtils.isValidConnectionProperties(sourceConnectionProperties);
		if (!isValidConnection) {
			log.warn("Connection error: could not connect to source database.");
			return false;
		}

		return true;
	}
}
