package fr.ifremer.adagio.core.test;

/*
 * #%L
 * SIH-Adagio Core for Allegro
 * $Id: DatabaseResource.java 607 2013-04-21 12:34:44Z tc1fbb1 $
 * $HeadURL: https://forge.ifremer.fr/svn/sih-adagio/trunk/adagio/core-allegro/src/test/java/fr/ifremer/adagio/core/DatabaseResource.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.List;
import java.util.Locale;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.junit.Assume;
import org.junit.rules.TestRule;
import org.junit.runner.Description;
import org.junit.runners.model.Statement;
import org.nuiton.i18n.I18n;
import org.nuiton.i18n.init.DefaultI18nInitializer;
import org.nuiton.i18n.init.UserI18nInitializer;

import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.io.Files;

import fr.ifremer.adagio.core.config.AdagioConfiguration;
import fr.ifremer.adagio.core.config.AdagioConfigurationOption;
import fr.ifremer.adagio.core.dao.technical.DaoUtils;
import fr.ifremer.adagio.core.service.ServiceLocator;

/**
 * To be able to manage database connection for unit test.
 * 
 * @author blavenie <benoit.lavenier@e-is.pro>
 * @since 3.3.3
 */
public abstract class DatabaseResource implements TestRule {

	/** Logger. */
	protected static final Log log = LogFactory.getLog(DatabaseResource.class);

	public static final String BUILD_ENVIRONMENT_DEFAULT = "hsqldb";
	public static final String HSQLDB_SRC_DATABASE_DIRECTORY = "src/test/db";
	

	public static long BUILD_TIMESTAMP = System.nanoTime();

	private File resourceDirectory;
	
	private String dbDirectory;

	protected final String beanFactoryReferenceLocation;

	protected final String beanRefFactoryReferenceId;
	
	protected final String hsqldbSrcDatabaseCreateScript;

	private final boolean writeDb;
	
	private String configName;
	
	private boolean witherror = false;

	protected Class<?> testClass;

	protected DatabaseResource(String configName, String beanFactoryReferenceLocation,
								String beanRefFactoryReferenceId,
								boolean writeDb) {
		this.configName = configName;
		this.beanFactoryReferenceLocation = beanFactoryReferenceLocation;
		this.beanRefFactoryReferenceId = beanRefFactoryReferenceId;
		this.writeDb = writeDb;
        hsqldbSrcDatabaseCreateScript = String.format("%s/%s.script", 
                HSQLDB_SRC_DATABASE_DIRECTORY,
                getTestDbName());
	}
	
	/**
	 * Return configuration files prefix (i.e. 'allegro-test')
	 * Could be override by external project
	 * @return the prefix to use to retrieve configuration files 
	 */
	protected abstract String getConfigFilesPrefix();

	public File getResourceDirectory(String name) {
		return new File(resourceDirectory, name);
	}

	public File getResourceDirectory() {
		return resourceDirectory;
	}
	
	protected boolean isWriteDb() {
		return writeDb;
	}

	@Override
	public Statement apply(final Statement base, final Description description) {

		return new Statement() {
			@Override
			public void evaluate() throws Throwable {
				before(description);
				try {
					base.evaluate();
				} catch (Throwable e) {
					witherror = true;
				} finally {
					after(description);
				}
			}
		};
	}

	protected void before(Description description) throws Throwable {
		testClass = description.getTestClass();

		boolean defaultDbName = StringUtils.isEmpty(configName);

		dbDirectory = null;
		if (defaultDbName) {
			configName = "db";
		}

		if (log.isInfoEnabled()) {
			log.info("Prepare test " + testClass);
		}

		resourceDirectory = getTestSpecificDirectory(testClass, "");
        addToDestroy(resourceDirectory);

		// Load building env
		String buildEnvironment = getBuildEnvironment();

		// check that config file is in classpath (avoid to find out why it does not works...)
		String configFilename = getConfigFilesPrefix();
		if (enableDb()) {
			configFilename += "-" + (writeDb ? "write" : "read");
		}
		if (!defaultDbName) {
			configFilename += "-" + configName;
		}
		String configFilenameNoEnv = configFilename + ".properties";
		if (StringUtils.isNotBlank(buildEnvironment)) {
			configFilename += "-" + buildEnvironment;
		}
		configFilename += ".properties";
		
		InputStream resourceAsStream = getClass().getResourceAsStream("/" + configFilename);
		if (resourceAsStream == null && StringUtils.isNotBlank(buildEnvironment)) {
			resourceAsStream = getClass().getResourceAsStream("/" + configFilenameNoEnv);
			Preconditions.checkNotNull(resourceAsStream, "Could not find " + configFilename + " or "+ configFilenameNoEnv +" in test class-path");
			configFilename = configFilenameNoEnv;
		}
		else {
			Preconditions.checkNotNull(resourceAsStream, "Could not find " + configFilename + " in test class-path");
		}

		// Prepare DB
		if (enableDb() && "hsqldb".equalsIgnoreCase(buildEnvironment)) {
			        
			dbDirectory = HSQLDB_SRC_DATABASE_DIRECTORY;
			if (!defaultDbName) {
				dbDirectory += configName;
			}
			TestUtil.checkDbExists(testClass, dbDirectory);

			if (writeDb) {
			    Properties p = new Properties();
			    p.load(resourceAsStream);
	            String jdbcUrl =  p.getProperty(AdagioConfigurationOption.JDBC_URL.getKey());
	            boolean serverMode = jdbcUrl != null && jdbcUrl.startsWith("jdbc:hsqldb:hsql://");
	                    
			    // If hsqld run on server mode 
			    if (serverMode) {
			        // Do not copy DB files, but display a warn
			        log.warn(String.format("Database running in server mode ! Please remove the property '%s' in file %s, to use a file database.", AdagioConfigurationOption.JDBC_URL.getKey(), configFilename));
			    }
			    else {			    
                    // Copy DB files into test directory
    				copyDb(new File(dbDirectory), "db", !writeDb, null);
    
    				// Update db directory with the new path
    				dbDirectory = new File(resourceDirectory, "db").getAbsolutePath();
    				dbDirectory = dbDirectory.replaceAll("[\\\\]", "/");
			    }
			} else {
				// Load db config properties
				File dbConfig = new File(dbDirectory, getTestDbName() + ".properties");
				Properties p = new Properties();
				BufferedReader reader = Files.newReader(dbConfig, Charsets.UTF_8);
				p.load(reader);
				reader.close();

				if (log.isDebugEnabled()) {
					log.debug("Db config: " + dbConfig + "\n" + p);
				}

				// make sure db is on readonly mode
				String readonly = p.getProperty("readonly");
				Preconditions.checkNotNull(readonly, "Could not find readonly property on db confg: " + dbConfig);
				Preconditions.checkState("true".equals(readonly), "readonly property must be at true value in read mode test in  db confg: "
						+ dbConfig);
			}
		}

		// Initialize configuration
		initConfiguration(configFilename);	
		
		// Init i18n
		initI18n();

		// Initialize spring context
		if (beanFactoryReferenceLocation != null) {
			ServiceLocator.instance().init(
					beanFactoryReferenceLocation,
					beanRefFactoryReferenceId);
		}
	}

	protected final Set<File> toDestroy = Sets.newHashSet();

	public void addToDestroy(File dir) {
		toDestroy.add(dir);
	}

	public void setProperty(File file, String key, String value) throws IOException {
		// Load old properties values
		Properties props = new Properties();
		BufferedReader reader = Files.newReader(file, Charsets.UTF_8);
		props.load(reader);
		reader.close();

		// Store new properties values
		props.setProperty(key, value);
		BufferedWriter writer = Files.newWriter(file, Charsets.UTF_8);
		props.store(writer, "");
		writer.close();
	}

	public void copyDb(File sourceDirectory, String targetDbDirectoryName, boolean readonly, Properties p) throws IOException {
		File targetDirectory = getResourceDirectory(targetDbDirectoryName);
		copyDb(sourceDirectory, targetDirectory, readonly, p, true);
	}
	
	public void copyDb(File sourceDirectory, File targetDirectory, boolean readonly, Properties p, boolean destroyAfterTest) throws IOException {
		if (!sourceDirectory.exists()) {

			if (log.isWarnEnabled()) {
				log.warn("Could not find db at " + sourceDirectory + ", test [" +
							testClass + "] is skipped.");
			}
			Assume.assumeTrue(false);
		}
		
		if (p != null) {
		    String jdbcUrl = DaoUtils.getJdbcUrl(targetDirectory, getTestDbName());
		    DaoUtils.fillConnectionProperties(p, jdbcUrl, "SA", "");
		}
		
		// Add to destroy files list
		if (destroyAfterTest) {
			addToDestroy(targetDirectory);
		}

		log.debug(String.format("Copy directory %s at %s", sourceDirectory.getPath(), targetDirectory.getPath()));
		FileUtils.copyDirectory(sourceDirectory, targetDirectory);

		// Set readonly property
		log.debug(String.format("Set database properties with readonly=%s", readonly));
		File dbConfig = new File(targetDirectory, getTestDbName() + ".properties");
		setProperty(dbConfig, "readonly", String.valueOf(readonly));
	}

	protected void after(Description description) throws IOException {
		if (log.isInfoEnabled()) {
			log.info("After test " + testClass);
		}
		
		ServiceLocator serviceLocator = ServiceLocator.instance();

		// If service and database has been started
		if (enableDb() && serviceLocator.isOpen()) {
		    Properties connectionProperties = AdagioConfiguration.getInstance().getConnectionProperties();
		    
	        // Shutdown if HSQLDB database is a file database (not server mode)
		    if (DaoUtils.isFileDatabase(DaoUtils.getUrl(connectionProperties))) {
    			try {
    	            DaoUtils.shutdownDatabase(connectionProperties);
    			} catch (Exception e) {
    				if (log.isErrorEnabled()) {
    					log.error("Could not close database.", e);
    				}
    				witherror = true;
    			}
		    }
    
    		// Shutdown spring context
			serviceLocator.shutdown();
		}

		if (!witherror) {
			for (File file : toDestroy) {
				if (file.exists()) {
					if (log.isInfoEnabled()) {
						log.info("Destroy directory: " + file);
					}
					try {
						FileUtils.deleteDirectory(file);
					} catch (IOException e) {
						if (log.isErrorEnabled()) {
							log.error("Could not delete directory: " + file + ". Please delete it manually.");
						}
					}
				}
			}
		}

		if (beanFactoryReferenceLocation != null) {

			// push back default configuration
			ServiceLocator.instance().init(null, null);
		}
	}

	public Connection createEmptyDb(String dbDirectory,
									String dbName) throws IOException, SQLException {
		return createEmptyDb(dbDirectory, dbName, null);
	}

	public Connection createEmptyDb(String dbDirectory,
									String dbName, Properties p) throws IOException, SQLException {
		File externalDbFile = getResourceDirectory(dbDirectory);
		return createEmptyDb(externalDbFile, dbName, p);
	}

	public static File getTestSpecificDirectory(Class<?> testClass,
												String name) throws IOException {
		// Trying to look for the temporary folder to store data for the test
		String tempDirPath = System.getProperty("java.io.tmpdir");
		if (tempDirPath == null) {
			// can this really occur ?
			tempDirPath = "";
			if (log.isWarnEnabled()) {
				log.warn("'\"java.io.tmpdir\" not defined");
			}
		}
		File tempDirFile = new File(tempDirPath);

		// create the directory to store database data
		String dataBasePath = testClass.getName()
								+ File.separator // a directory with the test class name
				+ name // a sub-directory with the method name
				+ '_'
								+ BUILD_TIMESTAMP; // and a timestamp
		File databaseFile = new File(tempDirFile, dataBasePath);
		FileUtils.forceMkdir(databaseFile);
		
		return databaseFile;
	}

	public Connection createEmptyDb(File directory,
									String dbName) throws SQLException, IOException {

		return createEmptyDb(directory, dbName, null);
	}

	public Connection createEmptyDb(File directory,
									String dbName, Properties p) throws SQLException, IOException {

		if (log.isInfoEnabled()) {
			log.info("Create new db at " + directory);
		}
		addToDestroy(directory);
		String jdbcUrl = DaoUtils.getJdbcUrl(directory, dbName);
		String user = "SA";
		String password = "";

		if (p != null) {
			DaoUtils.fillConnectionProperties(p, jdbcUrl, user, password);
		}
		File scriptFile = new File(hsqldbSrcDatabaseCreateScript);
		Preconditions.checkState(scriptFile.exists(), "Could not find db script at " + scriptFile);

		if (log.isInfoEnabled()) {
			log.info("Will use create script: " + scriptFile);
		}
		Connection connection = DaoUtils.createConnection(jdbcUrl, user, password);

		if (log.isInfoEnabled()) {
			log.info("Created connection at " + connection.getMetaData().getURL());
		}

		List<String> importScriptSql = getImportScriptSql(scriptFile);
		for (String sql : importScriptSql) {
			try {				
				PreparedStatement statement = connection.prepareStatement(sql);
				statement.execute();
			}
			catch(SQLException sqle) {
				log.warn("SQL command failed : " + sql, sqle);
				connection.close();
				throw sqle;
			}
		}
		connection.commit();
		return connection;
	}

	protected List<String> getImportScriptSql(File scriptFile) throws IOException {
		List<String> lines = Files.readLines(scriptFile, Charsets.UTF_8);

		List<String> result = Lists.newArrayListWithCapacity(lines.size());

		Predicate<String> predicate = new Predicate<String>() {

			Set<String> forbiddenStarts = Sets.newHashSet(
					"SET ",
					"CREATE USER ",
					"CREATE SCHEMA ",
					"GRANT DBA TO ");

			@Override
			public boolean apply(String input) {
				boolean accept = true;
				for (String forbiddenStart : forbiddenStarts) {
					if (input.startsWith(forbiddenStart)) {
						accept = false;
						break;
					}
				}
				return accept;
			}
		};
		for (String line : lines) {
			if (predicate.apply(line.trim().toUpperCase())) {
				if (line.contains("\\u000a")) {
					line = line.replaceAll("\\\\u000a", "\n");
				}
				result.add(line);
			}
		}
		return result;
	}
	
	public String getBuildEnvironment() {
		return getBuildEnvironment(null);
	}

	protected String getBuildEnvironment(String defaultEnvironement) {
		String buildEnv = System.getProperty("env");

		// Check validity
		if (buildEnv == null && StringUtils.isNotBlank(defaultEnvironement)) {
			buildEnv = defaultEnvironement;
			log.warn("Could not find build environment. Please add -Denv=<hsqldb|oracle|pgsql>. Test [" +
					testClass + "] will use default environment : " + defaultEnvironement);
		} else if ("hsqldb".equals(buildEnv) == false
				&& "oracle".equals(buildEnv) == false
				&& "pgsql".equals(buildEnv) == false) {

			if (log.isWarnEnabled()) {
				log.warn("Could not find build environment. Please add -Denv=<hsqldb|oracle|pgsql>. Test [" +
						testClass + "] will be skipped.");
			}
			Assume.assumeTrue(false);
		}
		return buildEnv;
	}	
	
	protected String[] getConfigArgs() {
		List<String> configArgs = Lists.newArrayList();
		configArgs.addAll(Lists.newArrayList(
				"--option", "adagio.basedir", resourceDirectory.getAbsolutePath()));
		if (dbDirectory != null) {
			configArgs.addAll(Lists.newArrayList("--option", "adagio.persistence.db.directory", dbDirectory));
		}
		return configArgs.toArray(new String[configArgs.size()]);
	}
	
	/**
	 * Convenience methods that could be override to initialize other configuration
	 * @param configFilename
	 * @param configArgs
	 */
	protected void initConfiguration(String configFilename) {
		String[] configArgs = getConfigArgs();
		AdagioConfiguration config = new AdagioConfiguration(configFilename, configArgs);
		AdagioConfiguration.setInstance(config);
	}
	
	protected void initI18n() throws IOException {
	    AdagioConfiguration config = AdagioConfiguration.getInstance();
        
        // --------------------------------------------------------------------//
        // init i18n
        // --------------------------------------------------------------------//
        File i18nDirectory = new File(config.getDataDirectory(), "i18n");
        if (i18nDirectory.exists()) {
            // clean i18n cache
            FileUtils.cleanDirectory(i18nDirectory);
        }

        FileUtils.forceMkdir(i18nDirectory);

        if (log.isDebugEnabled()) {
            log.debug("I18N directory: " + i18nDirectory);
        }

        Locale i18nLocale = config.getI18nLocale();

        if (log.isInfoEnabled()) {
            log.info(String.format("Starts i18n with locale [%s] at [%s]",
                    i18nLocale, i18nDirectory));
        }
        I18n.init(new UserI18nInitializer(
                i18nDirectory, new DefaultI18nInitializer(getI18nBundleName())),
                i18nLocale);
	}
	
	protected String getI18nBundleName() {
	    return "adagio-core-shared-i18n";
	}
	
	protected String getTestDbName() {
	    return "allegro";
	}
	
	/* -- Internal methods -- */
	
	private boolean enableDb() {
	    boolean prepareDb = beanFactoryReferenceLocation == null || !beanFactoryReferenceLocation.contains("WithNoDb");
	    return prepareDb;
	}
}
