package fr.ifremer.adagio.core.service.technical;

/*
 * #%L
 * SIH-Adagio :: Shared
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */


import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.ContextResource;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.springframework.core.io.UrlResource;
import org.springframework.util.Assert;
import org.springframework.util.ClassUtils;
import org.springframework.util.StringUtils;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;

public class SpringUtils {
	/** Logger. */
	private static final Log log = LogFactory.getLog(SpringUtils.class);

	
	protected SpringUtils() {
		// Helper class : do not instanciate
	}
	
	public static List<Resource> getResourcesFromPaths(String[] paths, ApplicationContext appContext, boolean checkIfResourceExists) throws IOException, FileNotFoundException {
		Preconditions.checkArgument(ArrayUtils.isNotEmpty(paths));
		Preconditions.checkNotNull(appContext);
		
		// For each path, retrieve corresponding resources
		List<Resource> resources = Lists.newArrayList();
		for (String path : paths) {
	        try {
	        	Resource[] pathResources = appContext.getResources(path);
		        resources.addAll(Arrays.asList(pathResources));
	        } catch (IOException e) {
	        	throw new IOException(String.format("Error while getting files from path: %s", path), e);
	        }
		}
		
        // Check if all resources exists
        if (checkIfResourceExists) {
	    	for(Resource resource : resources) {
	    		if (!resource.exists()) {
	    			throw new FileNotFoundException(String.format("File not found: %s", resource.getFilename()));
	    		}
	    	}
        }
		
        return resources;
	}
	

    public static Resource getResource(String location) {
        Assert.notNull(location, "Location must not be null");
        if (location.startsWith(ResourceLoader.CLASSPATH_URL_PREFIX)) {
            return new ClassPathResource(location.substring(ResourceLoader.CLASSPATH_URL_PREFIX.length()), getClassLoader());
        }
        else {
            try {
                // Try to parse the location as a URL...
                URL url = new URL(location);
                return new UrlResource(url);
            }
            catch (MalformedURLException ex) {
                // No URL -> resolve as resource path.
                return getResourceByPath(location);
            }
        }
    }
    
    protected static Resource getResourceByPath(String path) {
        return new ClassPathContextResource(path, getClassLoader());
    }
    
    protected static ClassLoader getClassLoader() {
        return ClassUtils.getDefaultClassLoader();
    }
    
    protected static class ClassPathContextResource extends ClassPathResource implements ContextResource {

        public ClassPathContextResource(String path, ClassLoader classLoader) {
            super(path, classLoader);
        }

        @Override
        public String getPathWithinContext() {
            return getPath();
        }

        @Override
        public Resource createRelative(String relativePath) {
            String pathToUse = StringUtils.applyRelativePath(getPath(), relativePath);
            return new ClassPathContextResource(pathToUse, getClassLoader());
        }
    }

}
