package fr.ifremer.adagio.core.dao.technical.liquibase;

/*
 * #%L
 * SIH-Adagio Core Shared
 * $Id: Liquibase.java 12765 2015-04-16 14:11:59Z bl05b3e $
 * $HeadURL: https://forge.ifremer.fr/svn/sih-adagio/tags/adagio-3.8.6.4/core-shared/src/main/java/fr/ifremer/adagio/core/dao/technical/liquibase/Liquibase.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.io.PrintStream;
import java.io.Writer;
import java.net.URL;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.Vector;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.sql.DataSource;
import javax.xml.parsers.ParserConfigurationException;

import liquibase.database.Database;
import liquibase.database.DatabaseFactory;
import liquibase.database.jvm.JdbcConnection;
import liquibase.diff.DiffResult;
import liquibase.diff.compare.CompareControl;
import liquibase.diff.output.DiffOutputControl;
import liquibase.diff.output.changelog.DiffToChangeLog;
import liquibase.diff.output.report.DiffToReport;
import liquibase.exception.DatabaseException;
import liquibase.exception.LiquibaseException;
import liquibase.integration.commandline.CommandLineUtils;
import liquibase.logging.Logger;
import liquibase.resource.ClassLoaderResourceAccessor;
import liquibase.resource.FileSystemResourceAccessor;
import liquibase.resource.ResourceAccessor;
import liquibase.structure.core.DatabaseObjectFactory;

import org.apache.commons.lang3.StringUtils;
import org.hibernate.cfg.Environment;
import org.nuiton.util.version.Version;
import org.nuiton.util.version.Versions;
import org.springframework.beans.BeanInstantiationException;
import org.springframework.beans.factory.BeanNameAware;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ResourceLoaderAware;
import org.springframework.context.annotation.Lazy;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;
import org.springframework.core.io.support.ResourcePatternUtils;
import org.springframework.jdbc.datasource.DataSourceUtils;
import org.springframework.stereotype.Component;
import org.springframework.util.ResourceUtils;

import fr.ifremer.adagio.core.AdagioTechnicalException;
import fr.ifremer.adagio.core.config.AdagioConfiguration;
import fr.ifremer.adagio.core.config.AdagioConfigurationOption;
import fr.ifremer.adagio.core.dao.technical.DaoUtils;

@Component("liquibase")
@Lazy
public class Liquibase implements InitializingBean, BeanNameAware, ResourceLoaderAware {

    protected final static String CHANGE_LOG_SNAPSHOT_SUFFIX = "-SNAPSHOT.xml";

    private String beanName;

    private ResourceLoader resourceLoader;

    private DataSource dataSource;

    private AdagioConfiguration config;

    private String changeLog;

    private String defaultSchema;

    private String contexts;

    private Map<String, String> parameters;

    protected Version maxChangeLogFileVersion;

    /**
     * Constructor used by Spring
     * @param dataSource
     * @param config
     */
    @Autowired
    public Liquibase(DataSource dataSource, AdagioConfiguration config) {
        this.dataSource = dataSource;
        this.config = config;
    }

    /**
     * Constructor used when Spring is not started (no datasource, and @Resource not initialized)
     * @param config
     */
    public Liquibase(AdagioConfiguration config) {
        this.dataSource = null;
        this.config = config;
        // Init change log
        setChangeLog(config.getLiquibaseChangeLogPath());
    }

    /**
     * Constructor used when Spring is not started (no datasource, and @Resource not initialized)
     * @param changeLogPath the path to the change log (e.g. the master changelog)
     */
    public Liquibase(String changeLogPath) {
        this.dataSource = null;
        this.config = AdagioConfiguration.getInstance();
        // Init change log
        setChangeLog(changeLogPath);
    }

    /**
     * Executed automatically when the bean is initialized.
     */
    @Override
    public void afterPropertiesSet() throws LiquibaseException {
        // Update the change log path, from configuration
        setChangeLog(config.getLiquibaseChangeLogPath());

        // Compute the max changelog file version
        computeMaxChangeLogFileVersion();

        boolean shouldRun = AdagioConfiguration.getInstance().useLiquibaseAutoRun();

        if (!shouldRun) {
            getLog().debug(
                    String.format("Liquibase did not run because properties '%s' set to false.",
                    AdagioConfigurationOption.LIQUIBASE_RUN_AUTO.getKey()));
            return;
        }

        executeUpdate();
    }

    public String getDatabaseProductName() throws DatabaseException {
        Connection connection = null;
        String name = "unknown";
        try {
            connection = createConnection();
            Database database =
                    DatabaseFactory.getInstance().findCorrectDatabaseImplementation(new JdbcConnection(dataSource.getConnection()));
            name = database.getDatabaseProductName();
        } catch (SQLException e) {
            throw new DatabaseException(e);
        } finally {
            if (connection != null) {
                try {
                    if (!connection.getAutoCommit()) {
                        connection.rollback();
                    }
                } catch (Exception e) {
                    getLog().warning("Problem rollback connection", e);
                }
                releaseConnection(connection);
            }
        }
        return name;
    }

    /**
     * @return The DataSource that liquibase will use to perform the migration.
     */
    public DataSource getDataSource() {
        return dataSource;
    }

    /**
     * The DataSource that liquibase will use to perform the migration.
     */
    public void setDataSource(DataSource dataSource) {
        this.dataSource = dataSource;
    }

    /**
     * @return a Resource that is able to resolve to a file or classpath resource.
     */
    public String getChangeLog() {
        return changeLog;
    }

    /**
     * Sets a Spring Resource that is able to resolve to a file or classpath resource.
     * An example might be <code>classpath:db-changelog.xml</code>.
     */
    public void setChangeLog(String dataModel) {

        this.changeLog = dataModel;
    }

    public String getContexts() {
        return contexts;
    }

    public void setContexts(String contexts) {
        this.contexts = contexts;
    }

    public String getDefaultSchema() {
        return defaultSchema;
    }

    public void setDefaultSchema(String defaultSchema) {
        this.defaultSchema = defaultSchema;
    }

    /**
     * Execute liquibase update, using change log
     * 
     * @throws LiquibaseException
     */
    public void executeUpdate() throws LiquibaseException {
        executeUpdate(null);
    }
    
    /**
     * Execute liquibase update, using change log
     * 
     * @param connectionProperties the properties for connection 
     * @throws LiquibaseException
     */
    public void executeUpdate(Properties connectionProperties) throws LiquibaseException {

        Connection c = null;
        liquibase.Liquibase liquibase;
        try {
            // open connection
            c = createConnection(connectionProperties);

            // create liquibase instance
            liquibase = createLiquibase(c);

            // First, release locks, then update and release locks again
            liquibase.forceReleaseLocks();
            performUpdate(liquibase);
            liquibase.forceReleaseLocks();

            // Compact database (if HsqlDB)            
            DaoUtils.compactDatabase(c);
            
        } catch (SQLException e) {
            throw new DatabaseException(e);
        } finally {
            if (c != null) {
                try {
                    c.rollback();
                } catch (SQLException e) {
                    // nothing to do
                }
                releaseConnection(c);
            }
        }

    }

    protected void performUpdate(liquibase.Liquibase liquibase) throws LiquibaseException {
        liquibase.update(getContexts());
    }

    /**
     * Execute liquibase status, using change log
     * 
     * @throws LiquibaseException
     */
    public void reportStatus(Writer writer) throws LiquibaseException {

        Connection c = null;
        liquibase.Liquibase liquibase;
        Writer myWriter = null;
        try {
            // open connection
            c = createConnection();

            // create liquibase instance
            liquibase = createLiquibase(c);

            // First, release locks, then update and release locks again
            liquibase.forceReleaseLocks();
            if (writer != null) {
                performReportStatus(liquibase, writer);
            }
            else {
                myWriter = new OutputStreamWriter(System.out);
                performReportStatus(liquibase, myWriter);
            }
            liquibase.forceReleaseLocks();

        } catch (SQLException e) {
            throw new DatabaseException(e);
        } finally {
            if (c != null) {
                try {
                    c.rollback();
                } catch (SQLException e) {
                    // nothing to do
                }
                releaseConnection(c);
            }
            if (myWriter != null) {
                try {
                    myWriter.close();
                } catch (IOException e) {
                    // nothing to do
                }
            }
        }

    }

    protected void performReportStatus(liquibase.Liquibase liquibase, Writer writer) throws LiquibaseException {
        liquibase.reportStatus(true, getContexts(), writer);
    }

    protected liquibase.Liquibase createLiquibase(Connection c) throws LiquibaseException {
        String adjustedChangeLog = getChangeLog();
        // If Spring started, no changes
        if (this.resourceLoader == null) {
            // Remove 'classpath:' and 'files:' prefixes
            adjustedChangeLog = adjustNoFilePrefix(adjustNoClasspath(adjustedChangeLog));
        }
        
        liquibase.Liquibase liquibase = new liquibase.Liquibase(adjustedChangeLog, createResourceAccessor(), createDatabase(c));
        if (parameters != null) {
            for (Map.Entry<String, String> entry : parameters.entrySet()) {
                liquibase.setChangeLogParameter(entry.getKey(), entry.getValue());
            }
        }

        return liquibase;
    }

    /**
     * Subclasses may override this method add change some database settings such as
     * default schema before returning the database object.
     * 
     * @param c
     * @return a Database implementation retrieved from the {@link DatabaseFactory}.
     * @throws DatabaseException
     */
    protected Database createDatabase(Connection c) throws DatabaseException {
        Database database = DatabaseFactory.getInstance().findCorrectDatabaseImplementation(new JdbcConnection(c));
        if (StringUtils.trimToNull(this.defaultSchema) != null) {
            database.setDefaultSchemaName(this.defaultSchema);
        }
        return database;
    }

    /**
     * Create a database connection to hibernate model.
     * This is useful for diff report
     * 
     * @throws DatabaseException
     */
    protected Database createHibernateDatabase() throws DatabaseException {
        Database referenceDatabase = CommandLineUtils.createDatabaseObject(
                this.getClass().getClassLoader(),
                "hibernate:classic:hibernate.cfg.xml",
                null,
                null,
                null,
                config.getJdbcCatalog(), config.getJdbcSchema(),
                false, false,
                null, null, null, null
                );

        return referenceDatabase;
    }

    public void setChangeLogParameters(Map<String, String> parameters) {
        this.parameters = parameters;
    }

    /**
     * Create a new resourceOpener.
     */
    protected ResourceAccessor createResourceAccessor() {
        // If Spring started, resolve using Spring
        if (this.resourceLoader != null) {
            // FIXME BLA : waiting Liquibase bug fix (see mantis #23602) 
            //return new SpringResourceOpener(getChangeLog());
            
            return new SpringResourceAccessor(getChangeLog());
        }
        
        // Classpath resource accessor
        if (isClasspathPrefixPresent(changeLog)) {
            return new ClassLoaderResourceAccessor(this.getClass().getClassLoader());
        }
        
        // File resource accessor
        return new FileSystemResourceAccessor(new File(adjustNoFilePrefix(changeLog)).getParent());
    }

    /**
     * Spring sets this automatically to the instance's configured bean name.
     */
    @Override
    public void setBeanName(String name) {
        this.beanName = name;
    }

    /**
     * @return the Spring-name of this instance.
     */
    public String getBeanName() {
        return beanName;
    }

    @Override
    public void setResourceLoader(ResourceLoader resourceLoader) {
        this.resourceLoader = resourceLoader;
    }

    public ResourceLoader getResourceLoader() {
        return resourceLoader;
    }

    @Override
    public String toString() {
        return getClass().getName() + "(" + this.getResourceLoader().toString() + ")";
    }

    protected void computeMaxChangeLogFileVersion() {
        this.maxChangeLogFileVersion = null;

        // Get the changelog path
        String changeLogPath = getChangeLog();
        if (StringUtils.isBlank(changeLogPath)) {
            return;
        }

        // Secure all separator (need for regex)
        changeLogPath = changeLogPath.replaceAll("\\\\", "/");

        // Get the parent folder path
        int index = changeLogPath.lastIndexOf('/');
        if (index == -1 || index == changeLogPath.length() - 1) {
            return;
        }

        // Compute a regex (based from changelog master file)
        String changeLogWithVersionRegex = changeLogPath.substring(index + 1);
        changeLogWithVersionRegex = changeLogWithVersionRegex.replaceAll("master\\.xml", "([0-9]\\\\.[.-_a-zA-Z]+)\\\\.xml");
        Pattern changeLogWithVersionPattern = Pattern.compile(changeLogWithVersionRegex);

        Version maxVersion = null;

        PathMatchingResourcePatternResolver resolver = new PathMatchingResourcePatternResolver(resourceLoader);

        try {
            // Get resources from classpath
            String pathPrefix = changeLogPath.substring(0, index);
            Resource[] resources = resolver.getResources(pathPrefix + "/db-changelog-*.xml");
            for (Resource resource : resources) {
                String filename = resource.getFilename();
                Matcher matcher = changeLogWithVersionPattern.matcher(filename);

                // If the filename match the changelog with version pattern
                if (matcher.matches()) {
                    String fileVersion = matcher.group(1);
                    // Skip SNAPSHOT versions
                    if (fileVersion.endsWith(CHANGE_LOG_SNAPSHOT_SUFFIX) == false) {
                        try {
                            Version version = Versions.valueOf(fileVersion);

                            // Store a version has max if need
                            if (maxVersion == null || maxVersion.compareTo(version) < 0) {
                                maxVersion = version;
                            }
                        } catch (IllegalArgumentException iae) {
                            // Bad version format : log but continue
                            getLog().warning(
                                    String.format(
                                            "Bad format version found in file: %s/%s. Ignoring this file when computing the max schema version.",
                                            changeLogPath, filename));
                        }
                    }
                }
            }
        } catch (IOException e) {
            throw new RuntimeException("Could not get changelog files", e);
        }

        if (maxVersion != null) {
            this.maxChangeLogFileVersion = maxVersion;
        }
    }

    /**
     * Get the max version from all change log files.
     * change log file with version must have a same pattern as the master changelog
     * 
     * @return the max version founded in files, or null if version found
     */
    public Version getMaxChangeLogFileVersion() {
        return this.maxChangeLogFileVersion;
    }

    /**
     * Generate a diff report (using text format)
     * 
     * @param outputFile
     * @param typesToControl
     *            a comma separated database object to check (i.e Table, View, Column...). If null, all types are
     *            checked
     * @throws LiquibaseException
     */
    public void reportDiff(File outputFile, String typesToControl) throws LiquibaseException {
        Connection c = null;
        liquibase.Liquibase liquibase;
        PrintStream writer = null;
        try {
            // open connection
            c = createConnection();

            // create liquibase instance
            liquibase = createLiquibase(c);

            // First, release locks, then update and release locks again
            liquibase.forceReleaseLocks();
            DiffResult diffResult = performDiff(liquibase, typesToControl);
            liquibase.forceReleaseLocks();

            // Write the result into report file
            writer = outputFile != null ? new PrintStream(outputFile) : null;

            new DiffToReport(diffResult, writer != null ? writer : System.out)
                    .print();

        } catch (SQLException e) {
            throw new DatabaseException(e);
        } catch (FileNotFoundException e) {
            throw new AdagioTechnicalException("Could not write diff report file.", e);
        } finally {
            if (c != null) {
                try {
                    c.rollback();
                } catch (SQLException e) {
                    // nothing to do
                }
                releaseConnection(c);
            }
            if (writer != null) {
                writer.close();
            }
        }
    }

    /**
     * Generate a changelog file, with all diff found
     * 
     * @param changeLogFile
     * @param typesToControl
     *            a comma separated database object to check (i.e Table, View, Column...). If null, all types are
     *            checked
     * @throws LiquibaseException
     */
    public void generateDiffChangelog(File changeLogFile, String typesToControl) throws LiquibaseException {
        Connection c = null;
        liquibase.Liquibase liquibase;
        PrintStream writer = null;
        try {
            // open connection
            c = createConnection();

            // create liquibase instance
            liquibase = createLiquibase(c);

            // First, release locks, then update and release locks again
            liquibase.forceReleaseLocks();
            DiffResult diffResult = performDiff(liquibase, typesToControl);
            liquibase.forceReleaseLocks();

            // Write the result into report file
            writer = changeLogFile != null ? new PrintStream(changeLogFile) : null;

            DiffOutputControl diffOutputControl = new DiffOutputControl(false, false, false);
            new DiffToChangeLog(diffResult, diffOutputControl)
                    .print(writer != null ? writer : System.out);

        } catch (SQLException e) {
            throw new DatabaseException(e);
        } catch (FileNotFoundException e) {
            throw new AdagioTechnicalException("Could not generate changelog file.", e);
        } catch (ParserConfigurationException e) {
            throw new AdagioTechnicalException("Could not generate changelog file.", e);
        } catch (IOException e) {
            throw new AdagioTechnicalException("Could not generate changelog file.", e);
        } finally {
            if (c != null) {
                try {
                    c.rollback();
                } catch (SQLException e) {
                    // nothing to do
                }
                releaseConnection(c);
            }
            if (writer != null) {
                writer.close();
            }
        }
    }

    /**
     * @param liquibase
     *            the connection to the target database
     * @param typesToControl
     *            a comma separated database object to check (i.e Table, View, Column...). If null, all types are
     *            checked
     * @return the diff result
     * @throws LiquibaseException
     */
    protected DiffResult performDiff(liquibase.Liquibase liquibase, String typesToControl) throws LiquibaseException {
        Database referenceDatabase = createHibernateDatabase();
        CompareControl compareControl = new CompareControl(DatabaseObjectFactory.getInstance().parseTypes(typesToControl));
        DiffResult diffResult = liquibase.diff(referenceDatabase, liquibase.getDatabase(), compareControl);
        return diffResult;
    }

    /*
     * FIXME BLA : uncomment when it will be possible to use Liquibase 3.2+ (waiting bug fis on Hsqldb1.8 (mantis #23602) 
     *
    public class SpringResourceOpener implements ResourceAccessor {

        private String parentFile;
        public SpringResourceOpener(String parentFile) {
            this.parentFile = parentFile;
        }

        @Override
        public Set<String> list(String relativeTo, String path, boolean includeFiles, boolean includeDirectories, boolean recursive) throws IOException {
            Set<String> returnSet = new HashSet<String>();

            Resource[] resources = ResourcePatternUtils.getResourcePatternResolver(getResourceLoader()).getResources(adjustClasspath(path));

            for (Resource res : resources) {
                returnSet.add(res.getURL().toExternalForm());
            }

            return returnSet;
        }

        @Override
        public Set<InputStream> getResourcesAsStream(String path) throws IOException {
            Set<InputStream> returnSet = new HashSet<InputStream>();
            Resource[] resources = ResourcePatternUtils.getResourcePatternResolver(getResourceLoader()).getResources(adjustClasspath(path));

            if (resources == null || resources.length == 0) {
                return null;
            }
            for (Resource resource : resources) {
                returnSet.add(resource.getURL().openStream());
            }

            return returnSet;
        }

        public Resource getResource(String file) {
            return getResourceLoader().getResource(adjustClasspath(file));
        }

        private String adjustClasspath(String file) {
            return isPrefixPresent(parentFile) && !isPrefixPresent(file) ? ResourceLoader.CLASSPATH_URL_PREFIX + file : file;
        }

        public boolean isPrefixPresent(String file) {
            if (file.startsWith("classpath") || file.startsWith("file:") || file.startsWith("url:")) {
                return true;
            }
            return false;
        }

        @Override
        public ClassLoader toClassLoader() {
            return getResourceLoader().getClassLoader();
        }
    }
    */
    
    public class SpringResourceAccessor implements ResourceAccessor {
        private String parentFile;

        public SpringResourceAccessor(String parentFile) {
            this.parentFile = parentFile;
        }

        @Override
        public InputStream getResourceAsStream(String file) throws IOException {
            try {
                Resource resource = getResource(file);
                return resource.getInputStream();
            } catch (FileNotFoundException ex) {
                return null;
            }
        }

        @Override
        public Enumeration<URL> getResources(String packageName) throws IOException {
            Vector<URL> tmp = new Vector<URL>();

            tmp.add(getResource(packageName).getURL());

            return tmp.elements();
        }

        public Resource getResource(String file) {
            return resourceLoader.getResource(adjustClasspath(this.parentFile, file));
        }

        @Override
        public ClassLoader toClassLoader() {
            return getResourceLoader().getClassLoader();
        }
    }

    protected Logger getLog() {
        return liquibase.logging.LogFactory.getInstance().getLog();
    }

    protected Connection createConnection() throws SQLException {
        return createConnection(null);
    }
    
    /**
     * Create a connection from the given properties.<p/>
     * If JDBC Url is equals to the datasource, use the datsource to create the connection
     * 
     * @param connectionProperties
     * @return
     * @throws SQLException
     */
    protected Connection createConnection(Properties connectionProperties) throws SQLException {
        Properties targetConnectionProperties = (connectionProperties != null) ? connectionProperties : config.getConnectionProperties();
        String jdbcUrl = targetConnectionProperties.getProperty(Environment.URL);
        if (Objects.equals(config.getJdbcURL(), jdbcUrl) && dataSource != null) {
            return DataSourceUtils.getConnection(dataSource);
        }
        return DaoUtils.createConnection(targetConnectionProperties);
    }

    protected void releaseConnection(Connection conn) {
        if (dataSource != null) {
            DataSourceUtils.releaseConnection(conn, dataSource);
            return;
        }
        DaoUtils.closeSilently(conn);
    }
    
    protected String adjustClasspath(String parentFile, String file) {
        return isClasspathPrefixPresent(parentFile) && !isClasspathPrefixPresent(file)
                ? ResourceLoader.CLASSPATH_URL_PREFIX + file
                : file;
    }
    
    protected String adjustNoClasspath(String file) {
        return isClasspathPrefixPresent(file)
                ? file.substring(ResourceLoader.CLASSPATH_URL_PREFIX.length())
                : file;
    }
    
    protected boolean isClasspathPrefixPresent(String file) {
        return file.startsWith(ResourceLoader.CLASSPATH_URL_PREFIX);
    }
    
    protected boolean isFilePrefixPresent(String file) {
        return file.startsWith(ResourceUtils.FILE_URL_PREFIX);
    }
    
    protected String adjustNoFilePrefix(String file) {
        return isFilePrefixPresent(file)
                ? file.substring(ResourceUtils.FILE_URL_PREFIX.length())
                : file;
    }

}
