package fr.ifremer.adagio.core.dao.technical.jdbc;

/*
 * #%L
 * SIH-Adagio :: Shared
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2015 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */


import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;

import javax.sql.DataSource;

import org.springframework.dao.DataAccessResourceFailureException;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowCallbackHandler;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.datasource.DataSourceUtils;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;

import fr.ifremer.adagio.core.AdagioTechnicalException;
import fr.ifremer.adagio.core.dao.technical.DaoUtils;

/**
 * Allow pure JDBC queries, that support query on dataSource connection OR from other database connection 
 * 
 * @author Benoit Lavenier <benoit.lavenier@e-is.pro>
 * @since 3.8.3
 * 
 */
public class OptionalDatasourceJdbcDaoSupport {

    private final DataSource dataSource;
    private final String databaSourceJdbcUrl;

    public OptionalDatasourceJdbcDaoSupport(DataSource dataSource) {
        this.dataSource = dataSource;
        this.databaSourceJdbcUrl = getJdbcUrl(dataSource);
    }

    public OptionalDatasourceJdbcDaoSupport() {
        this.dataSource = null;
        this.databaSourceJdbcUrl = null;
    }

    /* -- internal methods -- */

    protected <T> T query(Properties connectionProperties, String sql, Map<String, Object> paramMap, ResultSetExtractor<T> rse) {
        Connection connection = null;
        PreparedStatement statement = null;
        ResultSet rs = null;
        try {
            connection = createConnection(connectionProperties);
            statement = DaoUtils.bindQuery(connection, sql, paramMap);
            rs = statement.executeQuery();
            if (!rs.next()) {
                return null;
            }

            T target = rse.extractData(rs);

            return target;
        } catch (SQLException e) {
            throw new DataAccessResourceFailureException(String.format("Error while executing query [%s]: %s", sql, e.getMessage()), e);
        } finally {
            DaoUtils.closeSilently(rs);
            DaoUtils.closeSilently(statement);
            closeSilently(connection);
        }
    }

    /**
     * Query given SQL to create a prepared statement from SQL and a list of arguments to bind to the query,
     * reading the ResultSet with a ResultSetExtractor.
     */
    protected void query(Properties connectionProperties, String sql, Map<String, Object> paramMap, RowCallbackHandler rch) {
        Connection connection = null;
        PreparedStatement statement = null;
        ResultSet rs = null;
        try {
            connection = createConnection(connectionProperties);
            statement = DaoUtils.bindQuery(connection, sql, paramMap);
            rs = statement.executeQuery();
            while (rs.next()) {
                rch.processRow(rs);
            }
        } catch (SQLException e) {
            throw new DataAccessResourceFailureException(String.format("Error while executing query [%s]: %s", sql, e.getMessage()), e);
        } finally {
            DaoUtils.closeSilently(rs);
            DaoUtils.closeSilently(statement);
            closeSilently(connection);
        }
    }

    /**
     * Query given SQL to create a prepared statement from SQL and a list of arguments to bind to the query, reading the
     * ResultSet on a per-row basis with a RowCallbackHandler.
     */
    protected <T> List<T> query(Properties connectionProperties, String sql, Map<String, Object> paramMap, RowMapper<T> rowMapper) {
        Connection connection = null;
        PreparedStatement statement = null;
        ResultSet rs = null;
        try {
            connection = createConnection(connectionProperties);
            statement = DaoUtils.bindQuery(connection, sql, paramMap);
            rs = statement.executeQuery();
            List<T> result = Lists.newArrayList();
            int rowNumber = 1;
            while (rs.next()) {
                T row = rowMapper.mapRow(rs, rowNumber++);
                result.add(row);
            }
            return result;
            
        } catch (SQLException e) {
            throw new DataAccessResourceFailureException(String.format("Error while executing query [%s]: %s", sql, e.getMessage()), e);
        } finally {
            DaoUtils.closeSilently(rs);
            DaoUtils.closeSilently(statement);
            closeSilently(connection);
        }
    }

    protected Connection createConnection(Properties connectionProperties) throws SQLException {
        if (connectionProperties == null) {
            Preconditions.checkNotNull(dataSource);
            return DataSourceUtils.getConnection(dataSource);
        }

        String jdbcUrl = DaoUtils.getUrl(connectionProperties);
        if (dataSource != null && Objects.equals(databaSourceJdbcUrl, jdbcUrl)) {
            return DataSourceUtils.getConnection(dataSource);
        }
        return DaoUtils.createConnection(connectionProperties);
    }

    protected void closeSilently(Connection connection) {
        Preconditions.checkNotNull(connection);

        String jdbcUrl;
        try {
            jdbcUrl = connection.getMetaData().getURL();
            if (dataSource != null && Objects.equals(databaSourceJdbcUrl, jdbcUrl)) {
                DataSourceUtils.releaseConnection(connection, dataSource);
                return;
            }
        } catch (SQLException e) {
            // continue
        }

        DaoUtils.closeSilently(connection);
    }

    protected String getJdbcUrl(DataSource dataSource) {
        Preconditions.checkNotNull(dataSource);
        Connection connection = null;
        try {
            connection = dataSource.getConnection();
            return connection.getMetaData().getURL();
        } catch (SQLException e) {
            throw new AdagioTechnicalException("Could not read the JDBC URL from datasource.", e);
        } finally {
            DaoUtils.closeSilently(connection);
        }
    }

    
    protected Integer safeGetInteger(ResultSet rs, int columnIndex) throws SQLException {
        Object value = rs.getObject(columnIndex);
        if (value == null) {
            return null;
        }

        // Adagio / Oracle DBMS return a BigDecimal from NUMBER(10) columns
        if (value instanceof BigDecimal) {
            return ((BigDecimal) value).intValue();
        }

        // for all other java types: convert into Integer
        return Integer.parseInt(value.toString());
    }    
}
