package fr.ifremer.adagio.core.dao.technical.hibernate;

/*
 * #%L
 * SIH-Adagio Core Shared
 * $Id: HibernateDaoSupport.java 12486 2014-12-08 14:38:44Z lp1ee9d $
 * $HeadURL: https://forge.ifremer.fr/svn/sih-adagio/tags/adagio-3.8.6.4/core-shared/src/main/java/fr/ifremer/adagio/core/dao/technical/hibernate/HibernateDaoSupport.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.io.Serializable;
import java.sql.Timestamp;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.type.Type;
import org.springframework.dao.DataIntegrityViolationException;

import com.google.common.base.Preconditions;

import fr.ifremer.adagio.core.config.AdagioConfiguration;

public class HibernateDaoSupport {

    /** Logger. */
    protected static final Log logger =
            LogFactory.getLog(HibernateDaoSupport.class);

	protected boolean debugEntityLoad = false;
	
	private SessionFactory sessionFactory;
	
    private Calendar calendar = new GregorianCalendar();

	
	public HibernateDaoSupport() {
		this.debugEntityLoad = AdagioConfiguration.getInstance().debugEntityLoad();
	}

	protected void setSessionFactory(SessionFactory sessionFactory) {
		this.sessionFactory = sessionFactory;		
	}
	
	protected SessionFactory getSessionFactory() {
		return this.sessionFactory;
	}
	
	protected Session getSession() {
		return sessionFactory.getCurrentSession();
	}
	
	/**
	 * @deprecated use getSession() instead (not need since hibernate 4+ and Spring annotations)
	 * @param oldAPIArg
	 * @return
	 */
	protected Session getSession(boolean oldAPIArg) {
		return sessionFactory.getCurrentSession();
	}
	
	protected void deleteAll(Collection<?> entities) {
		Session session = getSession();
		for (Object entity: entities) {
			session.delete(entity);
		}
	}

	protected Object[] queryUnique(String queryName, Object... params) {

		Query query = createQuery(queryName, params);
		Object result = query.uniqueResult();
		return (Object[]) result;
	}

	protected <T> T queryUniqueTyped(String queryName, Object... params) {

		Query query = createQuery(queryName, params);
		Object result = query.uniqueResult();
		return (T) result;
	}

	protected Long queryCount(String queryName, Object... params){
		Query query = createQuery(queryName, params);
		Object result = query.uniqueResult();
        if (result instanceof Integer) {
            return ((Integer) result).longValue();
        }
        return (Long) result;
    }
    
    protected Iterator<Object[]> queryIterator(String queryName, Object... params) {

		Query query = createQuery(queryName, params);
		Iterator result = query.iterate();
		return result;
	}

	protected <T> Iterator<T> queryIteratorTyped(String queryName, Object... params) {

		Query query = createQuery(queryName, params);
		Iterator result = query.iterate();
		return result;
	}

	protected List<Object[]> queryList(String queryName, Object... params) {

		Query query = createQuery(queryName, params);
		List result = query.list();
		return result;
	}

	protected <T> List<T> queryListTyped(String queryName, Object... params) {

		Query query = createQuery(queryName, params);
		List result = query.list();
		return result;
	}

	protected int queryUpdate(String queryName, Object... params) {

		Query query = createQuery(queryName, params);

		int result = query.executeUpdate();
		return result;
	}

	protected Query createQuery(String queryName, Object... params) {
		Query query = getSession().getNamedQuery(queryName);
		return setQueryParams(query, queryName, params);
	}
	
	protected Query setQueryParams(Query query, String queryName, Object... params) {
		if (params.length > 0) {

			Preconditions.checkArgument(
			        params.length % 3 == 0,
			        "Params must be tuple (paramName, paramType, paramValue)");

			int nbParams = params.length / 3;

			for (int i = 0; i < nbParams; i++) {
				String paramName = (String) params[3 * i];
				Type paramType = (Type) params[3 * i + 1];
				Object paramValue = params[3 * i + 2];
				if (paramType != null) {
					query.setParameter(paramName, paramValue, paramType);
				}
				else if (paramValue instanceof Collection<?>) {
					query.setParameterList(paramName, (Collection<?>)paramValue);
				} else {
					query.setParameter(paramName, paramValue);
				}
				if (logger.isDebugEnabled()) {
					logger.debug("query [" + queryName + "] (param " + i
					             + " [" + paramName + '=' + paramValue + "])");
				}
			}
		}
		return query;
	}

	protected <T extends Serializable> T load(Class<? extends T> clazz, Serializable id) {
		if (debugEntityLoad) {
			T load = (T) getSession().get(clazz, id);
			if (load == null) {
				throw new DataIntegrityViolationException("Unable to load entity " + clazz.getName() + " with identifier '" + id + "': not found in database.");
			}
		}
		T load = (T) getSession().load(clazz, id);
		return load;
	}

	protected <T extends Serializable> T get(Class<? extends T> clazz, Serializable id) {
		T load = (T) getSession().get(clazz, id);
		return load;
	}

	/**
	 * @param queryName
	 *            a name of HQL query (i.e. defined in queries.hbm.xml)
	 * @param pageNumber
	 *            page offset, starting at 1
	 * @param pageSize
	 *            number of items in a page
	 * @param orderBy
	 *            the order by expression
	 * @param isAscending
	 * @param params
	 * @return
	 */
	protected Iterator<Object[]> queryIteratorWithPage(String queryName, int pageNumber, int pageSize, String orderBy, boolean isAscending,
	                                                   Object... params) {
		Query query = getSession().getNamedQuery(queryName);
		String queryString = query.getQueryString();

		if (orderBy != null) {
			queryString += " order by " + orderBy;
			if (!isAscending) {
				queryString += " desc";
			}
		}

		query = createQuery(queryString, params);

		if (pageSize != -1) {
			query.setMaxResults(pageSize);
			if (pageNumber > 1) {
				query.setMaxResults(pageSize);
				query.setFirstResult((pageNumber - 1) * pageSize);
			}
		}
		return query.iterate();
	}


    protected Date newCreateDate() {
        return dateWithNoTime(new Date());
    }

    protected Date dateWithNoTime(Date date) {
        calendar.setTime(date);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        return calendar.getTime();
   }

    protected Date dateWithNoMiliSecond(Date date) {
        calendar.setTime(date);
        calendar.set(Calendar.MILLISECOND, 0);
        return calendar.getTime();
    }

    protected Date dateWithNoSecondAndMiliSecond(Date date) {
        calendar.setTime(date);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        return calendar.getTime();
    }

    protected Date dateWithNoSecondAndOneMiliSecond(Date date) {
        calendar.setTime(date);
        calendar.add(Calendar.SECOND, 0);
        calendar.add(Calendar.MILLISECOND, 1);
        return calendar.getTime();
    }

    protected Date dateWithOneMiliSecond(Date date) {
        calendar.setTime(date);
        calendar.add(Calendar.MILLISECOND, 1);
        return calendar.getTime();
    }

    protected Date dateOfYearWithOneMiliSecond(int year) {
        calendar.setTimeInMillis(0);
        calendar.set(Calendar.YEAR, year);
        calendar.set(Calendar.MILLISECOND, 1);
        return calendar.getTime();
    }

    protected long dateOfYearWithOneMiliSecondInMili(int year) {
        calendar.setTimeInMillis(0);
        calendar.set(Calendar.YEAR, year);
        calendar.set(Calendar.MILLISECOND, 1);
        return calendar.getTimeInMillis();
    }
    
    /**
     * Test if the date has millisecond set. This yes, return null, then return the date itself.
     *
     * @param databaseValue the date stored in the database (could be fake date, not null only because of database constraints)
     * @return null if the date is a fake date
     */
    protected Date convertDatabase2UI(Timestamp databaseValue) {
        Date result;
        if (databaseValue == null) {
            result = null;
        } else {

            calendar.setTimeInMillis(databaseValue.getTime());
            if (calendar.get(Calendar.MILLISECOND) != 0) {
                result = null;
            } else {
                result = calendar.getTime();
            }
        }
        return result;
    }

    /**
     * Convert a UI date, when the database value is mandatory.
     * If the given value is null, use the default date, then set millisecond to '1', to be able to retrieve the null value later.
     *
     * @param uiValue the date used in the UI
     * @return null if the date is a fake date
     */
    protected Date convertUI2DatabaseMandatoryDate(Date uiValue,
                                                   Date defaultNotEmptyDate,
                                                   boolean addOneSecondToDefaultDate) {
        Date result;

        // if ui date is not empty, then use it (but reset millisecond)
        if (uiValue == null) {

            Preconditions.checkState(
                    defaultNotEmptyDate != null,
                    "'defaultNotEmptyDate' could not be null.");


            calendar.setTime(defaultNotEmptyDate);
            if (addOneSecondToDefaultDate) {
                calendar.add(Calendar.SECOND, 1);
            }
            calendar.set(Calendar.MILLISECOND, 1);
            result = calendar.getTime();
        } else {

            result = dateWithNoMiliSecond(uiValue);
        }

        return result;
    }

	
}
