package fr.ifremer.adagio.core.dao.technical.hibernate;

/*
 * #%L
 * SIH-Adagio Core Shared
 * $Id: DatabaseSchemaDaoImpl.java 12604 2015-01-30 15:06:51Z bl05b3e $
 * $HeadURL: https://forge.ifremer.fr/svn/sih-adagio/tags/adagio-3.8.6.4/core-shared/src/main/java/fr/ifremer/adagio/core/dao/technical/hibernate/DatabaseSchemaDaoImpl.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.Set;

import javax.sql.DataSource;

import liquibase.exception.LiquibaseException;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.HibernateException;
import org.hibernate.SessionFactory;
import org.hibernate.cfg.Configuration;
import org.hibernate.cfg.Environment;
import org.hibernate.dialect.Dialect;
import org.hibernate.engine.spi.SessionFactoryImplementor;
import org.hibernate.tool.hbm2ddl.SchemaExport;
import org.hibernate.tool.hbm2ddl.SchemaUpdate;
import org.nuiton.i18n.I18n;
import org.nuiton.util.version.Version;
import org.nuiton.util.version.Versions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Lazy;
import org.springframework.core.io.Resource;
import org.springframework.dao.DataAccessResourceFailureException;
import org.springframework.jdbc.CannotGetJdbcConnectionException;
import org.springframework.jdbc.datasource.DataSourceUtils;
import org.springframework.stereotype.Repository;
import org.springframework.util.ResourceUtils;

import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import fr.ifremer.adagio.core.AdagioTechnicalException;
import fr.ifremer.adagio.core.config.AdagioConfiguration;
import fr.ifremer.adagio.core.config.AdagioConfigurationOption;
import fr.ifremer.adagio.core.dao.technical.DaoUtils;
import fr.ifremer.adagio.core.dao.technical.DatabaseSchemaDao;
import fr.ifremer.adagio.core.dao.technical.DatabaseSchemaUpdateException;
import fr.ifremer.adagio.core.dao.technical.VersionNotFoundException;
import fr.ifremer.adagio.core.dao.technical.liquibase.Liquibase;
import fr.ifremer.adagio.core.service.technical.SpringUtils;

@Repository("databaseSchemaDao")
@Lazy
public class DatabaseSchemaDaoImpl
    extends HibernateDaoSupport
    implements DatabaseSchemaDao {

    /**
     * Logger.
     */
    private static final Log log
        = LogFactory.getLog(DatabaseSchemaDaoImpl.class);

    @Autowired
    private ApplicationContext appContext;

    @Autowired
    private Liquibase liquibase;

    @Autowired
    private DataSource dataSource;

    @Autowired
    private AdagioConfiguration config;

    private Dialect localDialect = null;

    /**
     * Constructor used by Spring
     *
     * @param config
     */
    @Autowired
    public DatabaseSchemaDaoImpl(SessionFactory sessionFactory) {
        super();
        setSessionFactory(sessionFactory);
    }

    /**
     * Constructor to use when Spring not started
     *
     * @param config
     */
    public DatabaseSchemaDaoImpl(AdagioConfiguration config) {
        super();
        this.config = config;
        this.liquibase = new Liquibase(config);
    }
    
    /**
     * Constructor to use when Spring not started
     *
     * @param config
     */
    public DatabaseSchemaDaoImpl(AdagioConfiguration config, Liquibase liquibase) {
        super();
        this.config = config;
        this.liquibase = liquibase;
    }

    @Override
    public void generateCreateSchemaFile(String filename) {
        if (filename == null || filename.isEmpty()) {
            throw new IllegalArgumentException("filename could not be null or empty.");
        }
        generateCreateSchemaFile(filename, false, false, true);
    }

    @Override
    public void generateCreateSchemaFile(String filename, boolean doExecute, boolean withDrop, boolean withCreate) {

        // Get the hibernate configuration for the extractor schema
        Configuration cfg = getHibernateConfiguration();

        // Apply the schema export
        SchemaExport se = new SchemaExport(cfg);
        se.setDelimiter(";");
        se.setOutputFile(filename);
        se.execute(false, doExecute, !withCreate, !withDrop);
    }

    @Override
    public void generateUpdateSchemaFile(String filename) {
        if (filename == null || filename.isEmpty()) {
            throw new IllegalArgumentException("filename could not be null or empty.");
        }
        generateUpdateSchemaFile(filename, false);
    }

    @Override
    public void generateUpdateSchemaFile(String filename, boolean doUpdate) {

        // Get the hibernate configuration for the extractor schema
        Configuration cfg = getHibernateConfiguration();

        // Apply the schema export
        SchemaUpdate su = new SchemaUpdate(cfg);
        su.setDelimiter(";");
        su.setOutputFile(filename);
        su.execute(false, false);
    }

    /**
     * Create a new hibernate configuration, with all hbm.xml files for the schema need for app
     *
     * @return the hibernate Configuration
     */
    private Configuration getHibernateConfiguration() {

        // Building a new configuration
        Configuration cfg = new Configuration();
        try {
            // Add hibernate files from ".dao.*" package :
            addRessourceToHibernateConfiguration(cfg, Package.getPackage("fr.ifremer.adagio.core.dao"), "**/*.hbm.xml");

            // Add an additional queries files :
            addRessourceToHibernateConfiguration(cfg, "", "queries.hbm.xml");
            addRessourceToHibernateConfiguration(cfg, "", AdagioConfiguration.getInstance().getHibernateClientQueriesFile());

        } catch (IOException e) {
            log.error("exportExtractorSchemaToFile failed", e); //$NON-NLS-1$
            throw new DataAccessResourceFailureException(e.getMessage(), e);
        }
        // Set hibernate dialect
        cfg.setProperty(Environment.DIALECT, AdagioConfiguration.getInstance().getHibernateDialect());

        // To be able to retrieve connection from datasource
        HibernateConnectionProvider.setDataSource(dataSource);
        cfg.setProperty(Environment.CONNECTION_PROVIDER, HibernateConnectionProvider.class.getName());

        return cfg;
    }

    private void addRessourceToHibernateConfiguration(Configuration cfg, Package aPackage, String filePattern) throws IOException {
        String packageName = aPackage.getName().replace(".", "/");
        addRessourceToHibernateConfiguration(cfg, packageName, "**/*.hbm.xml");

    }

    private void addRessourceToHibernateConfiguration(Configuration cfg, String classPathFolder, String filePattern) throws IOException {
        Preconditions.checkNotNull(appContext, "No ApplicationContext found. Make bean initialization has been done by Spring.");
        String fullName = null;
        if (classPathFolder != null && !classPathFolder.isEmpty()) {
            fullName = classPathFolder + "/" + filePattern;
        }
        else {
            fullName = filePattern;
        }
        org.springframework.core.io.Resource[] resources = appContext.getResources("classpath*:" + fullName);
        for (Resource resource : resources) {
            String path = resource.getURL().toString();
            if (classPathFolder != null && !classPathFolder.isEmpty()) {

                int index = path.lastIndexOf(classPathFolder);
                if (index != -1) {
                    path = path.substring(index);
                }
            }
            else {
                int index = path.lastIndexOf("/");
                if (index != -1) {
                    path = path.substring(index + 1);
                }
            }
            cfg.addResource(path);
        }
    }

    public Dialect getLocalDialect() {
        if (localDialect == null) {
            localDialect = ((SessionFactoryImplementor) getSessionFactory()).getDialect();
        }
        return localDialect;
    }

    @Override
    public void updateSchema() throws DatabaseSchemaUpdateException {
        updateSchema(null);
    }
    
    @Override
    public void updateSchema(Properties connectionProperties) throws DatabaseSchemaUpdateException {
        try {
            liquibase.executeUpdate(connectionProperties);
        } catch (LiquibaseException le) {
            if (log.isErrorEnabled()) {
                log.error(le.getMessage(), le);
            }
            throw new DatabaseSchemaUpdateException("Could not update schema", le);
        }
    }

    @Override
    public void generateStatusReport(File outputFile) throws IOException {
        FileWriter fw = new FileWriter(outputFile);
        try {
            liquibase.reportStatus(fw);
        } catch (LiquibaseException le) {
            if (log.isErrorEnabled()) {
                log.error(le.getMessage(), le);
            }
            throw new AdagioTechnicalException("Could not report database status", le);
        }
    }

    @Override
    public void generateDiffReport(File outputFile, String typesToControl) throws IOException {
        try {
            liquibase.reportDiff(outputFile, typesToControl);
        } catch (LiquibaseException le) {
            if (log.isErrorEnabled()) {
                log.error(le.getMessage(), le);
            }
            throw new AdagioTechnicalException("Could not report database diff", le);
        }
    }

    @Override
    public void generateDiffChangeLog(File outputChangeLogFile, String typesToControl) throws IOException {
        try {
            liquibase.generateDiffChangelog(outputChangeLogFile, typesToControl);
        } catch (LiquibaseException le) {
            if (log.isErrorEnabled()) {
                log.error(le.getMessage(), le);
            }
            throw new AdagioTechnicalException("Could not create database diff changelog", le);
        }
    }

    @Override
    public Version getSchemaVersion() throws VersionNotFoundException {
        String systemVersion;
        try {
            systemVersion = queryUniqueTyped("lastSystemVersion");
            if (StringUtils.isBlank(systemVersion)) {
                throw new VersionNotFoundException(String.format("Could not get the schema version. No version found in SYSTEM_VERSION table."));
            }
        } catch (HibernateException he) {
            throw new VersionNotFoundException(String.format("Could not get the schema version: %s", he.getMessage()));
        }
        try {
            return Versions.valueOf(systemVersion);
        } catch (IllegalArgumentException iae) {
            throw new VersionNotFoundException(String.format("Could not get the schema version. Bad schema version found table SYSTEM_VERSION: %s",
                systemVersion));
        }
    }

    @Override
    public Version getSchemaVersionIfUpdate() {
        return liquibase.getMaxChangeLogFileVersion();
    }

    @Override
    public boolean shouldUpdateSchema() throws VersionNotFoundException {
        return getSchemaVersion().compareTo(getSchemaVersionIfUpdate()) >= 0;
    }

    @Override
    public boolean isDbLoaded() {

        // Do not try to run the validation query if the DB not exists
        if (!isDbExists()) {
            log.warn("Database directory not found. Could not load database.");
            return false;
        }

        Connection connection = null;
        try {
            connection = DataSourceUtils.getConnection(dataSource);
        } catch (CannotGetJdbcConnectionException ex) {
            log.error(ex);
            DataSourceUtils.releaseConnection(connection, dataSource);
            return false;
        }

        // Retrieve a validation query, from configuration
        String dbValidatioNQuery = config.getDbValidationQuery();
        if (StringUtils.isNotBlank(dbValidatioNQuery)) {
            log.debug(String.format("Check if the database is loaded, using validation query: %s", dbValidatioNQuery));

            // try to execute the validation query
            Statement stmt = null;
            try {
                stmt = connection.createStatement();
                stmt.execute(dbValidatioNQuery);
            } catch (SQLException ex) {
                log.error(ex);
                return false;
            } finally {
                DaoUtils.closeSilently(stmt);
                DataSourceUtils.releaseConnection(connection, dataSource);
            }
        }
        else {
            DataSourceUtils.releaseConnection(connection, dataSource);
        }

        return true;
    }

    @Override
    public boolean isDbExists() {
        String jdbcUrl = config.getJdbcURL();

        if (DaoUtils.isFileDatabase(jdbcUrl) == false) {
            return true;
        }

        File f = new File(config.getDbDirectory(), config.getDbName() + ".script");
        return f.exists();
    }

    @Override
    public void generateNewDb(File dbDirectory, boolean replaceIfExists, File scriptFile, Properties connectionProperties) {
        Preconditions.checkNotNull(dbDirectory);

        // Log target connection
        if (log.isInfoEnabled()) {
            log.info(I18n.t("adagio.persistence.newEmptyDatabase.directory", dbDirectory));
        }
        // Check output directory validity
        if (dbDirectory.exists() && !dbDirectory.isDirectory()) {
            throw new AdagioTechnicalException(
                I18n.t("adagio.persistence.newEmptyDatabase.notValidDirectory.error", dbDirectory));
        }

        // Make sure the directory could be created
        try {
            FileUtils.forceMkdir(dbDirectory);
        } catch (IOException e) {
            throw new AdagioTechnicalException(
                I18n.t("adagio.persistence.newEmptyDatabase.mkdir.error", dbDirectory),
                e);
        }

        if (ArrayUtils.isNotEmpty(dbDirectory.listFiles())) {
            if (replaceIfExists) {
                log.info(I18n.t("adagio.persistence.newEmptyDatabase.deleteDirectory", dbDirectory));
                try {
                    FileUtils.deleteDirectory(dbDirectory);
                } catch (IOException e) {
                    throw new AdagioTechnicalException(
                        I18n.t("adagio.persistence.newEmptyDatabase.deleteDirectory.error", dbDirectory), e);
                }
            }
            else {
                throw new AdagioTechnicalException(
                    I18n.t("adagio.persistence.newEmptyDatabase.notEmptyDirectory.error", dbDirectory));
            }
        }

        // Get connections properties :
        Properties targetConnectionProperties = connectionProperties != null ? connectionProperties : config.getConnectionProperties();

        // Check connections
        if (!checkConnection(config, targetConnectionProperties)) {
            return;
        }

        try {
            // Create the database
            createEmptyDb(config, targetConnectionProperties, scriptFile);
        } catch (SQLException e) {
            throw new AdagioTechnicalException(
                I18n.t("adagio.persistence.newEmptyDatabase.create.error"),
                e);
        } catch (IOException e) {
            throw new AdagioTechnicalException(
                I18n.t("adagio.persistence.newEmptyDatabase.create.error"),
                e);
        }

        try {
            // Shutdown database at end
            DaoUtils.shutdownDatabase(targetConnectionProperties);
        } catch (SQLException e) {
            throw new AdagioTechnicalException(
                I18n.t("adagio.persistence.newEmptyDatabase.shutdown.error"),
                e);
        }
    }

    @Override
    public void generateNewDb(File dbDirectory, boolean replaceIfExists) {
        generateNewDb(dbDirectory, replaceIfExists, null, null);
    }

    /* -- Internal methods --*/
    protected boolean checkConnection(
        AdagioConfiguration config,
        Properties targetConnectionProperties) {

        // Log target connection
        if (log.isInfoEnabled()) {
            log.info("Connecting to target database...");
            log.info(String.format(" Database directory: %s", config.getDbDirectory()));
            log.info(String.format(" JDBC Driver: %s", config.getJdbcDriver()));
            log.info(String.format(" JDBC URL: %s", config.getJdbcURL()));
            log.info(String.format(" JDBC Username: %s", config.getJdbcUsername()));
        }

        // Check target connection
        boolean isValidConnection = DaoUtils.isValidConnectionProperties(targetConnectionProperties);
        if (!isValidConnection) {
            log.error("Connection error: could not connect to target database.");
            return false;
        }

        return true;
    }

    public void createEmptyDb(AdagioConfiguration config, Properties targetConnectionProperties, File scriptFile) throws SQLException, IOException {
        // Getting the script file
        String scriptPath = scriptFile == null ? config.getDbCreateScriptPath() : scriptFile.getAbsolutePath();
        Preconditions
            .checkArgument(
                StringUtils.isNotBlank(scriptPath),
                String.format(
                    "No path for the DB script has been set in the configuration. This is need to create a new database. Please set the option [%s] in configuration file.",
                    AdagioConfigurationOption.DB_CREATE_SCRIPT_PATH));
        scriptPath = scriptPath.replaceAll("\\\\", "/");
        if (log.isInfoEnabled()) {
            log.info("Will use create script: " + scriptPath);
        }

        // Make sure the path is an URL (if not, add "file:" prefix)
        String scriptPathWithPrefix = scriptPath;
        if (!ResourceUtils.isUrl(scriptPath)) {
            scriptPathWithPrefix = ResourceUtils.FILE_URL_PREFIX + scriptPath;
        }

        Resource scriptResource = SpringUtils.getResource(scriptPathWithPrefix);
        if (!scriptResource.exists()) {
            throw new AdagioTechnicalException(String.format("Could not find DB script file, at %s", scriptPath));
        }

        Connection connection = DaoUtils.createConnection(targetConnectionProperties);
        try {
            List<String> importScriptSql = getImportScriptSql(scriptResource);
            for (String sql : importScriptSql) {
                PreparedStatement statement = null;
                try {
                    statement = connection.prepareStatement(sql);
                    statement.execute();
                } catch (SQLException sqle) {
                    log.warn("SQL command failed : " + sql, sqle);
                    throw sqle;
                } finally {
                    DaoUtils.closeSilently(statement);
                }

            }
            connection.commit();
        } finally {
            DaoUtils.closeSilently(connection);
        }
    }

    protected List<String> getImportScriptSql(Resource scriptResource) throws IOException {
        List<String> result = Lists.newArrayList();

        Predicate<String> predicate = new Predicate<String>() {

            Set<String> forbiddenStarts = Sets.newHashSet(
                "SET ",
                "CREATE USER ",
                "CREATE SCHEMA ",
                "GRANT DBA TO ");

            @Override
            public boolean apply(String input) {
                boolean accept = true;
                for (String forbiddenStart : forbiddenStarts) {
                    if (input.startsWith(forbiddenStart)
                        // Allow this instruction
                        && !input.startsWith("SET WRITE_DELAY")) {
                        accept = false;
                        break;
                    }
                }
                return accept;
            }
        };

        InputStream is = scriptResource.getInputStream();
        try {
            Iterator<String> lines = IOUtils.lineIterator(is, Charsets.UTF_8);

            while (lines.hasNext()) {
                String line = lines.next().trim().toUpperCase();
                if (predicate.apply(line)) {
                    if (line.contains("\\U000A")) {
                        line = line.replaceAll("\\\\U000A", "\n");
                    }
                    // Reset sequence to zero
                    if (line.startsWith("CREATE SEQUENCE")) {
                        line = line.replaceAll("START WITH [0-9]+", "START WITH 0");
                    }
                    result.add(line);
                }
            }
        } finally {
            IOUtils.closeQuietly(is);
        }
        return result;
    }
}
