package fr.ifremer.adagio.core.config;

/*
 * #%L
 * SIH-Adagio Core Shared
 * $Id: EnumerationsInitializationHelper.java 11870 2013-12-03 10:10:32Z tc1fbb1 $
 * $HeadURL: https://forge.ifremer.fr/svn/sih-adagio/trunk/adagio/core-shared/src/main/java/fr/ifremer/adagio/core/dao/EnumerationsInitializationHelper.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
import java.io.IOException;
import java.util.List;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.config.ApplicationConfig;
import org.nuiton.util.RecursiveProperties;
import org.reflections.Reflections;
import org.springframework.core.io.Resource;

import com.google.common.collect.Lists;

import fr.ifremer.adagio.core.dao.technical.AdagioEnumerationDef;
import java.io.InputStreamReader;
import org.apache.commons.io.Charsets;
import org.apache.commons.io.IOUtils;

/**
 * Classe utilitaire pour vérifier que les valeurs d'énumération sont correctement configurées.
 *
 * @author Benoit Lavenier <benoit.lavenier@e-is.pro>
 */
public class AdagioEnumerationHelper {

    /**
     * Logger.
     */
    private static final Log log = LogFactory.getLog(AdagioEnumerationHelper.class);

    private static final String MODEL_PACKAGE_NAME = "fr.ifremer.adagio.core.dao";

    private static List<AdagioEnumerationDef<?>> cachedModelEnumsAsConfigOptions = null;

    @SuppressWarnings("rawtypes")
    public static List<AdagioEnumerationDef<?>> getAllModelEnumerations() {
        if (cachedModelEnumsAsConfigOptions != null) {
            return cachedModelEnumsAsConfigOptions;
        }

        // Retrieve enumerations classes used in adagio
        Reflections reflections = Reflections.collect();
        if (reflections == null) {
            reflections = new Reflections(MODEL_PACKAGE_NAME);
        }
        Set<Class<? extends AdagioEnumerationDef>> enumerationClasses
            = reflections.getSubTypesOf(AdagioEnumerationDef.class);
        if (log.isDebugEnabled()) {
            log.debug(String.format("%s enumeration classes detected in package [%s]", enumerationClasses.size(), MODEL_PACKAGE_NAME));
        }

        List<AdagioEnumerationDef<?>> options = Lists.newArrayList();
        for (Class<? extends AdagioEnumerationDef> enumClass : enumerationClasses) {
            AdagioEnumerationDef<?>[] enums = enumClass.getEnumConstants();
            if (ArrayUtils.isEmpty(enums)) {
                log.warn(String.format("Enumeration class [%s] has no value. Skipped.", enumClass.getName()));
            }
            else {
                for (AdagioEnumerationDef<?> aEnum : enums) {
                    AdagioEnumerationDef<?> configOption = (AdagioEnumerationDef<?>) aEnum;
                    options.add(configOption);
                }
            }
        }

        cachedModelEnumsAsConfigOptions = options;
        if (log.isDebugEnabled()) {
            log.debug(String.format("%s enumeration values detected in package [%s]", options.size(), MODEL_PACKAGE_NAME));
        }

        return cachedModelEnumsAsConfigOptions;
    }

    @SuppressWarnings({"rawtypes", "unchecked"})
    public static void reload(ApplicationConfig applicationConfig, List<org.springframework.core.io.Resource> resources) {

        Properties enumerationProperties = new RecursiveProperties();
        if (log.isDebugEnabled()) {
            log.debug(String.format("Starting to load enumeration values..."));
        }

        for (Resource resource : resources) {
            InputStreamReader isr = null;
            try {
                if (log.isDebugEnabled()) {
                    log.debug("Load enumeration file: "
                        + resource.getURI().toString());
                }
                isr = new InputStreamReader(resource.getInputStream(), Charsets.UTF_8);
                enumerationProperties.load(isr);
            } catch (IOException e) {
                log.warn(String.format("Could not load enumeration file: %s. File skipped.", resource.getFilename()));
            } finally {
                IOUtils.closeQuietly(isr);
            }
        }

        // Set each property as a default option for enumeration
        for (Object key : enumerationProperties.keySet()) {
            String value = enumerationProperties.getProperty((String) key);
            applicationConfig.setDefaultOption(AdagioEnumerationDef.CONFIG_OPTION_PREFIX + (String) key, value);
            if (log.isTraceEnabled()) {
                log.trace(String.format(" %s%s (%s)", AdagioEnumerationDef.CONFIG_OPTION_PREFIX, (String) key, value));
            }
        }

        // Refresh enumeration values
        List<AdagioEnumerationDef<?>> modelOptions = getAllModelEnumerations();
        for (AdagioEnumerationDef enumDef : modelOptions) {
            Class clazz = enumDef.getType();
            String key = enumDef.getKey();
            String stringValue = applicationConfig.getOption(key);

            Object value = applicationConfig.getOption(clazz, key);
            if (value != null
                && !enumDef.getValue().equals(value)) {

                // Check for conversion error
                if (stringValue.equals(value.toString()) == false) {
                    log.warn(String.format("Incompatible value '%s' for property '%s' (%s expected): Value skipped.", stringValue, key, enumDef.getType().getSimpleName()));
                }
                else {
                    Object oldValue = enumDef.getValue();
                    try {
                        enumDef.setValue(value);
                    } catch (ClassCastException cce) {
                        log.warn(String.format("Could not set %s with value %s", key, value));
                    }
                    if (log.isTraceEnabled()) {
                        log.trace(String.format(" %s (%s -> %s)", key, oldValue, value));
                    }
                }
            }
        }
    }

}
