package fr.ifremer.adagio.core.service;

/*
 * #%L
 * SIH-Adagio Core Shared
 * $Id: ServiceLocator.java 12053 2014-04-22 09:34:15Z tc1fbb1 $
 * $HeadURL: https://forge.ifremer.fr/svn/sih-adagio/tags/adagio-3.8.6.3/core-shared/src/main/java/fr/ifremer/adagio/core/service/ServiceLocator.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.io.Closeable;
import java.io.IOException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.access.BeanFactoryLocator;
import org.springframework.beans.factory.access.BeanFactoryReference;
import org.springframework.context.ApplicationContext;
import org.springframework.context.access.ContextSingletonBeanFactoryLocator;
import org.springframework.context.support.AbstractApplicationContext;

import fr.ifremer.adagio.core.service.referential.location.LocationService;
import fr.ifremer.adagio.core.service.technical.CacheService;

/**
 * Locates and provides all available application services.
 */
public class ServiceLocator implements Closeable {

	/* Logger */
	private static final Log log = LogFactory.getLog(ServiceLocator.class);

	/**
     * The default bean reference factory location.
     */
    private static final String DEFAULT_BEAN_REFERENCE_LOCATION = "beanRefFactory.xml";

    /**
     * The default bean reference factory ID.
     */
    private static final String DEFAULT_BEAN_REFERENCE_ID = "beanRefFactory";

    /**
     * The shared instance of this ServiceLocator.
     */
    private static ServiceLocator instance = new ServiceLocator();

    /**
     * Indicates if the spring context is open or not.
     */
    private boolean open = false;

    protected ServiceLocator() {
        // shouldn't be instantiated
        init(null, null);
    }

    protected ServiceLocator(String beanFactoryReferenceLocation,
                             String beanRefFactoryReferenceId) {
        init(beanFactoryReferenceLocation, beanRefFactoryReferenceId);
    }

    /**
     * replace the default shared instance of this Class
     *
     * @param newInstance the new shared service locator instance.
     */
    public static void setInstance(ServiceLocator newInstance) {
        instance = newInstance;
    }

    /**
     * Gets the shared instance of this Class
     *
     * @return the shared service locator instance.
     */
    public static ServiceLocator instance() {
        return instance;
    }

    /**
     * The bean factory reference instance.
     */
    private BeanFactoryReference beanFactoryReference;

    /**
     * The bean factory reference location.
     */
    private String beanFactoryReferenceLocation;

    /**
     * The bean factory reference id.
     */
    private String beanRefFactoryReferenceId;

    /**
     * Initializes the Spring application context from the given <code>beanFactoryReferenceLocation</code>. If <code>null</code> is
     * specified for the <code>beanFactoryReferenceLocation</code> then the
     * default application context will be used.
     *
     * @param beanFactoryReferenceLocation the location of the beanRefFactory reference.
     */
    public synchronized void init(String beanFactoryReferenceLocation,
                                  String beanRefFactoryReferenceId) {
		// Log if default values are overrided
    	if (log.isDebugEnabled() && beanFactoryReferenceLocation != null && beanRefFactoryReferenceId != null) {
			log.debug(String.format("Initializing ServiceLocator to use Spring bean factory [%s] at: %s", beanRefFactoryReferenceId,
			        beanFactoryReferenceLocation));
		}

        this.beanFactoryReferenceLocation =
                beanFactoryReferenceLocation == null ?
                DEFAULT_BEAN_REFERENCE_LOCATION :
                beanFactoryReferenceLocation;
        this.beanRefFactoryReferenceId = beanRefFactoryReferenceId == null ?
                                         DEFAULT_BEAN_REFERENCE_ID :
                                         beanRefFactoryReferenceId;
        this.beanFactoryReference = null;
    }

    /**
     * Initializes the Spring application context from the given <code>beanFactoryReferenceLocation</code>. If <code>null</code> is
     * specified for the <code>beanFactoryReferenceLocation</code> then the
     * default application context will be used.
     *
     * @param beanFactoryReferenceLocation the location of the beanRefFactory reference.
     */
    public synchronized void init(String beanFactoryReferenceLocation) {
        this.beanFactoryReferenceLocation = beanFactoryReferenceLocation == null ?
                                            DEFAULT_BEAN_REFERENCE_LOCATION :
                                            beanFactoryReferenceLocation;
        this.beanFactoryReference = null;
    }

    /**
     * Shuts down the ServiceLocator and releases any used resources.
     */
    public synchronized void shutdown() {
    	// Do not try to close if not already opened
    	if (!open) {
    		return;
    	}
    	if (log.isDebugEnabled()) {
			log.debug("Close Spring application context");
		}
    	
        ((AbstractApplicationContext) getContext()).close();
        if (beanFactoryReference != null) {
            beanFactoryReference.release();
            beanFactoryReference = null;
        }
        open = false;
    }

    /**
     * Get a service.
     *
     * @param name        name of the service (i.e name of the srping bean)
     * @param serviceType type of service
     * @param <S>         type of the service
     * @return the instanciated service
     */
    public <S> S getService(String name, Class<S> serviceType) {
        return getContext().getBean(name, serviceType);
    }

    /**
     * Gets an instance of {@link LocationService}.
     */
    public final LocationService getLocationService() {
        return getService("locationService", LocationService.class);
    }
 

    /**
     * Gets an instance of {@link CacheService}.
     */
    public final CacheService getCacheService() {
        return getService("cacheService", CacheService.class);
    }

    /**
     * @return {@code true} if spring context is open, {@code false} otherwise.
     * @since 3.5.2
     */
    public boolean isOpen() {
        return open;
    }

    /**
     * Gets the Spring ApplicationContext.
     */
    protected synchronized ApplicationContext getContext() {
        if (beanFactoryReference == null) {
            if (log.isDebugEnabled() && beanFactoryReferenceLocation != null && beanRefFactoryReferenceId != null) {
    			log.debug(String.format("Starting Spring application context using bean factory [%s] from file: %s", beanRefFactoryReferenceId,
    			        beanFactoryReferenceLocation));
    		}
            BeanFactoryLocator beanFactoryLocator =
                    ContextSingletonBeanFactoryLocator.getInstance(
                            beanFactoryReferenceLocation);
            beanFactoryReference = beanFactoryLocator
                    .useBeanFactory(beanRefFactoryReferenceId);
            
            open = true;
        }
        return (ApplicationContext) beanFactoryReference.getFactory();
    }

	@Override
	public void close() throws IOException {		
		shutdown();
	}
}