package fr.ifremer.adagio.core.service;

/*
 * #%L
 * SIH Allegro ObsDeb :: Core
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2013 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.io.File;
import java.io.IOException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.version.Version;
import org.nuiton.util.version.Versions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import fr.ifremer.adagio.core.AdagioTechnicalException;
import fr.ifremer.adagio.core.config.AdagioConfiguration;
import fr.ifremer.adagio.core.dao.technical.DatabaseSchemaDao;
import fr.ifremer.adagio.core.dao.technical.DatabaseSchemaUpdateException;
import fr.ifremer.adagio.core.dao.technical.VersionNotFoundException;

/**
 *
 * @author Lionel Touseau <lionel.touseau@e-is.pro>
 */
@Service("databaseSchemaService")
public class DatabaseSchemaServiceImpl implements DatabaseSchemaService {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(DatabaseSchemaServiceImpl.class);
    
    @Autowired
    protected AdagioConfiguration config;

    @Autowired
    protected DatabaseSchemaDao databaseSchemaDao;
    
    @Override
    public Version getDbVersion() {
        Version result;
        try {
            if (!isDbLoaded()) {
                throw new VersionNotFoundException("db is not open");
            }
            result = databaseSchemaDao.getSchemaVersion();
        } catch (VersionNotFoundException e) {
            if (log.isWarnEnabled()) {
                log.warn(e.getMessage());
            }
            result = null;
        }
        return result;
    }
    
    @Override
    public Version getApplicationVersion() {
        Version result = databaseSchemaDao.getSchemaVersionIfUpdate();
        return result;
    }

    @Override
    public void updateSchema() {
        try {
            databaseSchemaDao.updateSchema();
        } catch (DatabaseSchemaUpdateException e) {
            throw new AdagioTechnicalException(e.getCause());
        }
    }

    @Override
    public boolean isDbLoaded() {
        return databaseSchemaDao.isDbLoaded();
    }
    
    @Override
    public boolean isDbExists() {    	
        return databaseSchemaDao.isDbExists();
    }
    
    @Override
    public void generateStatusReport(File outputFile) throws IOException {
        if (outputFile == null || !outputFile.getParentFile().isDirectory() || !outputFile.canWrite()) {
            log.error("Could not write into the output file. Please make sure the given path is a valid path.");
            return;
        }        

        databaseSchemaDao.generateStatusReport(outputFile);
    }
    
    @Override
    public void generateDiffReport(File outputFile) throws IOException {
        if (outputFile == null || !outputFile.getParentFile().isDirectory() || !outputFile.canWrite()) {
            log.error("Could not write into the output file. Please make sure the given path is a valid path.");
            return;
        }        
        databaseSchemaDao.generateDiffReport(outputFile, config.getLiquibaseDiffTypes());
    }
    
    @Override
    public void generateDiffChangeLog(File outputFile) throws IOException {
        if (outputFile == null 
                || !outputFile.getParentFile().isDirectory() 
                || (outputFile.exists() && !outputFile.canWrite())) {
            log.error("Could not write into the output file. Please make sure the given path is a valid path.");
            return;
        }        
        databaseSchemaDao.generateDiffChangeLog(outputFile, config.getLiquibaseDiffTypes());
    }
}
