package fr.ifremer.adagio.core.dao.technical.hibernate;

/*
 * #%L
 * SIH-Adagio :: Shared
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2015 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */


import java.io.File;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;

import fr.ifremer.adagio.core.dao.technical.DateUtils;

/**
 * Collection of helper methods for dealing with configuration settings.
 * @see {@link org.hibernate.internal.util.config.ConfigurationHelper}
 * @author Benoit Lavenier <benoit.lavenier@e-is.pro>
 * @since 3.7.2
 *
 */
public class ConfigurationHelper {

    public static Integer getInteger(String key, Properties properties) {
        return org.hibernate.internal.util.config.ConfigurationHelper.getInteger(key, properties);
    }
    
    public static String getString(String key, Properties properties) {
        return org.hibernate.internal.util.config.ConfigurationHelper.getString(key, properties);
    }

    public static String getString(String key, Properties properties, String defaultValue) {
        return org.hibernate.internal.util.config.ConfigurationHelper.getString(key, properties, defaultValue);
    }

    public static Multimap<String, String> getMultimap(String key, Properties properties) {
        String property = getString(key, properties);
        return getMultimap(property);
    }
    
    public static long getLong(String key, Properties properties, int defaultValue) {
        return org.hibernate.internal.util.config.ConfigurationHelper.getLong(key, properties, defaultValue);
    }
    
    public static boolean getBoolean(String key, Properties properties, boolean defaultValue) {
        return org.hibernate.internal.util.config.ConfigurationHelper.getBoolean(key, properties, defaultValue);
    }
    
    public static int getInt(String key, Properties properties, int defaultValue) {
        return org.hibernate.internal.util.config.ConfigurationHelper.getInt(key, properties, defaultValue);
    }
    
    public static Multimap<String, String> getMultimap(String property) {
        if (StringUtils.isBlank(property)) {
            return null;
        }
        
        Multimap<String, String> result = ArrayListMultimap.create();
        
        Map<String, String> valuesStrByKey = Splitter
                .on(',')
                .trimResults()
                .withKeyValueSeparator(Splitter.on('|').trimResults())
                .split(property);
        for (String mapKey: valuesStrByKey.keySet()) {
            String valuesStr = valuesStrByKey.get(mapKey);
            List<String> valuesList = Splitter.on(';').trimResults().splitToList(valuesStr);

            // Add to result
            result.putAll(mapKey, valuesList);
        }

        return result;
    }
    
    public static void setMultimap(String key, Multimap<String, String> multimap, Properties properties) {
        if (multimap == null || multimap.isEmpty()) {
            properties.remove(key);
            return;
        }

        StringBuilder propertyBuilder = new StringBuilder();
        for (String mapKey : multimap.keySet()) {
            Collection<String> mapValues = multimap.get(mapKey);
            propertyBuilder.append(',')
                    .append(mapKey.toUpperCase())
                    .append('|');
            Joiner.on(';')
                    .appendTo(propertyBuilder, mapValues);
        }

        properties.setProperty(key, propertyBuilder.substring(1));
    }
    
    public static void setDate(String key, Date date, String pattern, boolean withTime, Properties properties) {
        if (date == null) {
            properties.remove(key);
            return;
        }
        if (!withTime) {
            date = DateUtils.truncate(date, Calendar.DAY_OF_MONTH);
        }
        String property = DateUtils.formatDate(date, pattern);
        properties.setProperty(key, property);
    }
    
    public static void setString(String key, String property, Properties properties) {
        if (property == null) {
            properties.remove(key);
            return;
        }
        properties.setProperty(key, property);
    }
    
    public static void setBoolean(String key, Boolean property, Properties properties) {
        if (property == null) {
            properties.remove(key);
            return;
        }
        properties.setProperty(key, property.toString());
    }
    
    public static void setFile(String key, File file, Properties properties) {
        if (file == null) {
            properties.remove(key);
            return;
        }
        properties.setProperty(key, file.getAbsolutePath());
    }


}
