package fr.ifremer.adagio.core.dao.technical.ehcache;

/*
 * #%L
 * SIH-Adagio Core Shared
 * $Id: EhCacheManagerFactoryBean.java 12431 2014-11-21 15:55:30Z bl05b3e $
 * $HeadURL: https://forge.ifremer.fr/svn/sih-adagio/tags/adagio-3.8.6.3/core-shared/src/main/java/fr/ifremer/adagio/core/dao/technical/ehcache/EhCacheManagerFactoryBean.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import net.sf.ehcache.CacheException;
import net.sf.ehcache.CacheManager;
import net.sf.ehcache.config.Configuration;
import net.sf.ehcache.config.ConfigurationFactory;
import net.sf.ehcache.config.DiskStoreConfiguration;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.core.io.Resource;

import java.io.File;
import java.io.IOException;

/**
 * This class as been writing using this post :
 * http://stackoverflow.com/questions/1533409/spring-context-property-placholder-ehcahe-configuration
 * Its aims to manage the EhCache diskstore path in a config file.
 * <p/>
 * <p/>
 * {@link FactoryBean} that exposes an EHCache {@link CacheManager}
 * instance (independent or shared), configured from a specified config location.
 * <p/>
 * <p>If no config location is specified, a CacheManager will be configured from
 * "ehcache.xml" in the root of the class path (that is, default EHCache initialization
 * - as defined in the EHCache docs - will apply).
 * <p/>
 * <p>Setting up a separate EhCacheManagerFactoryBean is also advisable when using
 * EhCacheFactoryBean, as it provides a (by default) independent CacheManager instance
 * and cares for proper shutdown of the CacheManager. EhCacheManagerFactoryBean is
 * also necessary for loading EHCache configuration from a non-default config location.
 * <p/>
 * <p>Note: As of Spring 3.0, Spring's EHCache support requires EHCache 1.3 or higher.
 *
 * @author Dmitriy Kopylenko
 * @author Juergen Hoeller
 * @author Benoit Lavenier
 * @author Tony CHEMIT
 * @see #setConfigLocation
 * @see #setShared
 * @see CacheManager
 * @since 3.1.2
 */
public class EhCacheManagerFactoryBean implements FactoryBean<CacheManager>, InitializingBean, DisposableBean {
    protected final Log logger = LogFactory.getLog(getClass());

    private String diskStorePath = null;

    private Resource configLocation;

    private boolean shared = false;

    private String cacheManagerName;

    private CacheManager cacheManager;

    public String getDiskStorePath() {
        return diskStorePath;
    }

    public void setDiskStorePath(String diskStorePath) {
        this.diskStorePath = diskStorePath;
    }

    /**
     * Set the location of the EHCache config file. A typical value is "/WEB-INF/ehcache.xml".
     * <p>Default is "ehcache.xml" in the root of the class path, or if not found,
     * "ehcache-failsafe.xml" in the EHCache jar (default EHCache initialization).
     *
     * @see CacheManager#create(java.io.InputStream)
     * @see CacheManager#CacheManager(java.io.InputStream)
     */
    public void setConfigLocation(Resource configLocation) {
        this.configLocation = configLocation;
    }

    /**
     * Set whether the EHCache CacheManager should be shared (as a singleton at the VM level)
     * or independent (typically local within the application). Default is "false", creating
     * an independent instance.
     *
     * @see CacheManager#create()
     * @see CacheManager#CacheManager()
     */
    public void setShared(boolean shared) {
        this.shared = shared;
    }

    /**
     * Set the name of the EHCache CacheManager (if a specific name is desired).
     *
     * @see CacheManager#setName(String)
     */
    public void setCacheManagerName(String cacheManagerName) {
        this.cacheManagerName = cacheManagerName;
    }

    @Override
    public void afterPropertiesSet() throws IOException, CacheException {

        Configuration config;

        if (this.configLocation != null) {
            config = ConfigurationFactory.parseConfiguration(this.configLocation.getInputStream());
        } else {
            config = ConfigurationFactory.parseConfiguration();
        }

        DiskStoreConfiguration diskStoreConfiguration = config.getDiskStoreConfiguration();
        boolean noDiskStoreConfiguration = diskStoreConfiguration == null;
        if (noDiskStoreConfiguration) {
            diskStoreConfiguration = new DiskStoreConfiguration();
        }
        
        File diskStore = new File(getDiskStorePath());
        if (!diskStore.isDirectory() || !diskStore.exists()) {
	        logger.info("Using EHCache disk store at " + getDiskStorePath() + " - directory not exists yet");
        }
        else if (logger.isDebugEnabled()) {
	        logger.debug("Using EHCache disk store at " + getDiskStorePath() + " (replace old config: " + diskStoreConfiguration.getOriginalPath()
                    + ")");	
        }
        
        diskStoreConfiguration.setPath(getDiskStorePath());
        if (noDiskStoreConfiguration) {
            config.addDiskStore(diskStoreConfiguration);
        }
        if (this.cacheManagerName != null) {
            config.setName(this.cacheManagerName);
        }
        if (this.shared) {
            this.cacheManager = CacheManager.create(config);
        } else {
            this.cacheManager = new CacheManager(config);
        }
    }

    @Override
    public CacheManager getObject() {
        return this.cacheManager;
    }

    @Override
    public Class<? extends CacheManager> getObjectType() {
        return (this.cacheManager != null ? this.cacheManager.getClass() : CacheManager.class);
    }

    @Override
    public boolean isSingleton() {
        return true;
    }

    @Override
    public void destroy() {
        logger.debug("Shutting down EHCache CacheManager");
        this.cacheManager.shutdown();
    }
}
