package fr.ifremer.adagio.core.dao.technical;

/*
 * #%L
 * SIH-Adagio :: Shared
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2015 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */


import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.version.Version;
import org.nuiton.util.version.Versions;

import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;

import fr.ifremer.adagio.core.AdagioTechnicalException;
import java.text.SimpleDateFormat;

public class DateVersions {

    protected DateVersions() {
        // Helper class
    }

    private static final Log log = LogFactory.getLog(DateVersions.class);

    /**
     * convert a version to a date
     *
     * ex: version = 2014.9.22.115930 will return 2014-09-22 11:59:30
     *
     * @param version
     * @return
     */
    public static Date convertVersion2Date(Version version) {
        Preconditions.checkNotNull(version);
        return convertVersion2Date(version.toString());
    }

    /**
     * convert a version to a date
     *
     * ex: version = 2014.9.22.115930 will return 2014-09-22 11:59:30
     *
     * @param version
     * @return
     */
    public static Date convertVersion2Date(String version) {
        Preconditions.checkNotNull(version);

        String[] versionParts = Splitter.on('.').splitToList(version).toArray(new String[4]);

        Calendar result = new GregorianCalendar();
        boolean parseError = false;
        try {
            // year 
            int year = Integer.parseInt(versionParts[0]);
            result.set(Calendar.YEAR, year);

            // month 
            int month = Integer.parseInt(versionParts[1]);
            result.set(Calendar.MONTH, month - 1);

            // day 
            int day = Integer.parseInt(versionParts[2]);
            result.set(Calendar.DAY_OF_MONTH, day);

            if (StringUtils.isBlank(versionParts[3])) {
                result.set(Calendar.SECOND, 0);
                result.set(Calendar.MINUTE, 0);
                result.set(Calendar.HOUR_OF_DAY, 0);
            }
            else {
                String time = versionParts[3];
                if (time.length() > 6) {
                    parseError = true;
                }
                else {
                    time = StringUtils.leftPad(time, 6, '0');

                    // seconds
                    int seconds = Integer.parseInt(time.substring(time.length() - 2));
                    result.set(Calendar.SECOND, seconds);

                    // minutes
                    int minutes = Integer.parseInt(time.substring(time.length() - 4, time.length() - 2));
                    result.set(Calendar.MINUTE, minutes);

                    // hours
                    int hours = Integer.parseInt(time.substring(0, time.length() - 4));
                    result.set(Calendar.HOUR_OF_DAY, hours);
                }
            }

            // no milliseconds
            result.set(Calendar.MILLISECOND, 0);

        } catch (Exception ex) {
            parseError = true;
        }
        if (parseError) {
            throw new AdagioTechnicalException(String.format("the database version '%s' can't be converted to a synchronization date", version));
        }

        return result.getTime();
    }

    /**
     * convert a version to a date, with no parse error
     *
     * ex: version = 2014.9.22.115930 will return 2014-09-22 11:59:30
     *
     * @param version
     * @return the date, or null if parse error
     */
    public static Date safeConvertVersion2Date(String version) {
        if (StringUtils.isBlank(version)) {
            return null;
        }
        try {
            return convertVersion2Date(version);
        } catch (AdagioTechnicalException e) {
            log.error(e.getMessage(), e);
        }
        return null;
    }

    /**
     * Convert a date into Version, using format 'yyyy.MM.dd.HHmmss'
     * @param date a not null date
     * @return
     */
    public static Version convertDate2Version(Date date) {
        Preconditions.checkNotNull(date);
        
        String dateStr = new SimpleDateFormat("yyyy.MM.dd.HHmmss").format(date);
        Version version = Versions.valueOf(dateStr);
        return version;
    }
}
