package fr.ifremer.adagio.core.dao.technical;

import java.io.File;
import java.io.IOException;

import java.util.Properties;

/*
 * #%L
 * SIH-Adagio Core Shared
 * $Id: DatabaseSchemaDao.java 12594 2015-01-28 12:43:57Z bl05b3e $
 * $HeadURL: https://forge.ifremer.fr/svn/sih-adagio/tags/adagio-3.8.6.3/core-shared/src/main/java/fr/ifremer/adagio/core/dao/technical/DatabaseSchemaDao.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

public interface DatabaseSchemaDao {

    /**
     * Generate a file with all SQL for database creation
     *
     * @param filename full path of the file to generate
     */
    void generateCreateSchemaFile(String filename);

    /**
     * @param filename   The file to generate, or null if only execution is need
     * @param doExecute  The SQL script must be execute on database ?
     * @param withDrop   generate drop statement ?
     * @param withCreate generate create statement ?
     */
    void generateCreateSchemaFile(String filename, boolean doExecute, boolean withDrop, boolean withCreate);

    /**
     * Generate a file with update SQL statement.
     * SQL statements will NOT be executed on database
     *
     * @param filename
     */
    void generateUpdateSchemaFile(String filename);

    /**
     * Generate a file with update SQL statement, and/or execute it on database.
     *
     * @param filename The file to generate, or null if only execution on database is need
     * @param doUpdate true if execution is need on database
     */
    void generateUpdateSchemaFile(String filename, boolean doUpdate);

    /**
     * Execute all changes need on database schema
     *
     * @throws DatabaseSchemaUpdateException if could not update schema
     * @since 3.4.0
     */
    void updateSchema() throws DatabaseSchemaUpdateException;

    /**
     * Execute all changes need on database schema, from the given connection
     *
     * @param connectionProperties the connection properties. If null, will default (see config.getConnectionProperties())
     * @throws DatabaseSchemaUpdateException if could not update schema
     * @since 3.7.0
     */
    void updateSchema(Properties connectionProperties) throws DatabaseSchemaUpdateException;

    /**
     * Retrieve the schema version, from table SYSTEM_VERSION,
     *
     * @return The database version (i.e. '3.2.3' @see Version)
     * @throws VersionNotFoundException if the version could not be found, or has a bad format
     * @since 3.4.0
     */
    org.nuiton.util.version.Version getSchemaVersion() throws VersionNotFoundException;

    /**
     * Get the database schema version if updates is apply
     * (the version that the database should have if updateSchema() was called)
     *
     * @return The database version (i.e. '3.2.3' @see Version)
     * @since 3.4.0
     */
    org.nuiton.util.version.Version getSchemaVersionIfUpdate();

    /**
     * Check if a update schema if need
     * This is equivalent to : <code>getSchemaVersion().compareTo(getSchemaVersionIfUpdate()) >= 0</code>
     *
     * @return true if a update is need
     * @since 3.4.0
     */
    boolean shouldUpdateSchema() throws VersionNotFoundException;
    
    /**
     * Check if connection could be open.
     * If a validation query has been set in configuration, test it
     * @return if db is loaded
     */
    boolean isDbLoaded();
    
    /**
     * Check if db files exists. If not a database file, return true
     * @return if db files exists
     */
    boolean isDbExists();
    
    /**
     * Report into a file the liquibase status of database 
     */
    void generateStatusReport(File outputFile) throws IOException;
    
    /**
     * Generate a diff of database 
     * @param typesToControl a comma separated database object to check (i.e Table, View, Column...). If null, all types are checked
     */
    void generateDiffReport(File outputFile, String typesToControl) throws IOException;
    
    /**
     * Generate a diff change log  
     * @param typesToControl a comma separated database object to check (i.e Table, View, Column...). If null, all types are checked
     */
    void generateDiffChangeLog(File outputChangelogFile, String typesToControl) throws IOException;
    
    /**
     * Generate a new DB directory
     * @param dbDirectory
     * @param replaceIfExists
     */
    void generateNewDb(File dbDirectory, boolean replaceIfExists);

    /**
     * Generate a new DB directory with the specified script and connection properties
     * @param dbDirectory
     * @param replaceIfExists
     * @param scriptFile a file with the HSQLDB script (e.g. allegro.script), or null for default script file 
     * @param connectionProperties
     */
    void generateNewDb(File dbDirectory, boolean replaceIfExists, File scriptFile, Properties connectionProperties);
}
