package fr.ifremer.adagio.core.config;

/*
 * #%L
 * SIH-Adagio :: Core for Allegro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import static org.nuiton.i18n.I18n.t;

import java.io.File;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.config.ApplicationConfig;
import org.nuiton.config.ApplicationConfigHelper;
import org.nuiton.config.ApplicationConfigProvider;
import org.nuiton.config.ArgumentsParserException;
import org.nuiton.util.version.Version;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanInitializationException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.config.PropertyPlaceholderConfigurer;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.core.io.Resource;

import com.google.common.base.Charsets;

import fr.ifremer.adagio.core.AdagioTechnicalException;
import fr.ifremer.adagio.core.dao.technical.DaoUtils;
import fr.ifremer.adagio.core.service.technical.SpringUtils;

public class AdagioConfiguration extends PropertyPlaceholderConfigurer implements ApplicationContextAware, InitializingBean {
    /** Logger. */
    private static final Log log = LogFactory.getLog(AdagioConfiguration.class);

    /**
     * Delegate application config.
     */
    protected final ApplicationConfig applicationConfig;

    private static AdagioConfiguration instance;

    public static AdagioConfiguration getInstance() {
        return instance;
    }

    public static void setInstance(AdagioConfiguration instance) {
        AdagioConfiguration.instance = instance;
    }

    protected final String[] optionKeyToNotSave;

    protected File configFile;

    private ApplicationContext appContext = null;

    private boolean isInitialize = false;

    public AdagioConfiguration(ApplicationConfig applicationConfig) {
        super();
        this.applicationConfig = applicationConfig;
        this.optionKeyToNotSave = null;
        
        // Disable batch processing if need
        boolean isJdbcBatchEnable = applicationConfig.getOptionAsBoolean(AdagioConfigurationOption.JDBC_BATCH_ENABLE.getKey());
        if (!isJdbcBatchEnable) {
            // Disable Hibernate batch update
            applicationConfig.setOption(AdagioConfigurationOption.JDBC_BATCH_SIZE.getKey(), "0");
        }
    }

    public AdagioConfiguration(String file, String... args) {
        super();
        this.applicationConfig = new ApplicationConfig();
        this.applicationConfig.setEncoding(Charsets.UTF_8.name());
        this.applicationConfig.setConfigFileName(file);

        // get all config providers
        Set<ApplicationConfigProvider> providers =
                ApplicationConfigHelper.getProviders(null,
                        null,
                        null,
                        true);

        // load all default options
        ApplicationConfigHelper.loadAllDefaultOption(applicationConfig,
                providers);

        // Load actions
        for (ApplicationConfigProvider provider : providers) {
            applicationConfig.loadActions(provider.getActions());
        }
        
        // Define Alias
        addAlias(applicationConfig);

        // get all transient and final option keys
        Set<String> optionToSkip =
                ApplicationConfigHelper.getTransientOptionKeys(providers);

        if (log.isDebugEnabled()) {
            log.debug("Option that won't be saved: " + optionToSkip);
        }
        optionKeyToNotSave = optionToSkip.toArray(new String[optionToSkip.size()]);

        try {
            applicationConfig.parse(args);

        } catch (ArgumentsParserException e) {
            throw new AdagioTechnicalException(t("adagio.config.parse.error"), e);
        }

        // TODO Review this, this is very dirty to do this...
        File allegroBasedir = applicationConfig.getOptionAsFile(
                AdagioConfigurationOption.BASEDIR.getKey());

        if (allegroBasedir == null) {
            allegroBasedir = new File("");
        }
        if (!allegroBasedir.isAbsolute()) {
            allegroBasedir = new File(allegroBasedir.getAbsolutePath());
        }
        if (allegroBasedir.getName().equals("..")) {
            allegroBasedir = allegroBasedir.getParentFile().getParentFile();
        }
        if (allegroBasedir.getName().equals(".")) {
            allegroBasedir = allegroBasedir.getParentFile();
        }
        if (log.isInfoEnabled()) {
            log.info("Application basedir: " + allegroBasedir);
        }
        applicationConfig.setOption(
                AdagioConfigurationOption.BASEDIR.getKey(),
                allegroBasedir.getAbsolutePath());
        
        // Disable batch processing if need
        boolean isJdbcBatchEnable = applicationConfig.getOptionAsBoolean(AdagioConfigurationOption.JDBC_BATCH_ENABLE.getKey());
        if (!isJdbcBatchEnable) {
            // Disable Hibernate batch update
            applicationConfig.setOption(AdagioConfigurationOption.JDBC_BATCH_SIZE.getKey(), "0");
        }
    }

    /**
     * Add alias to the given ApplicationConfig. <p/>
     * This method could be override to add specific alias
     * 
     * @param applicationConfig
     */
    protected void addAlias(ApplicationConfig applicationConfig) {
        applicationConfig.addAlias("-u", "--option", AdagioConfigurationOption.JDBC_USERNAME.getKey());
        applicationConfig.addAlias("--user", "--option", AdagioConfigurationOption.JDBC_USERNAME.getKey());
        applicationConfig.addAlias("-p", "--option", AdagioConfigurationOption.JDBC_PASSWORD.getKey());
        applicationConfig.addAlias("--password", "--option", AdagioConfigurationOption.JDBC_PASSWORD.getKey());
        applicationConfig.addAlias("-db", "--option", AdagioConfigurationOption.JDBC_URL.getKey());
        applicationConfig.addAlias("--database", "--option", AdagioConfigurationOption.JDBC_URL.getKey());
        applicationConfig.addAlias("--output", "--option", AdagioConfigurationOption.LIQUIBASE_OUTPUT_FILE.getKey());
    }

    @Override
    public void setApplicationContext(ApplicationContext appContext) throws BeansException {
        this.appContext = appContext;
    }

    @Override
    public void afterPropertiesSet() throws Exception {

        if (isInitialize) {
            return;
        }
        isInitialize = true;

        // Retrieve paths from configuration
        String[] enumerationFilePaths = getEnumerationFilesPath();
        if (ArrayUtils.isEmpty(enumerationFilePaths)) {
            throw new BeanInitializationException(
                    String.format(
                            "No enumeration files path has been set in the configuration. This is need to initialize enumeration values. Please set the option [%s] in configuration file.",
                            AdagioConfigurationOption.DB_ENUMERATION_RESOURCE));
        }
        if (log.isDebugEnabled()) {
            log.debug(String.format("Getting enumeration files from path: %s", Arrays.toString(enumerationFilePaths)));
        }

        // For each path, retrieve corresponding resources
        List<Resource> resources = SpringUtils.getResourcesFromPaths(enumerationFilePaths, appContext, true);

        // Check if some files has been found
        if (CollectionUtils.isEmpty(resources)) {
            throw new BeanInitializationException(String.format("No enumeration files could be found from path %s",
                    Arrays.toString(enumerationFilePaths)));
        }
        if (log.isDebugEnabled()) {
            log.debug(String.format("%s enumeration files found from path: %s", resources.size(), Arrays.toString(enumerationFilePaths)));
        }

        // Set enumeration default values (from enumeration file)
        AdagioEnumerationHelper.reload(applicationConfig, resources);
    }

    public File getConfigFile() {
        if (configFile == null) {
            File dir = getBasedir();
            if (dir == null || !dir.exists()) {
                dir = new File(applicationConfig.getUserConfigDirectory());
            }
            configFile = new File(dir, applicationConfig.getConfigFileName());
        }
        return configFile;
    }

    /** @return {@link AdagioConfigurationOption#BASEDIR} value */
    public File getBasedir() {
        File result = applicationConfig.getOptionAsFile(AdagioConfigurationOption.BASEDIR.getKey());
        return result;
    }

    /**
     *  @return {@link AdagioConfigurationOption#DATA_DIRECTORY} value */
    public File getDataDirectory() {
        File result = applicationConfig.getOptionAsFile(AdagioConfigurationOption.DATA_DIRECTORY.getKey());
        return result;
    }

    public ApplicationConfig getApplicationConfig() {
        return applicationConfig;
    }

    @Override
    protected String resolvePlaceholder(String placeholder, Properties props) {
        if (applicationConfig == null) {
            throw new AdagioTechnicalException(
                    "AdagioConfiguration.applicationConfig must not be null. Please initialize AdagioConfiguration instance with a not null applicationConfig BEFORE starting Spring.");
        }

        // Try to resolve placeholder from application configuration
        String optionValue = applicationConfig.getOption(placeholder);
        if (optionValue != null) {
            return optionValue;
        }

        // If not found in configuration, delegate to the default Spring mecanism
        return super.resolvePlaceholder(placeholder, props);
    }

    public File getTempDirectory() {
        return applicationConfig.getOptionAsFile(AdagioConfigurationOption.TMP_DIRECTORY.getKey());
    }
    
    public File getSynchroDirectory() {
        return applicationConfig.getOptionAsFile(AdagioConfigurationOption.SYNCHRO_DIRECTORY.getKey());
    }

    public File getSynchroExportDirectoryByUser(int userId) {
        File result = new File(
                getSynchroDirectory(),
                new StringBuilder()
                        .append(userId)
                        .append(File.separator)
                        .append("export")
                        .toString());
        return result;
    }

    public File getSynchroImportDirectoryByUser(int userId) {
        File result = new File(
                getSynchroDirectory(),
                new StringBuilder()
                        .append(userId)
                        .append(File.separator)
                        .append("import")
                        .toString());
        return result;
    }

    public void setSynchroDirectory(File synchroDirectory) {
        applicationConfig.setOption(AdagioConfigurationOption.SYNCHRO_DIRECTORY.getKey(), synchroDirectory.getPath());
    }

    public File getDbDirectory() {
        return applicationConfig.getOptionAsFile(AdagioConfigurationOption.DB_DIRECTORY.getKey());
    }

    public void setDbDirectory(File dbDirectory) {
        applicationConfig.setOption(AdagioConfigurationOption.DB_DIRECTORY.getKey(), dbDirectory.getPath());
    }

    public File getDbAttachmentDirectory() {
        return applicationConfig.getOptionAsFile(AdagioConfigurationOption.DB_ATTACHMENT_DIRECTORY.getKey());
    }

    public File getCacheDirectory() {
        return applicationConfig.getOptionAsFile(AdagioConfigurationOption.DB_CACHE_DIRECTORY.getKey());
    }

    public File getDbBackupDirectory() {
        return applicationConfig.getOptionAsFile(AdagioConfigurationOption.DB_BACKUP_DIRECTORY.getKey());
    }

    public boolean useBacthTreeCache() {
        return applicationConfig.getOptionAsBoolean(AdagioConfigurationOption.CACHE_BACTH_TREE.getKey());
    }

    public boolean useLiquibaseAutoRun() {
        return applicationConfig.getOptionAsBoolean(AdagioConfigurationOption.LIQUIBASE_RUN_AUTO.getKey());
    }

    public String getLiquibaseChangeLogPath() {
        return applicationConfig.getOption(AdagioConfigurationOption.LIQUIBASE_CHANGE_LOG_PATH.getKey());
    }

    public String getDbCreateScriptPath() {
        return applicationConfig.getOption(AdagioConfigurationOption.DB_CREATE_SCRIPT_PATH.getKey());
    }

    public String getHibernateDialect() {
        return applicationConfig.getOption(AdagioConfigurationOption.HIBERNATE_DIALECT.getKey());
    }

    public String getHibernateClientQueriesFile() {
        return applicationConfig.getOption(AdagioConfigurationOption.HIBERNATE_CLIENT_QUERIES_FILE.getKey());
    }

    public String getJdbcDriver() {
        return applicationConfig.getOption(AdagioConfigurationOption.JDBC_DRIVER.getKey());
    }

    public String getJdbcURL() {
        return applicationConfig.getOption(AdagioConfigurationOption.JDBC_URL.getKey());
    }

    public String getJdbcCatalog() {
        return applicationConfig.getOption(AdagioConfigurationOption.JDBC_SCHEMA.getKey());
    }

    public String getJdbcSchema() {
        return applicationConfig.getOption(AdagioConfigurationOption.JDBC_SCHEMA.getKey());
    }

    public boolean debugEntityLoad() {
        return applicationConfig.getOptionAsBoolean(AdagioConfigurationOption.DEBUG_ENTITY_LOAD.getKey());
    }

    public String[] getEnumerationFilesPath() {
        String enumerationFilesPath = applicationConfig.getOption(AdagioConfigurationOption.DB_ENUMERATION_RESOURCE.getKey());
        String[] enumerationFilesPathArray = enumerationFilesPath.split(",");
        return enumerationFilesPathArray;
    }

    public String getDbName() {
        return applicationConfig.getOption(AdagioConfigurationOption.DB_NAME.getKey());
    }

    public String getDbValidationQuery() {
        return applicationConfig.getOption(AdagioConfigurationOption.DB_VALIDATION_QUERY.getKey());
    }

    public boolean isDbCheckConstantsEnable() {
        return applicationConfig.getOptionAsBoolean(AdagioConfigurationOption.DB_CHECK_CONSTANTS.getKey());
    }

    public String getJdbcUsername() {
        return applicationConfig.getOption(AdagioConfigurationOption.JDBC_USERNAME.getKey());
    }

    public String getJdbcPassword() {
        return applicationConfig.getOption(AdagioConfigurationOption.JDBC_PASSWORD.getKey());
    }

    public boolean isJdbcBatchEnable() {
        return applicationConfig.getOptionAsBoolean(AdagioConfigurationOption.JDBC_BATCH_ENABLE.getKey());
    }

    public int getJdbcBatchSize() {
        if (!isJdbcBatchEnable()) {
            return 0; // 0 = disable Hibernate batch processing (better error message)
        }
        int jdbcBatchSize = applicationConfig.getOptionAsInt(AdagioConfigurationOption.JDBC_BATCH_SIZE.getKey());

        // If negative value, use the default value
        if (jdbcBatchSize < 0) {
            jdbcBatchSize = Integer.parseInt(AdagioConfigurationOption.JDBC_BATCH_SIZE.getDefaultValue());
        }
        return jdbcBatchSize;
    }

    public String getStatusCodeTemporary() {
        return applicationConfig.getOption(AdagioConfigurationOption.STATUS_CODE_TEMPORARY.getKey());
    }

    public String getStatusCodeValid() {
        return applicationConfig.getOption(AdagioConfigurationOption.STATUS_CODE_ENABLE.getKey());
    }    

    public String getImportTranscribingItemTypeIds() {
        return applicationConfig.getOption(AdagioConfigurationOption.IMPORT_TRANSCRIBING_ITEM_TYPE_IDS.getKey());
    }

    public String getImportGroupingClassificationIds() {
        return applicationConfig.getOption(AdagioConfigurationOption.IMPORT_GROUPING_CLASSIFICATION_IDS.getKey());
    }

    public String getImportProgramCodes() {
        return applicationConfig.getOption(AdagioConfigurationOption.IMPORT_PROGRAM_CODES.getKey());
    }
    
    public String getImportDataPkIncludes() {
        return applicationConfig.getOption(AdagioConfigurationOption.IMPORT_DATA_PK_INCLUDES.getKey());
    }
    
    public int getImportNbYearDataHistory() {
        return applicationConfig.getOptionAsInt(AdagioConfigurationOption.IMPORT_NB_YEARS_DATA_HISTORY.getKey());
    }    

    public int getExportDataUpdateDateDelayInSecond() {
        return applicationConfig.getOptionAsInt(AdagioConfigurationOption.EXPORT_DATA_UPDATE_DATE_DELAY.getKey());
    }
    
    public int getImportReferentialUpdateDateOffsetInSecond() {
        return applicationConfig.getOptionAsInt(AdagioConfigurationOption.IMPORT_REFERENTIAL_UPDATE_DATE_OFFSET.getKey());
    }

    public Version getVersion() {
        return applicationConfig.getOptionAsVersion(AdagioConfigurationOption.VERSION.getKey());
    }

    public String getVesselRegistryProgramCode() {
        return applicationConfig.getOption(AdagioConfigurationOption.PROGRAM_CODE_VESSEL_REGISTRY.getKey());
    }

    public File getI18nDirectory() {
        return applicationConfig.getOptionAsFile(
                AdagioConfigurationOption.I18N_DIRECTORY.getKey());
    }

    public Locale getI18nLocale() {
        return applicationConfig.getOptionAsLocale(
                AdagioConfigurationOption.I18N_LOCALE.getKey());
    }

    public void setI18nLocale(Locale locale) {
        applicationConfig.setOption(AdagioConfigurationOption.I18N_LOCALE.getKey(), locale.toString());
    }

    public Properties getConnectionProperties() {
        return DaoUtils.getConnectionProperties(
                getJdbcURL(),
                getJdbcUsername(),
                getJdbcPassword(),
                null,
                getHibernateDialect(),
                getJdbcDriver());
    }

    public String getLiquibaseDiffTypes() {
        return applicationConfig.getOption(AdagioConfigurationOption.LIQUIBASE_DIFF_TYPES.getKey());
    }

    public File getLiquibaseOutputFile() {
        return applicationConfig.getOptionAsFile(AdagioConfigurationOption.LIQUIBASE_OUTPUT_FILE.getKey());
    }
}
