package fr.ifremer.adagio.core.service.referential.synchro.intercept;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import com.google.common.eventbus.Subscribe;

import fr.ifremer.adagio.core.config.AdagioConfiguration;
import fr.ifremer.adagio.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata;
import fr.ifremer.adagio.synchro.meta.event.CreateQueryEvent;
import fr.ifremer.adagio.synchro.query.SynchroQueryBuilder;
import fr.ifremer.adagio.synchro.query.SynchroQueryOperator;

/**
 * Filter to keep on rows on the default SIH program, and avoid to get all rows
 * 
 * @author Benoit Lavenier <benoit.lavenier@e-is.pro>
 * @since 3.7.0
 * 
 */
public class VesselTablesQueriesInterceptor extends ReferentialAbstractSynchroInterceptor {

	private static final Log log = LogFactory.getLog(VesselTablesQueriesInterceptor.class);

	public static final String COLUMN_PROGRAM_FK = "program_fk";
	public static final String COLUMN_VESSEL_FK = "vessel_fk";

	@Override
	public boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table) {
		String tableName = table.getName().toUpperCase();

		return "VESSEL".equals(tableName)
				|| "VESSEL_FEATURES".equals(tableName)
				|| "VESSEL_REGISTRATION_PERIOD".equals(tableName)
				|| "VESSEL_OWNER_PERIOD".equals(tableName)
				|| "VESSEL_OWNER".equals(tableName);
	}

	@Subscribe
	public void handleCreateQuery(CreateQueryEvent e) {
		switch (e.queryName) {
		case count:
		case countFromUpdateDate:
		case select:
		case selectFromUpdateDate:
			// Add restriction on vessels
			e.sql = addVesselRestriction(e.source, e.sql);
			break;

		default:
			break;
		}
	}

	protected String addVesselRestriction(SynchroTableMetadata table, String sql) {

		boolean hasVesselColumn = hasColumns(table, COLUMN_VESSEL_FK);
		boolean hasProgramColumn = hasColumns(table, COLUMN_PROGRAM_FK);

		SynchroQueryBuilder query = SynchroQueryBuilder.newBuilder(sql);

		String programCode = AdagioConfiguration.getInstance().getVesselRegistryProgramCode();

		// from
		if (hasVesselColumn) {
			query.addJoin(String.format(" INNER JOIN VESSEL v ON v.code=t.%s AND v.%s='%s'",
					COLUMN_VESSEL_FK,
					COLUMN_PROGRAM_FK,
					programCode
					));
		}

		// where
		if (!hasVesselColumn && hasProgramColumn) {

			query.addWhere(SynchroQueryOperator.AND,
					String.format(" t.%s='%s'",
							COLUMN_PROGRAM_FK,
							programCode
							));
		}

		String newSql = query.build();
		if (log.isTraceEnabled()) {
			log.trace("Replace default SQL with: " + newSql);
		}

		return newSql;
	}
}
