package fr.ifremer.adagio.core.service.referential.synchro.intercept;

/*
 * #%L
 * SIH-Adagio :: Core for Allegro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.sql.Timestamp;
import java.util.Collection;
import java.util.Set;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import com.google.common.eventbus.Subscribe;

import fr.ifremer.adagio.synchro.config.SynchroConfiguration;
import fr.ifremer.adagio.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.adagio.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroJoinMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata;
import fr.ifremer.adagio.synchro.meta.event.LoadJoinEvent;
import fr.ifremer.adagio.synchro.meta.event.LoadTableEvent;
import fr.ifremer.adagio.synchro.service.SynchroDatabaseConfiguration;

/**
 * Manage only data table with columns 'id' AND 'remote_id' :
 * <ul>
 * <li>Set as root only if the table as an object_type (OBJECT_TYPE is need for join with PERSON_SESION_VESSEL)
 * <li>when exporting to server, override 'update_date' column with a systimestamp
 * <p/>
 * 
 * 
 * @author Benoit Lavenier <benoit.lavenier@e-is.pro>
 * @since 3.6.3
 * 
 */
public class ReferentialTableInterceptor extends ReferentialAbstractSynchroInterceptor {

	private static final Log log = LogFactory.getLog(ReferentialTableInterceptor.class);

	public static final String COLUMN_UPDATE_DATE = "update_date";

	protected static Multimap<String, String> childJoinIncludes;
	protected static Multimap<String, String> childJoinExcludes;
	static {
		childJoinIncludes = initChildJoinIncludes();
		childJoinExcludes = initChildJoinExcludes();
	}

	private Set<String> referentialTableIncludes;
	private Timestamp systimestamp = null;

	@Override
	protected void init(SynchroDatabaseConfiguration config) {
		super.init(config);
		referentialTableIncludes = SynchroConfiguration.getInstance().getImportReferentialTablesIncludes();
	}

	@Override
	public boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table) {
		if (CollectionUtils.isEmpty(referentialTableIncludes)) {
			return false;
		}

		boolean isReferentialTable = referentialTableIncludes.contains(table.getName());

		return isReferentialTable;
	}

	@Override
	public SynchroInterceptorBase clone() {
		ReferentialTableInterceptor result = (ReferentialTableInterceptor) super.clone();
		result.referentialTableIncludes = this.referentialTableIncludes;
		result.systimestamp = this.systimestamp;
		return result;
	}

	@Subscribe
	public void handleTableLoad(LoadTableEvent e) {

		SynchroTableMetadata table = e.table;

		// Set table as root if synchronization status column exists
		boolean hasUpdateDate = hasColumns(table, COLUMN_UPDATE_DATE);

		// Set isRoot :
		// - if not already done - e.g. DeletedItemHistoryInterceptor already set as true,
		// so do not override this to false !
		// - is has an 'update_date' column
		if (!table.isRoot() && hasUpdateDate) {
			table.setRoot(true);
			return;
		}

		// Set STATUS as root even if no update_date
		if ("STATUS".equalsIgnoreCase(table.getName())) {
			table.setRoot(true);
		}
	}

	@Subscribe
	public void handleJoinLoad(LoadJoinEvent e) {
		SynchroJoinMetadata join = e.join;

		String fkTableName = join.getFkTable().getName().toUpperCase();
		String fkColumnName = join.getFkColumn().getName().toUpperCase();

		// Disable all join to a root table
		if (join.isChild() && hasColumns(join.getFkTable(), COLUMN_UPDATE_DATE)) {
			if (log.isDebugEnabled()) {
				log.debug(String.format("Disabling %s: child table is a root table", join.toString()));
			}
			join.setIsValid(false);
			return;
		}

		// Disable link to itself
		if (join.getFkTable() == join.getPkTable()) {
			if (log.isDebugEnabled()) {
				log.debug(String.format("Disabling %s: recursive link", join.toString()));
			}
			join.setIsValid(false);
			return;
		}

		// Keep a child join if its includes
		if (join.isChild()) {
			// Is join explicitly include ?
			Collection<String> columnIncludes = childJoinIncludes.get(fkTableName);
			// Tables that are NOT listed will have all children joins exclude.
			boolean isColumnInclude = CollectionUtils.isNotEmpty(columnIncludes) && columnIncludes.contains(fkColumnName);

			// Is join explicitly exclude ?
			Collection<String> columnExcludes = childJoinExcludes.get(fkTableName);
			boolean isColumnExclude = CollectionUtils.isNotEmpty(columnExcludes) && columnExcludes.contains(fkColumnName);

			if (!isColumnInclude || isColumnExclude) {
				if (log.isDebugEnabled()) {
					log.debug(String.format("Disabling %s: include=%s, exclude=%s",
							join.toString(),
							isColumnInclude,
							isColumnExclude));
				}
				join.setIsValid(false);
			}
		}
	}

	/* -- Internal methods -- */

	/**
	 * Only this child link will be kept (other are invalidate).<br/>
	 * Tables that are NOT listed here will have all children joins <u>exclude</u>.
	 * 
	 * @return
	 */
	protected static Multimap<String, String> initChildJoinIncludes() {
		Multimap<String, String> result = ArrayListMultimap.create();

		//
		result.put("APPLIED_STRATEGY", "STRATEGY_FK");
		result.put("APPLIED_PERIOD", "APPLIED_STRATEGY_FK");
		result.put("DEPARTMENT_HIERARCHY", "PARENT_DEPARTMENT_FK");
		result.put("FRACTION2MATRIX", "FRACTION_FK");
		result.put("GROUPING_ITEM", "GROUPING_FK");
		result.put("GROUPING_ITEM_HIERARCHY", "GROUPING_FK");
		result.put("GROUPING_VESSEL_HIERARCHY", "GROUPING_FK");
		result.put("LOCATION_ASSOCIATION", "PARENT_LOCATION_FK");
		result.put("GEAR_ASSOCIATION", "FROM_GEAR_FK");
		result.put("LOCATION_HIERARCHY", "PARENT_LOCATION_FK");
		result.put("LOCATION_HIERARCHY_EXCEPTION", "PARENT_LOCATION_FK");
		result.put("LOCATION_HIERARCHY_OVERRIDE", "PARENT_LOCATION_FK");
		result.put("QUALITATIVE_VALUE", "PARAMETER_FK");
		result.put("PERSON2USER_PROFIL", "PERSON_FK");
		result.put("PROGRAM2DEPARTMENT", "PROGRAM_FK");
		result.put("PROGRAM2LOCATION", "PROGRAM_FK");
		result.put("PROGRAM2LOCATION_CLASSIF", "PROGRAM_FK");
		result.put("PROGRAM2PERSON", "PROGRAM_FK");
		result.put("PROGRAM2PERSON_EXCEPTION", "PROGRAM2PERSON_FK");
		result.put("PMFM_STRATEGY", "STRATEGY_FK");
		result.put("PMFM_APPLIED_STRATEGY", "APPLIED_STRATEGY_FK");
		result.put("PMFM2QUALITATIVE_VALUE", "PMFM_FK");
		result.put("REFERENCE_DOCUMENT2AUTHOR", "REFERENCE_DOCUMENT_FK");
		result.put("REFERENCE_TAXON_STRATEGY", "STRATEGY_FK");
		result.put("STRATEGY2GEAR", "STRATEGY_FK");
		result.put("STRATEGY2MANAGER_PERSON", "STRATEGY_FK");
		result.put("SPATIAL_ITEM2LOCATION", "SPATIAL_ITEM_FK");
		result.put("SPATIAL_ITEM2LOCATION", "LOCATION_FK"); // mantis #23383 (add a link, to enable delete cascade from
															// LOCATION)
		result.put("TAXON_GROUP2TAXON_HIERARCHY", "TAXON_GROUP_FK");
		result.put("TAXON_GROUP_HIERARCHY", "CHILD_TAXON_GROUP_FK");
		result.put("VESSEL_REGISTRATION_PERIOD", "VESSEL_FK");
		result.put("VESSEL_OWNER_PERIOD", "VESSEL_FK");
		result.put("TAXON_INFORMATION_HISTORY", "DOCUMENT_REFERENCE_FK");
		// mantis #23381 (note 68762) - ignoring UPDATE_DATE and use TAXON_GROUP as parent
		result.put("TAXON_GROUP_HISTORICAL_RECORD", "TAXON_GROUP_FK");
		result.put("TAXON_GROUP_INFORMATION", "TAXON_GROUP_FK");
		result.put("GEAR_CLASSIFICATION_ASSOCIATIO", "FROM_GEAR_CLASSIFICATION_FK");

		return result;
	}

	protected static Multimap<String, String> initChildJoinExcludes() {
		Multimap<String, String> result = ArrayListMultimap.create();

		// this child link will be excluded
		result.put("TAXON_GROUP", "PARENT_TAXON_GROUP_FK");
		// no need to follow recursive links (since integrity constraints are disabled for referentials)
		result.put("DEPARTMENT", "PARENT_DEPARTMENT_FK");

		return result;
	}
}
