package fr.ifremer.adagio.core.service.referential.synchro.intercept;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.util.Set;

import org.apache.commons.collections4.CollectionUtils;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import com.google.common.base.Joiner;
import com.google.common.collect.Sets;
import com.google.common.eventbus.Subscribe;

import fr.ifremer.adagio.core.service.data.synchro.intercept.ObjectTypeHelper;
import fr.ifremer.adagio.synchro.config.SynchroConfiguration;
import fr.ifremer.adagio.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.adagio.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.adagio.synchro.meta.event.CreateQueryEvent;
import fr.ifremer.adagio.synchro.query.SynchroQueryBuilder;
import fr.ifremer.adagio.synchro.query.SynchroQueryOperator;
import fr.ifremer.adagio.synchro.service.SynchroDatabaseConfiguration;

/**
 * Import DELETED_ITEM_HISTORY rows, limited to referential tables
 * 
 * @author Benoit Lavenier <benoit.lavenier@e-is.pro>
 * @since 3.7.0
 * 
 */
public class DeletedItemHistoryInterceptor extends ReferentialAbstractSynchroInterceptor {

	private String whereClauseOnTableIncludes = null;

	@Override
	public SynchroInterceptorBase clone() {
		DeletedItemHistoryInterceptor newBean = (DeletedItemHistoryInterceptor) super.clone();
		newBean.whereClauseOnTableIncludes = this.whereClauseOnTableIncludes;
		return newBean;
	}

	@Override
	protected void init(SynchroDatabaseConfiguration config) {
		super.init(config);
		whereClauseOnTableIncludes = createWhereClauseOnTableIncludes();
	}

	@Override
	public boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table) {

		return "DELETED_ITEM_HISTORY".equalsIgnoreCase(table.getName());
	}

	@Subscribe
	public void handleCreateQuery(CreateQueryEvent e) {
		switch (e.queryName) {
		case count:
		case countFromUpdateDate:
		case select:
		case selectFromUpdateDate:
		case selectMaxUpdateDate:
			e.sql = SynchroQueryBuilder.newBuilder(e.sql)
					.addWhere(SynchroQueryOperator.AND, whereClauseOnTableIncludes)
					.build();
			break;

		default:
			break;
		}
	}

	private String createWhereClauseOnTableIncludes() {
		Set<String> tableToIncludes = SynchroConfiguration.getInstance().getImportReferentialTablesIncludes();

		// If no tables configured : should return no lines
		if (CollectionUtils.isEmpty(tableToIncludes)) {
			return "1=2";
		}

		Set<String> allObjectTypes = Sets.newHashSet();
		for (String tableName : tableToIncludes) {
			Set<String> objectTypes = ObjectTypeHelper.getObjectTypeFromTableName(tableName);
			if (CollectionUtils.isNotEmpty(objectTypes)) {
				allObjectTypes.addAll(objectTypes);
			}
			else {
				// By default, add the table name itself (i.e. PERSON_SESSION_VESSEL)
				allObjectTypes.add(tableName);
			}
		}
		return new StringBuilder()
				.append("OBJECT_TYPE_FK IN ('")
				.append(Joiner.on("','").join(allObjectTypes))
				.append("')")
				.toString();
	}

}
