package fr.ifremer.adagio.core.service.referential.synchro;

/*
 * #%L
 * SIH-Adagio :: Core for Allegro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.io.File;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Collection;
import java.util.Deque;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.annotation.Nullable;
import javax.sql.DataSource;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.I18n;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Queues;
import com.google.common.collect.Sets;

import fr.ifremer.adagio.core.dao.technical.hibernate.TemporaryDataHelper;
import fr.ifremer.adagio.core.service.data.synchro.intercept.ObjectTypeHelper;
import fr.ifremer.adagio.synchro.SynchroTechnicalException;
import fr.ifremer.adagio.synchro.config.SynchroConfiguration;
import fr.ifremer.adagio.synchro.dao.DaoFactoryImpl;
import fr.ifremer.adagio.synchro.dao.SynchroBaseDao;
import fr.ifremer.adagio.synchro.dao.SynchroTableDao;
import fr.ifremer.adagio.synchro.meta.SynchroColumnMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroMetadataUtils;
import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata;
import fr.ifremer.adagio.synchro.service.RejectedRowStatus;
import fr.ifremer.adagio.synchro.service.SynchroContext;
import fr.ifremer.adagio.synchro.service.SynchroDatabaseConfiguration;
import fr.ifremer.adagio.synchro.service.SynchroResult;
import fr.ifremer.adagio.synchro.service.SynchroTableOperation;

/**
 * A class defined as a Spring Bean
 * 
 * @author Benoit
 * 
 */
@Service("referentialSynchroService2")
@Lazy
public class ReferentialSynchroServiceImpl
		extends fr.ifremer.adagio.synchro.service.SynchroServiceImpl
		implements ReferentialSynchroService {

	private static final Log log =
			LogFactory.getLog(ReferentialSynchroServiceImpl.class);

	private static boolean DISABLE_INTEGRITY_CONSTRAINTS = true;
	private static boolean ALLOW_MISSING_OPTIONAL_COLUMN = true;
	private static boolean ALLOW_ADDITIONAL_MANDATORY_COLUMN_IN_SOURCE_SCHEMA = true;
	private static boolean KEEP_WHERE_CLAUSE_ON_QUERIES_BY_FKS = true;

	private static final String TABLE_DELETED_ITEM_HISTORY = "DELETED_ITEM_HISTORY";

	/**
	 * Constants need for insertion into table TEMP_QUERY_PARAMETER, for delete by comparison
	 */
	private static final String TQP_DELETE_BY_COMPARISION_PREFIX = "DELETE#";
	private static final int TQP_DEFAULT_PERSON_ID = -1;

	// do not use a too big cache size, for referential tables
	// because one table should be processed only once
	private static int DAO_CACHE_SIZE = 2;

	public ReferentialSynchroServiceImpl(DataSource dataSource, SynchroConfiguration config) {
		super(dataSource,
				config,
				DISABLE_INTEGRITY_CONSTRAINTS,
				ALLOW_MISSING_OPTIONAL_COLUMN,
				ALLOW_ADDITIONAL_MANDATORY_COLUMN_IN_SOURCE_SCHEMA,
				KEEP_WHERE_CLAUSE_ON_QUERIES_BY_FKS);
		setDaoCacheSize(DAO_CACHE_SIZE);
	}

	public ReferentialSynchroServiceImpl() {
		super(DISABLE_INTEGRITY_CONSTRAINTS,
				ALLOW_MISSING_OPTIONAL_COLUMN,
				ALLOW_ADDITIONAL_MANDATORY_COLUMN_IN_SOURCE_SCHEMA,
				KEEP_WHERE_CLAUSE_ON_QUERIES_BY_FKS);
		setDaoCacheSize(DAO_CACHE_SIZE);
	}

	@Override
	public SynchroContext createSynchroContext(File sourceDbDirectory, Timestamp lastSynchronizationDate) {
		SynchroContext context = super.createSynchroContext(sourceDbDirectory,
				config.getImportReferentialTablesIncludes());
		context.setLastSynchronizationDate(lastSynchronizationDate);
		return context;
	}

	@Override
	public SynchroContext createSynchroContext(Properties sourceConnectionProperties, Timestamp lastSynchronizationDate) {
		SynchroContext context = super.createSynchroContext(sourceConnectionProperties,
				config.getImportReferentialTablesIncludes());
		context.setLastSynchronizationDate(lastSynchronizationDate);
		return context;
	}

	@Override
	public void prepare(SynchroContext synchroContext) {

		SynchroDatabaseConfiguration target = synchroContext.getTarget();
		SynchroDatabaseConfiguration source = synchroContext.getSource();

		// Always ignore some columns (mantis #22629)
		source.addColumnExclude("LOCATION_ASSOCIATION", "update_date");
		target.addColumnExclude("LOCATION_ASSOCIATION", "update_date");
		source.addColumnExclude("GEAR_ASSOCIATION", "update_date");
		target.addColumnExclude("GEAR_ASSOCIATION", "update_date");
		source.addColumnExclude("TAXON_GROUP_HISTORICAL_RECORD", "update_date");
		target.addColumnExclude("TAXON_GROUP_HISTORICAL_RECORD", "update_date");
		source.addColumnExclude("TAXON_GROUP_INFORMATION", "update_date");
		target.addColumnExclude("TAXON_GROUP_INFORMATION", "update_date");
		source.addColumnExclude("GEAR_CLASSIFICATION_ASSOCIATIO", "update_date");
		target.addColumnExclude("GEAR_CLASSIFICATION_ASSOCIATIO", "update_date");

		super.prepare(synchroContext);
	}

	@Override
	protected void releaseSynch(Connection targetConnection, SynchroContext context) throws SQLException {
		super.releaseSynch(targetConnection, context);
	}

	@Override
	public void synchronize(SynchroContext context) {
		super.synchronize(context);

		// Log if reject exists
		SynchroResult result = context.getResult();
		if (!result.getRejectedRows().isEmpty()) {
			log.warn(I18n.t("adagio.synchro.synchronizeReferential.rejects", result.getRejectedRows().toString()));
		}
	}

	@Override
	public Timestamp getSourceLastUpdateDate(SynchroContext synchroContext) {
		return super.getSourceLastUpdateDate(synchroContext);
	}

	/* -- Internal methods -- */

	/*
	 * @Override
	 * protected void deleteRows(SynchroTableDao targetDao,
	 * List<List<Object>> pks,
	 * check
	 * SynchroResult result,
	 * Deque<SynchroTableOperation> pendingOperations)
	 * throws SQLException {
	 * 
	 * boolean allowSkipRow = targetDao.getCurrentOperation().isEnableProgress();
	 * 
	 * try {
	 * super.deleteRows(targetDao,
	 * pks,
	 * result,
	 * pendingOperations);
	 * }
	 * // mantis 23134 : if delete failed (e.g. data still linked), reject the row
	 * catch (SQLException sqle) {
	 * 
	 * 
	 * if (allowSkipRow) {
	 * rejectUnableToDeleteRow(
	 * targetDao.getTable().getName(),
	 * result,
	 * pks,
	 * sqle
	 * );
	 * }
	 * else {
	 * StringBuilder pksStr = new StringBuilder();
	 * for (List<Object> pk: pks) {
	 * String pkStr = SynchroTableMetadata.toPkStr(pk);
	 * pksStr.append(",").append(pkStr);
	 * }
	 * 
	 * throw new SynchroUnableToDeleteRowException(
	 * targetDao.getTable().getName(),
	 * pksStr.substring(1));
	 * }
	 * }
	 * }
	 */

	@Override
	protected List<SynchroTableOperation> getRootOperations(
			DaoFactoryImpl sourceDaoFactory,
			DaoFactoryImpl targetDaoFactory,
			SynchroDatabaseMetadata dbMeta,
			SynchroContext context) throws SQLException {
		List<SynchroTableOperation> result = Lists.newArrayList();

		// Add default operations
		Collection<SynchroTableOperation> defaultOperations = super.getRootOperations(sourceDaoFactory, targetDaoFactory, dbMeta, context);
		result.addAll(defaultOperations);

		// Add delete items history operation
		Collection<SynchroTableOperation> deletedItemOperations = getRootDeleteOperations(sourceDaoFactory, targetDaoFactory, dbMeta, context);
		result.addAll(deletedItemOperations);

		return result;
	}

	private Collection<SynchroTableOperation> getRootDeleteOperations(
			DaoFactoryImpl sourceDaoFactory,
			DaoFactoryImpl targetDaoFactory,
			SynchroDatabaseMetadata dbMeta,
			SynchroContext context) throws SQLException {
		Preconditions.checkArgument(dbMeta.getConfiguration().isFullMetadataEnable());

		Deque<SynchroTableOperation> result = Queues.newArrayDeque();

		SynchroTableDao dihSourceDao = sourceDaoFactory.getSourceDao(TABLE_DELETED_ITEM_HISTORY);
		Set<String> includedDataTables = config.getImportReferentialTablesIncludes();

		// Read rows of DELETED_ITEM_HISTORY (from the temp DB)
		Map<String, Object> bindings = createSelectBindingsForTable(context, TABLE_DELETED_ITEM_HISTORY);
		ResultSet dihResultSet = dihSourceDao.getData(bindings);

		List<Object> dihIdsToRemove = Lists.newArrayList();

		Set<String> tableNamesWithDelete = Sets.newHashSet();

		log.info(I18n.t("adagio.synchro.synchronizeReferential.deletedItems"));

		boolean doDelete = !context.getTarget().isMirrorDatabase();
		while (dihResultSet.next()) {

			String objectType = dihResultSet.getString("OBJECT_TYPE_FK");
			String tableName = ObjectTypeHelper.getTableNameFromObjectType(objectType);

			boolean isReferentialTable = StringUtils.isNotBlank(tableName)
					&& includedDataTables.contains(tableName.toUpperCase());

			if (isReferentialTable) {
				SynchroTableDao targetDao = targetDaoFactory.getSourceDao(tableName);
				SynchroTableMetadata table = targetDao.getTable();

				String objectCode = dihResultSet.getString("OBJECT_CODE");
				String objectId = dihResultSet.getString("OBJECT_ID");

				// Delete by a PK (id or code)
				if (StringUtils.isNotBlank(objectCode) || StringUtils.isNotBlank(objectId)) {

					if (doDelete) {
						List<Object> pk = StringUtils.isNotBlank(objectCode)
								? ImmutableList.<Object> of(objectCode)
								: ImmutableList.<Object> of(objectId);

						boolean hasChildTables = targetDao.getTable().hasChildJoins();

						SynchroTableOperation operation = new SynchroTableOperation(tableName, context);
						operation.setEnableProgress(true);
						result.add(operation);

						operation.addMissingDelete(pk);

						// If has children, add child deletion to result
						if (hasChildTables) {
							addDeleteChildrenToDeque(table, ImmutableList.of(pk), result, context);
						}
					}
				}

				// No id or code:
				// Should delete after a comparison between local's and remote's PKs
				else {
					tableNamesWithDelete.add(tableName);
				}
			}
		}

		if (CollectionUtils.isNotEmpty(dihIdsToRemove)) {
			SynchroTableOperation operation = new SynchroTableOperation(TABLE_DELETED_ITEM_HISTORY, context);
			operation.addChildrenToDeleteFromOneColumn(TABLE_DELETED_ITEM_HISTORY, "remote_id", dihIdsToRemove);
			result.add(operation);
		}

		if (CollectionUtils.isNotEmpty(tableNamesWithDelete)) {

			// If target is a a temp DB (e.g. Server DB -> Temp DB)
			if (!doDelete) {
				saveTablesWithDelete(tableNamesWithDelete,
						sourceDaoFactory,
						targetDaoFactory,
						dbMeta,
						context);
			}

			// If source is a temp DB (e.g. Temp DB -> Local DB)
			else if (doDelete && context.getSource().isMirrorDatabase()) {
				addDeletedItemsFromTables(
						tableNamesWithDelete,
						result,
						sourceDaoFactory,
						targetDaoFactory,
						dbMeta,
						context);
			}

			// Else (could be Server DB -> Local DB)
			// (e.g. for unit test, or when direct connection to server DB)
			else if (doDelete) {
				saveTablesWithDelete(tableNamesWithDelete,
						sourceDaoFactory,
						targetDaoFactory,
						dbMeta,
						context);

				addDeletedItemsFromTables(
						tableNamesWithDelete,
						result,
						targetDaoFactory, // workaround, to read existing PK from target and not source DB
						targetDaoFactory,
						dbMeta,
						context);
			}
		}

		return result;
	}

	protected void saveTablesWithDelete(
			Set<String> tableNames,
			DaoFactoryImpl sourceDaoFactory,
			DaoFactoryImpl targetDaoFactory,
			SynchroDatabaseMetadata dbMeta,
			SynchroContext context) throws SQLException {

		SynchroBaseDao targetBaseDao = targetDaoFactory.getDao();

		// Delete previous PKs for delete by comparison
		targetBaseDao.executeDeleteTempQueryParameter(TQP_DELETE_BY_COMPARISION_PREFIX + "%", true, TQP_DEFAULT_PERSON_ID);

		for (String tableName : tableNames) {

			SynchroTableDao sourceDao = sourceDaoFactory.getSourceDao(tableName);
			Set<String> pkStrs = sourceDao.getPksStr();

			targetBaseDao.executeInsertIntoTempQueryParameter(
					ImmutableList.<Object> copyOf(pkStrs),
					TQP_DELETE_BY_COMPARISION_PREFIX + tableName,
					TQP_DEFAULT_PERSON_ID
					);
		}
	}

	/**
	 * This method will create then add deleted operation, in the pending operations queue.<br/>
	 * This need a TEMP_QUERY_PARAMETER filled with ID of deleted table
	 * 
	 * @param tableNames
	 * @param operations
	 * @param sourceDaoFactory
	 * @param targetDaoFactory
	 * @param dbMeta
	 * @param context
	 * @throws SQLException
	 */
	protected void addDeletedItemsFromTables(
			Set<String> tableNames,
			Deque<SynchroTableOperation> operations,
			DaoFactoryImpl sourceDaoFactory,
			DaoFactoryImpl targetDaoFactory,
			SynchroDatabaseMetadata dbMeta,
			SynchroContext context) throws SQLException {

		// Create a DAO on TQP (TEMP_QUERY_PARAMETER) table
		SynchroTableDao tqpDao = sourceDaoFactory.getSourceDao(SynchroBaseDao.TEMP_QUERY_PARAMETER_TABLE);
		int tqpValueColumnIndex = tqpDao.getTable().getColumnIndex(SynchroBaseDao.TEMP_QUERY_PARAMETER_VALUE_COLUMN);
		Preconditions.checkArgument(tqpValueColumnIndex != -1);

		// Prepare some variables need to read rows on TQP
		Map<String, Object> emptyBinding = Maps.newHashMap();
		Set<String> fkNames = ImmutableSet.of(SynchroBaseDao.TEMP_QUERY_PARAMETER_PARAM_COLUMN);

		// For each row that has deletion
		for (String tableName : tableNames) {
			SynchroTableMetadata table = dbMeta.getTable(tableName);
			boolean hasChildTables = table.hasChildJoins();
			Set<String> tablePkNames = table.getPkNames();
			int pkCount = tablePkNames.size();

			// Retrieve PK Str (stored by method 'saveTablesWithDelete()')
			List<Object> fkValue = ImmutableList.<Object> of(TQP_DELETE_BY_COMPARISION_PREFIX + tableName);
			ResultSet rs = tqpDao.getDataByFks(
					fkNames,
					ImmutableList.of(fkValue),
					emptyBinding);
			List<List<Object>> pks = Lists.newArrayList();
			while (rs.next()) {
				String pkStr = rs.getString(tqpValueColumnIndex + 1);
				List<Object> pk = SynchroTableMetadata.fromPkStr(pkStr);

				// Make sure the PK str was well formed
				if (pkCount != pk.size()) {
					String expectedPkStrExample = Joiner.on(String.format(">%s<", SynchroTableMetadata.PK_SEPARATOR)).join(tablePkNames);
					throw new SynchroTechnicalException(String.format(
							"Unable to import delete on %s: invalid PK found in the source database (in %s). Should have %s column (e.g. %s).",
							tableName,
							SynchroBaseDao.TEMP_QUERY_PARAMETER_TABLE,
							pkCount,
							expectedPkStrExample));
				}
				pks.add(pk);
			}
			rs.close();

			// Check TQP has been correctly filled
			if (pks.size() == 0) {
				throw new SynchroTechnicalException(String.format(
						"Unable to import delete on %s: No PK found in the source database (in %s). Unable to compare and find PKs to delete.",
						tableName,
						SynchroBaseDao.TEMP_QUERY_PARAMETER_TABLE));
			}

			SynchroTableOperation operation = new SynchroTableOperation(tableName, context);
			operation.setEnableProgress(true);
			SynchroTableDao targetTableDao = targetDaoFactory.getTargetDao(tableName, null, operation);

			List<List<Object>> pksToDelete = targetTableDao.getPksByNotFoundFks(
					tablePkNames,
					pks,
					emptyBinding);

			// Excluded temporary rows (keep temporary rows)
			pksToDelete = filterExcludeTemporary(table, pksToDelete);

			// If some rows need to be deleted
			if (CollectionUtils.isNotEmpty(pksToDelete)) {
				// Fill the operation as a 'delete operation'
				operation.addAllMissingDelete(pksToDelete);

				// If has children, add child deletion to result
				if (hasChildTables) {
					addDeleteChildrenToDeque(table, pksToDelete, operations, context);
				}

				// Add operation to the result list
				operations.add(operation);
			}

		}

		// Clean processed row from TempQueryParameter
		targetDaoFactory.getDao().executeDeleteTempQueryParameter(TQP_DELETE_BY_COMPARISION_PREFIX + "%", true, TQP_DEFAULT_PERSON_ID);
	}

	protected List<List<Object>> filterExcludeTemporary(
			SynchroTableMetadata table,
			List<List<Object>> pks) {
		Set<String> tablePkNames = table.getPkNames();

		if (tablePkNames.size() > 1) {
			return pks;
		}

		SynchroColumnMetadata pkColumn = table.getColumn(tablePkNames.iterator().next());
		Collection<List<Object>> result;

		// If pk is a numeric (e.g. an ID column)
		if (SynchroMetadataUtils.isNumericType(pkColumn)) {
			result = Collections2.filter(pks,
					new Predicate<List<Object>>() {
						@Override
						public boolean apply(@Nullable List<Object> input) {
							long pk = Long.parseLong(input.get(0).toString());
							return !TemporaryDataHelper.isTemporaryId(pk);
						}
					});
		}

		// If pk is a string (e.g. a CODE column)
		else {

			result = Collections2.filter(pks,
					new Predicate<List<Object>>() {
						@Override
						public boolean apply(@Nullable List<Object> input) {
							String pk = input.get(0).toString();
							return !TemporaryDataHelper.isTemporaryCode(pk);
						}
					});
		}

		return ImmutableList.copyOf(result);
	}

	protected final void rejectUnableToDeleteRow(
			String tableName,
			SynchroResult result,
			List<List<Object>> pks,
			SQLException sqle
			) throws SQLException {

		for (List<Object> pk : pks) {
			String pkStr = SynchroTableMetadata.toPkStr(pk);
			result.addReject(tableName,
					pkStr,
					RejectedRowStatus.DELETE_ERROR.name(),
					sqle.getMessage());
		}

		// throw new SynchroUnableToDeleteRowException(
		// targetDao.getTable().getName(),
		// pksStr.substring(1));

	}
}
