package fr.ifremer.adagio.core.service.data.synchro.intercept.internal;

/*
 * #%L
 * SIH-Adagio :: Core for Allegro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.List;

import org.nuiton.i18n.I18n;

import com.google.common.base.Preconditions;

import fr.ifremer.adagio.core.service.data.synchro.DataSynchroDatabaseConfiguration;
import fr.ifremer.adagio.synchro.SynchroTechnicalException;
import fr.ifremer.adagio.synchro.dao.Daos;
import fr.ifremer.adagio.synchro.dao.SynchroBaseDao;
import fr.ifremer.adagio.synchro.dao.SynchroTableDao;
import fr.ifremer.adagio.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.adagio.synchro.intercept.SynchroOperationRepository;

public class ImportRemoteIdInterceptor extends SynchroInterceptorBase {

	private final String tableName;

	private final String columnName;

	private final int columnIndex;

	private final boolean isNullable;

	private final String selectIdFromRemoteIdQuery;

	private DataSynchroDatabaseConfiguration config;

	private PreparedStatement selectIdFromRemoteIdStatement = null;

	public ImportRemoteIdInterceptor(DataSynchroDatabaseConfiguration config, String tableName, String columnName, int columnIndex, boolean isNullable) {
		super();
		Preconditions.checkArgument(columnIndex >= 0);
		this.tableName = tableName;
		this.columnName = columnName;
		this.columnIndex = columnIndex;
		this.isNullable = isNullable;
		this.config = config;
		this.selectIdFromRemoteIdQuery = initSelectIdFromRemoteIdQuery(config, tableName);
		setEnableOnRead(true);
		setEnableOnWrite(true);
	}

	@Override
	public SynchroInterceptorBase clone() {
		return new ImportRemoteIdInterceptor(
				config,
				tableName,
				columnName,
				columnIndex,
				isNullable);
	}

	@Override
	protected void doOnWrite(Object[] data, List<Object> pk, SynchroTableDao sourceDao, SynchroTableDao targetDao, SynchroOperationRepository buffer)
			throws SQLException {
		Integer remoteId = (Integer) data[columnIndex];
		if (remoteId == null) {
			return;
		}

		Integer localId = getIdFromRemoteId(targetDao, remoteId);
		if (localId == null) {
			// Mandatory column, could not continue
			if (!isNullable) {
				throw new SynchroTechnicalException(
						I18n.t("adagio.synchro.synchronizeData.noIdFromRemoteId.error",
								config.getColumnId(),
								tableName.toUpperCase(),
								config.getColumnRemoteId(),
								remoteId));
			}

			// Optional column : add to pending changes (will be processed later)
			buffer.addMissingColumnUpdate(columnName, pk, remoteId);
			data[columnIndex] = null;
		}
		else {
			data[columnIndex] = localId;
		}
	}

	@Override
	protected void doOnRead(Object[] data, SynchroTableDao sourceDao, SynchroTableDao targetDao) throws SQLException {
		Integer remoteId = null;
		if (data[columnIndex] instanceof String) {
			remoteId = Integer.valueOf((String) data[columnIndex]);
		}
		else {
			remoteId = (Integer) data[columnIndex];
		}
		if (remoteId == null) {
			return;
		}

		Integer localId = getIdFromRemoteId(targetDao, remoteId);
		if (localId == null) {
			// no match: could not continue
			throw new SynchroTechnicalException(
					I18n.t("adagio.synchro.synchronizeData.noIdFromRemoteId.error",
							config.getColumnId(),
							tableName.toUpperCase(),
							config.getColumnRemoteId(),
							remoteId));
		}

		data[columnIndex] = localId;
	}

	/* -- Internal methods -- */

	protected Integer getIdFromRemoteId(SynchroBaseDao dao, Integer remoteId) throws SQLException {

		Integer result = (Integer) dao.getUniqueTyped(selectIdFromRemoteIdQuery, new Object[] { remoteId });
		return result;
	}

	@Override
	protected void doClose() throws IOException {
		super.doClose();

		// Close statement
		Daos.closeSilently(selectIdFromRemoteIdStatement);
		selectIdFromRemoteIdStatement = null;
	}

	protected String initSelectIdFromRemoteIdQuery(DataSynchroDatabaseConfiguration config, String tableName) {
		return String.format("SELECT %s FROM %s where %s=?",
				config.getColumnId(),
				tableName,
				config.getColumnRemoteId()
				);
	}

}