package fr.ifremer.adagio.core.service.data.synchro.intercept.internal;

/*
 * #%L
 * SIH-Adagio :: Core for Allegro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.List;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;

import fr.ifremer.adagio.core.dao.technical.synchronization.SynchronizationStatus;
import fr.ifremer.adagio.core.service.data.synchro.DataSynchroDatabaseConfiguration;
import fr.ifremer.adagio.synchro.dao.Daos;
import fr.ifremer.adagio.synchro.dao.SynchroBaseDao;
import fr.ifremer.adagio.synchro.dao.SynchroTableDao;
import fr.ifremer.adagio.synchro.intercept.SynchroDeletedRowException;
import fr.ifremer.adagio.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.adagio.synchro.intercept.SynchroOperationRepository;
import fr.ifremer.adagio.synchro.service.SynchroResult;

/**
 * Read/Write Interceptor use on data table, to manager ID/REMOTE_ID conversion when exporting to server.
 * If column exists, rewrite update_date with systimestamp.
 * 
 * @author Benoit Lavenier <benoit.lavenier@e-is.pro>
 * @since 3.7.1
 * 
 */
public class ExportPkRemoteIdInterceptor extends SynchroInterceptorBase {

	public static final String COLUMN_SYNCHRONIZATION_STATUS = "synchronization_status";
	public static final String SYNCHRONIZATION_STATUS_SYNC = SynchronizationStatus.SYNCHRONIZED.getValue();

	private final String tableName;

	private final String columnName;

	private final int columnIndex;

	private final boolean hasChildJoins;

	private final String selectRemoteIdFromIdQuery;

	private final String updateRemoteIdByIdQuery;

	private final boolean withSynchronizationStatus;

	private final boolean withUpdateDate;

	private final int updateDateColumnIndex;

	private final Timestamp systimestamp;

	private final DataSynchroDatabaseConfiguration config;

	private PreparedStatement selectRemoteIdFromIdStatement = null;

	private PreparedStatement updateRemoteIdByIdStatement = null;

	public ExportPkRemoteIdInterceptor(DataSynchroDatabaseConfiguration config, String tableName, String columnName, int columnIndex,
			boolean withSynchronizationStatus,
			boolean withUpdateDate,
			int updateDateColumnIndex,
			Timestamp systimestamp, boolean hasChildJoins) {
		super();
		Preconditions.checkArgument(columnIndex >= 0);
		this.tableName = tableName;
		this.config = config;
		this.columnName = columnName;
		this.columnIndex = columnIndex;
		this.hasChildJoins = hasChildJoins;
		this.withSynchronizationStatus = withSynchronizationStatus;
		this.withUpdateDate = withUpdateDate;
		this.updateDateColumnIndex = updateDateColumnIndex;
		this.systimestamp = systimestamp;
		this.selectRemoteIdFromIdQuery = initSelectRemoteIdFromIdQuery(config, tableName);
		this.updateRemoteIdByIdQuery = initUpdateRemoteIdByIdQuery(config, tableName);
		setEnableOnWrite(true);
		setEnableOnRead(true);
	}

	@Override
	public SynchroInterceptorBase clone() {
		return new ExportPkRemoteIdInterceptor(
				config,
				tableName,
				columnName,
				columnIndex,
				withSynchronizationStatus,
				withUpdateDate,
				updateDateColumnIndex,
				systimestamp,
				hasChildJoins);
	}

	@Override
	protected void doOnWrite(Object[] data, List<Object> pk, SynchroTableDao sourceDao, SynchroTableDao targetDao,
			SynchroOperationRepository operationContext) throws SQLException {
		if (data[columnIndex] == null) {
			return;
		}
		long localId = Long.parseLong(data[columnIndex].toString());

		SynchroResult result = operationContext.getSynchroContext().getResult();

		Number remoteId = getRemoteIdFromId(sourceDao, localId);

		// If new row (=insert)
		if (remoteId == null) {
			// set the given Pk to fill the ID column
			remoteId = (Number) pk.get(0);

			// Remember to set remote_id on source local DB
			result.addSourceMissingColumnUpdate(
					tableName,
					config.getColumnRemoteId(),
					ImmutableList.<Object> of(localId),
					remoteId);

			// Update on Temp DB (e.g. need by ExportFkRemoteIdInterceptor to fill FK, or when a row is proceed twice)
			updateRemoteIdById(sourceDao, localId, remoteId);
		}
		else if (!remoteId.equals(pk.get(0))) {
			// The given PK is not equals to the remote_id
			// this should append when insertion is detected (PK not exists on target DB)
			// but it should be an update (because remote_id is not null on local DB)
			// => the row with id=remote_id may have been deleted on target DB (usually = on the server DB)
			throw new SynchroDeletedRowException(tableName,
					String.valueOf(localId),
					remoteId.toString());
		}
		data[columnIndex] = remoteId;

		// if column 'synchronization_status' exists on table
		if (withSynchronizationStatus) {
			// Set to 'SYNC' (on source local DB)
			result.addSourceMissingColumnUpdate(
					tableName,
					COLUMN_SYNCHRONIZATION_STATUS,
					ImmutableList.<Object> of(localId),
					SYNCHRONIZATION_STATUS_SYNC);
		}

		// If column 'update_date' exists on table
		if (withUpdateDate) {
			// Overwrite
			data[updateDateColumnIndex] = systimestamp;

			// Set to systimestamp (on source local DB)
			result.addSourceMissingColumnUpdate(
					tableName,
					config.getColumnUpdateDate(),
					ImmutableList.<Object> of(localId),
					systimestamp.toString());
		}
	}

	@Override
	protected void doOnRead(Object[] data, SynchroTableDao sourceDao, SynchroTableDao targetDao) throws SQLException {
		if (data[columnIndex] == null) {
			return;
		}
		Long localId = Long.parseLong(data[columnIndex].toString());

		Number remoteId = getRemoteIdFromId(sourceDao, localId);
		if (remoteId == null) {
			// New exported row has no remote_id
			// -> set to -1 to disable unique constraint check on remoteId
			data[columnIndex] = -1;
		}
		else {
			data[columnIndex] = remoteId;
		}

		// Overwrite column 'update_date'
		if (withUpdateDate) {
			data[updateDateColumnIndex] = systimestamp;
		}
	}

	/* -- Internal methods -- */

	protected Number getRemoteIdFromId(SynchroBaseDao dao, long localId) throws SQLException {
		// TODO BLA : see if we can used cached query/value like such code:
		// --> Number result = (Number)dao.getUniqueTyped(selectRemoteIdFromIdQuery, new Object[] {localId});
		// This is not sure: value may changed ?? => Need to be tested.

		if (selectRemoteIdFromIdStatement == null || selectRemoteIdFromIdStatement.isClosed()) {
			selectRemoteIdFromIdStatement = dao.getPreparedStatement(selectRemoteIdFromIdQuery);
		}
		selectRemoteIdFromIdStatement.setLong(1, localId);
		ResultSet resultSet = null;
		try {
			resultSet = selectRemoteIdFromIdStatement.executeQuery();
			if (!resultSet.next()) {
				return null;
			}
			if (resultSet.getObject(1) == null) {
				return null;
			}

			return (Number) resultSet.getObject(1);
		} finally {
			Daos.closeSilently(resultSet);
		}
	}

	protected boolean updateRemoteIdById(SynchroBaseDao dao, long localId, Number remoteId) throws SQLException {
		if (updateRemoteIdByIdStatement == null || updateRemoteIdByIdStatement.isClosed()) {
			updateRemoteIdByIdStatement = dao.getPreparedStatement(updateRemoteIdByIdQuery);
		}
		updateRemoteIdByIdStatement.setObject(1, remoteId);
		updateRemoteIdByIdStatement.setLong(2, localId);

		return updateRemoteIdByIdStatement.execute();
	}

	@Override
	protected void doClose() throws IOException {
		super.doClose();

		// Close select statement
		Daos.closeSilently(selectRemoteIdFromIdStatement);
		selectRemoteIdFromIdStatement = null;

		// Close update statement
		Daos.closeSilently(updateRemoteIdByIdStatement);
		updateRemoteIdByIdStatement = null;
	}

	protected String initSelectRemoteIdFromIdQuery(DataSynchroDatabaseConfiguration config, String tableName) {
		return String.format("SELECT %s FROM %s where %s=?",
				config.getColumnRemoteId(),
				tableName,
				config.getColumnId()
				);
	}

	protected String initUpdateRemoteIdByIdQuery(DataSynchroDatabaseConfiguration config, String tableName) {
		return String.format("UPDATE %s set %s=? where %s=?",
				tableName,
				config.getColumnRemoteId(),
				config.getColumnId()
				);
	}

}