package fr.ifremer.adagio.core.service.data.synchro.intercept.internal;

/*
 * #%L
 * SIH-Adagio :: Core for Allegro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.List;

import org.nuiton.i18n.I18n;

import com.google.common.base.Preconditions;

import fr.ifremer.adagio.core.service.data.synchro.DataSynchroDatabaseConfiguration;
import fr.ifremer.adagio.synchro.SynchroTechnicalException;
import fr.ifremer.adagio.synchro.dao.Daos;
import fr.ifremer.adagio.synchro.dao.SynchroBaseDao;
import fr.ifremer.adagio.synchro.dao.SynchroTableDao;
import fr.ifremer.adagio.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.adagio.synchro.intercept.SynchroOperationRepository;

public class ExportFkRemoteIdInterceptor extends SynchroInterceptorBase {

	private final String tableName;

	private final String columnName;

	private final int columnIndex;

	private final boolean isNullable;

	private final String selectRemoteIdFromIdQuery;

	private final DataSynchroDatabaseConfiguration config;

	private PreparedStatement selectRemoteIdFromIdStatement = null;

	public ExportFkRemoteIdInterceptor(DataSynchroDatabaseConfiguration config, String tableName, String columnName, int columnIndex,
			boolean isNullable) {
		super();
		Preconditions.checkArgument(columnIndex >= 0);
		this.config = config;
		this.tableName = tableName;
		this.columnName = columnName;
		this.columnIndex = columnIndex;
		this.isNullable = isNullable;
		this.selectRemoteIdFromIdQuery = initSelectRemoteIdFromIdQuery(config, tableName);
		setEnableOnWrite(true);
		setEnableOnRead(true);
	}

	@Override
	public SynchroInterceptorBase clone() {
		return new ExportFkRemoteIdInterceptor(
				config,
				tableName,
				columnName,
				columnIndex,
				isNullable);
	}

	@Override
	protected void doOnWrite(Object[] data,
			List<Object> pk,
			SynchroTableDao sourceDao,
			SynchroTableDao targetDao,
			SynchroOperationRepository operationContext)
			throws SQLException {

		if (data[columnIndex] == null) {
			return;
		}
		long localId = Long.parseLong(data[columnIndex].toString());

		Number remoteId = getRemoteIdFromId(sourceDao, localId);
		if (remoteId == null) {
			// Mandatory column, could not continue
			if (!isNullable) {
				throw new SynchroTechnicalException(
						I18n.t("adagio.synchro.synchronizeData.noRemoteIdFromId.error",
								config.getColumnRemoteId(),
								tableName.toUpperCase(),
								config.getColumnId(),
								localId));
			}

			// Optional column : add to pending changes (will be processed later)
			operationContext.addMissingColumnUpdate(columnName, pk, localId);
			data[columnIndex] = null;
		}
		else {
			data[columnIndex] = remoteId;
		}
	}

	@Override
	protected void doOnRead(Object[] data, SynchroTableDao sourceDao, SynchroTableDao targetDao) throws SQLException {
		if (data[columnIndex] == null) {
			return;
		}
		long localId = Long.parseLong(data[columnIndex].toString());

		Number remoteId = getRemoteIdFromId(sourceDao, localId);
		if (remoteId == null) {
			// no match: maybe be the referenced row will be inserted later
			data[columnIndex] = null;
			return;
		}

		data[columnIndex] = remoteId;
	}

	/* -- Internal methods -- */

	protected Number getRemoteIdFromId(SynchroBaseDao dao, long localId) throws SQLException {

		Number result = (Number) dao.getUniqueTyped(selectRemoteIdFromIdQuery, new Object[] { localId });

		return result;
	}

	@Override
	protected void doClose() throws IOException {
		super.doClose();

		// Close statement
		Daos.closeSilently(selectRemoteIdFromIdStatement);
		selectRemoteIdFromIdStatement = null;
	}

	protected String initSelectRemoteIdFromIdQuery(DataSynchroDatabaseConfiguration config, String tableName) {
		return String.format("SELECT %s FROM %s where %s=?",
				config.getColumnRemoteId(),
				tableName,
				config.getColumnId()
				);
	}

}