package fr.ifremer.adagio.core.service.data.synchro.intercept;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.util.Collection;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.eventbus.Subscribe;

import fr.ifremer.adagio.core.config.AdagioConfiguration;
import fr.ifremer.adagio.core.dao.technical.synchronization.SynchronizationStatus;
import fr.ifremer.adagio.core.service.data.synchro.DataSynchroDatabaseConfiguration;
import fr.ifremer.adagio.core.service.data.synchro.DataSynchroDirection;
import fr.ifremer.adagio.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata;
import fr.ifremer.adagio.synchro.meta.event.CreateQueryEvent;
import fr.ifremer.adagio.synchro.meta.event.LoadTableEvent;
import fr.ifremer.adagio.synchro.query.SynchroQueryBuilder;
import fr.ifremer.adagio.synchro.query.SynchroQueryName;
import fr.ifremer.adagio.synchro.query.SynchroQueryOperator;

public class RootDataTableInterceptor extends DataAbstractSynchroInterceptor {

	public static final String COLUMN_ID = "id";
	public static final String COLUMN_VESSEL_FK = "vessel_fk";
	public static final String COLUMN_PROGRAM_FK = "program_fk";
	public static final String COLUMN_SYNCHRONIZATION_STATUS = "synchronization_status";

	@Override
	public boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table) {
		boolean isRootTable = isRootTable(table);
		return isRootTable;
	}

	@Subscribe
	public void handleLoadTable(LoadTableEvent e) {
		// Table should be set as root manually
		// (because landing should not be define as root)
		// if (!table.isRoot()) {
		// e.table.setRoot(true);
		// }
	}

	@Subscribe
	public void handleQuery(CreateQueryEvent e) {
		DataSynchroDirection direction = getConfig().getDirection();

		switch (e.queryName) {
		// Select queries : remove unsed columns
		case count:
		case countFromUpdateDate:
		case select:
		case selectFromUpdateDate:
		case selectMaxUpdateDate:
			if (direction == DataSynchroDirection.IMPORT_SERVER2TEMP) {
				// Add restriction on person session
				e.sql = addRestrictionOnImport(e.source, e.queryName, e.sql);

				// Add a filter for debug
				e.sql = addDebugRestriction(e.sql);
			}
			else if (direction == DataSynchroDirection.EXPORT_LOCAL2TEMP) {
				// Add restriction on person session
				e.sql = addRestrictionOnExport(e.source, e.queryName, e.sql);

				// Add a filter for debug
				e.sql = addDebugRestriction(e.sql);
			}
			break;

		default:
			break;
		}
	}

	protected String addRestrictionOnImport(SynchroTableMetadata table, SynchroQueryName queryName, String sql) {

		DataSynchroDatabaseConfiguration databaseConfiguration = getConfig();
		boolean enableUpdateDateFilter = queryName.name().endsWith("ToUpdate");

		Set<String> objectTypes = ObjectTypeHelper.getObjectTypeFromTableName(table.getName());
		Preconditions.checkNotNull(CollectionUtils.isNotEmpty(objectTypes));
		String objectTypeList = new StringBuilder().append("'").append(Joiner.on("','").join(objectTypes)).append("'")
				.toString();

		SynchroQueryBuilder queryBuilder = SynchroQueryBuilder.newBuilder(queryName, sql);

		// Retrieve the session id
		int personSessionId = checkAndGetPersonSessionId();

		// select
		// -> need to add a 'distinct' because of inner join on LANDING
		if (queryName == SynchroQueryName.count
				|| queryName == SynchroQueryName.countFromUpdateDate) {
			queryBuilder.replaceColumn("count(*)", "count(distinct t.ID)");
		}
		else {
			queryBuilder.setColumnDistinct(true);
		}

		// from
		queryBuilder.addJoin(
				"INNER JOIN PERSON_SESSION_VESSEL psv"
						+ " ON psv.vessel_fk=t.vessel_fk"
						+ String.format(" AND psv.person_session_fk=%s", personSessionId)
						+ " AND psv.program_fk=t.program_fk");

		// where: if rights updated
		if (enableUpdateDateFilter) {
			queryBuilder.addWhere(SynchroQueryOperator.OR, "psv.update_date", ">", ":updateDate");
		}

		// where: limit rights on object types
		queryBuilder.addWhere(SynchroQueryOperator.AND,
				String.format("psv.object_type_fk IN (%s)", objectTypeList));

		// where: programs
		String programFilter = createProgramCodesFilter("t.program_fk IN (%s)");
		if (StringUtils.isNotBlank(programFilter)) {
			queryBuilder.addWhere(SynchroQueryOperator.AND, programFilter);
		}

		// where: limit to pks (for import by Pk)
		String pkFilter = createPkFilter(table.getName());
		if (StringUtils.isNotBlank(pkFilter)) {
			// Apply Pk filter, but do not apply date restriction
			queryBuilder.addWhere(SynchroQueryOperator.AND, pkFilter);
		}

		// where: filter on period, if a date column exists AND no Pk filter
		else if (databaseConfiguration.getDataStartDate() != null
				&& databaseConfiguration.getDataEndDate() != null) {
			// Retrieve the date column to user
			String dateColumnName = getFirstExistingColumnName(table,
					"return_date_time",
					"sale_start_date",
					"landing_date_time",
					"end_date_time",
					"start_date_time"
					);
			// If found, add a filter on start/end dates
			if (dateColumnName != null) {
				queryBuilder.addWhere(SynchroQueryOperator.AND, String.format(
						"t.%s  >= :startDate AND t.%s <= :endDate", dateColumnName, dateColumnName));
			}
		}

		return queryBuilder.build();
	}

	protected String addRestrictionOnExport(SynchroTableMetadata table, SynchroQueryName queryName, String sql) {

		Set<String> objectTypes = ObjectTypeHelper.getObjectTypeFromTableName(table.getName());
		Preconditions.checkNotNull(CollectionUtils.isNotEmpty(objectTypes));
		String objectTypeList = new StringBuilder().append("'").append(Joiner.on("','").join(objectTypes)).append("'")
				.toString();

		SynchroQueryBuilder queryBuilder = SynchroQueryBuilder.newBuilder(queryName, sql);

		// Retrieve the person id
		int personId = checkAndGetPersonId();

		// select
		// -> need to add a 'distinct' because of inner join on LANDING
		if (queryName == SynchroQueryName.count
				|| queryName == SynchroQueryName.countFromUpdateDate) {
			queryBuilder.replaceColumn("count(*)", "count(distinct t.ID)");
		}
		else {
			queryBuilder.setColumnDistinct(true);
		}

		// join on PERSON_SESSION

		queryBuilder.addJoin(
				"INNER JOIN PERSON_SESSION_VESSEL psv"
						+ " ON psv.vessel_fk=t.vessel_fk"
						+ " AND psv.program_fk=t.program_fk"
						+ " INNER JOIN PERSON_SESSION ps ON ps.id=psv.person_session_fk"
						+ String.format(" AND ps.person_fk=%s", personId));

		// where: only 'ready to sync' data
		queryBuilder.addWhere(SynchroQueryOperator.AND,
				String.format("%s='%s'", COLUMN_SYNCHRONIZATION_STATUS, SynchronizationStatus.READY_TO_SYNCHRONIZE.value()));

		// where: limit rights on object types
		queryBuilder.addWhere(SynchroQueryOperator.AND,
				String.format("psv.object_type_fk IN (%s)", objectTypeList));

		// where: programs
		String programFilter = createProgramCodesFilter("t.program_fk IN (%s)");
		if (programFilter.length() > 0) {
			queryBuilder.addWhere(SynchroQueryOperator.AND, programFilter);
		}

		return queryBuilder.build();
	}

	protected boolean isRootTable(TableMetadata table) {
		boolean hasRootColumns = hasColumns(table, COLUMN_VESSEL_FK, COLUMN_PROGRAM_FK, getConfig().getColumnUpdateDate());
		boolean hasObjectType = CollectionUtils.isNotEmpty(ObjectTypeHelper.getObjectTypeFromTableName(table.getName()));

		return hasRootColumns && hasObjectType;
	}

	protected String addDebugRestriction(String sql) {

		String vesselIncludesStr = AdagioConfiguration.getInstance().getApplicationConfig().getOption("adagio.synchro.import.vessels.includes");
		if (StringUtils.isBlank(vesselIncludesStr)) {
			return sql;
		}

		// Vessels to includes
		StringBuilder vesselsParam = new StringBuilder();
		for (String vesselCode : vesselIncludesStr.split(",")) {
			vesselsParam.append(",'")
					.append(vesselCode)
					.append("'");
		}
		return SynchroQueryBuilder.newBuilder(sql)
				.addWhere(SynchroQueryOperator.AND, String.format("t.vessel_fk in (%s)", vesselsParam.substring(1)))
				.build();
	}

	protected String createProgramCodesFilter(String stringToFormat) {
		String programCodes = AdagioConfiguration.getInstance().getImportProgramCodes();
		if (StringUtils.isBlank(programCodes)) {
			return null;
		}
		return String.format(stringToFormat,
				"'"
						+ Joiner.on("','").join(Splitter.on(',').split(programCodes))
						+ "'");
	}

	protected String createPkFilter(String tableName) {
		if (getConfig().getPkIncludes() == null
				|| getConfig().getPkIncludes().isEmpty()) {
			return null;
		}

		Collection<String> pkStrs = getConfig()
				.getPkIncludes()
				.get(tableName.toUpperCase());

		if (CollectionUtils.isEmpty(pkStrs)) {
			// There is filter by PK, but not on this table
			// so exclude this table
			return "1=2";
		}

		StringBuilder inBuilder = new StringBuilder();
		for (String pkStr : pkStrs) {
			inBuilder.append(',').append(pkStr);
		}

		return String.format("t.%s IN (%s)",
				COLUMN_ID,
				inBuilder.substring(1)
				);
	}

	protected String getFirstExistingColumnName(SynchroTableMetadata table, String... columnNames) {
		Set<String> delegateColumns = table.getColumnNames();
		for (String columnName : columnNames) {
			if (delegateColumns.contains(columnName.toLowerCase())) {
				return columnName;
			}
		}
		return null;
	}
}
