package fr.ifremer.adagio.core.service.data.synchro.intercept;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.eventbus.Subscribe;
import fr.ifremer.adagio.core.service.data.synchro.DataSynchroDatabaseConfiguration;
import fr.ifremer.adagio.synchro.dao.Daos;
import fr.ifremer.adagio.synchro.dao.SynchroTableDao;
import fr.ifremer.adagio.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.adagio.synchro.intercept.SynchroOperationRepository;
import fr.ifremer.adagio.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.adagio.synchro.meta.event.LoadTableEvent;
import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;
import org.hibernate.tool.hbm2ddl.TableMetadata;

/**
 * 
 * @author Lionel Touseau <lionel.touseau@e-is.pro>
 * @since 3.7.2
 */
public class OperationInterceptor extends DataAbstractSynchroInterceptor {

	private int catchBatchFkColumnIndex = -1;
	private boolean enableIntegrityConstraints = true;

	private static final String TABLE_BATCH = "BATCH";
	private static final String TABLE_OPERATION = "OPERATION";

	public static final String COLUMN_CATCH_BATCH = "catch_batch_fk";
	private static final String COLUMN_ID = "id";

	private PreparedStatement selectCatchBatchIdStatement = null;

	public OperationInterceptor() {
		super();
		setEnableOnWrite(true);
	}

	@Override
	public void init(DataSynchroDatabaseConfiguration config) {
		super.init(config);
		enableIntegrityConstraints = config.isIntegrityConstraintEnable();
	}

	@Override
	public SynchroInterceptorBase clone() {
		OperationInterceptor newBean = (OperationInterceptor) super.clone();
		newBean.catchBatchFkColumnIndex = this.catchBatchFkColumnIndex;
		newBean.enableIntegrityConstraints = this.enableIntegrityConstraints;
		return newBean;
	}

	@Override
	public boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table) {
		return TABLE_OPERATION.equalsIgnoreCase(table.getName());
	}

	@Subscribe
	public void handleTableLoad(LoadTableEvent e) {
		// get catch batch column index
		catchBatchFkColumnIndex = e.table.getSelectColumnIndex(COLUMN_CATCH_BATCH);

		e.table.setRoot(false);
	}

	@Override
	protected void doOnWrite(Object[] data, List<Object> pk, SynchroTableDao sourceDao, SynchroTableDao targetDao, SynchroOperationRepository buffer)
			throws SQLException {
		// If root entity: no transformation
		if (buffer == null) {
			return;
		}
		Object catchBatchFk = data[catchBatchFkColumnIndex];
		if (catchBatchFk != null) {
			buffer.addChildToUpdateFromOneColumn(TABLE_BATCH, COLUMN_ID, catchBatchFk);

			if (enableIntegrityConstraints) {
				// set as Null, and update later
				data[catchBatchFkColumnIndex] = null;

				buffer.addMissingColumnUpdate(COLUMN_CATCH_BATCH, pk, catchBatchFk);
			}
		}
	}

	@Override
	protected void doOnDelete(List<Object> pk, SynchroTableDao sourceDao, SynchroTableDao targetDao, SynchroOperationRepository buffer)
			throws SQLException {
		// Warning: use parseInt because ID could a String or an Integer
		Integer operationId = Integer.parseInt(pk.get(0).toString());

		// Retrieve catch batch id
		Integer catchBatchId = getCatchBatchId(targetDao.getConnection(), operationId);

		// If exists, add to children to delete
		if (catchBatchId != null) {
			buffer.addChildToDeleteFromOneColumn(TABLE_BATCH, COLUMN_ID, catchBatchId);
		}
	};

	@Override
	protected void doOnDetach(List<Object> pk, SynchroTableDao sourceDao, SynchroTableDao targetDao, SynchroOperationRepository buffer)
			throws SQLException {
		// Warning: use parseInt because ID could a String or an Integer
		Integer operationId = Integer.parseInt(pk.get(0).toString());

		// Retrieve fishing trip id
		Integer catchBatchId = getCatchBatchId(targetDao.getConnection(), operationId);

		// If exists, add to children to delete
		if (catchBatchId != null) {
			buffer.addChildToDetachFromOneColumn(TABLE_BATCH, COLUMN_ID, catchBatchId);
		}
	};

	@Override
	protected void doClose() throws IOException {
		super.doClose();

		Daos.closeSilently(selectCatchBatchIdStatement);
		selectCatchBatchIdStatement = null;
	}

	/* -- internal methods -- */

	protected Integer getCatchBatchId(Connection connection, int OperationId) throws SQLException {
		if (selectCatchBatchIdStatement == null) {
			selectCatchBatchIdStatement = connection.prepareStatement(initSelectCatchBatchIdStatement());
		}

		selectCatchBatchIdStatement.setInt(1, OperationId);

		ResultSet resultSet = null;
		try {
			resultSet = selectCatchBatchIdStatement.executeQuery();

			if (!resultSet.next()) {
				return null;
			}
			if (resultSet.getObject(1) == null) {
				return null;
			}
			return resultSet.getInt(1);
		} finally {
			Daos.closeSilently(resultSet);
		}
	}

	protected String initSelectCatchBatchIdStatement() {
		return String.format("SELECT %s FROM %s where %s=?",
				COLUMN_CATCH_BATCH,
				TABLE_OPERATION,
				COLUMN_ID
				);
	}

}
