package fr.ifremer.adagio.core.service.data.synchro.intercept;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.LockMode;
import org.hibernate.tool.hbm2ddl.ColumnMetadata;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableList;
import com.google.common.eventbus.Subscribe;

import fr.ifremer.adagio.core.config.AdagioConfiguration;
import fr.ifremer.adagio.core.dao.technical.synchronization.SynchronizationStatus;
import fr.ifremer.adagio.core.service.data.synchro.DataSynchroDirection;
import fr.ifremer.adagio.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata.DuplicateKeyStrategy;
import fr.ifremer.adagio.synchro.meta.event.CreateQueryEvent;
import fr.ifremer.adagio.synchro.meta.event.LoadTableEvent;
import fr.ifremer.adagio.synchro.query.SynchroQueryBuilder;
import fr.ifremer.adagio.synchro.query.SynchroQueryName;
import fr.ifremer.adagio.synchro.query.SynchroQueryOperator;

public class ObservedLocationInterceptor extends DataAbstractSynchroInterceptor {

	public static final String COLUMN_PROGRAM_FK = "program_fk";
	public static final String COLUMN_START_DATE_TIME = "start_date_time";
	public static final String COLUMN_LOCATION_FK = "location_fk";
	public static final String COLUMN_RECORDER_PERSON_FK = "recorder_person_fk";
	public static final String COLUMN_SYNCHRONIZATION_STATUS = "synchronization_status";

	public static final List<String> NATURAL_ID_COLUMN_NAMES = ImmutableList.of(COLUMN_PROGRAM_FK, COLUMN_START_DATE_TIME, COLUMN_LOCATION_FK,
			COLUMN_RECORDER_PERSON_FK);

	@Override
	public boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table) {
		if (!"OBSERVED_LOCATION".equalsIgnoreCase(table.getName())) {
			return false;
		}

		Map<String, ColumnMetadata> delegateColumns = SynchroTableMetadata.getColumns(table);
		boolean hasNeedColumns = delegateColumns.containsKey(COLUMN_PROGRAM_FK);

		return hasNeedColumns;
	}

	@Subscribe
	public void handleTableLoad(LoadTableEvent e) {
		SynchroTableMetadata table = e.table;

		// Define as root table
		table.setRoot(true);

		// When exporting to the server
		if (getConfig().getDirection() == DataSynchroDirection.EXPORT_TEMP2SERVER) {
			// Define natural id
			table.addUniqueConstraint("NATURAL_ID_UNIQUE_C", NATURAL_ID_COLUMN_NAMES, DuplicateKeyStrategy.REJECT);

			// Enable lock on update
			table.setLockOnUpdate(LockMode.UPGRADE_NOWAIT);
		}
	}

	@Subscribe
	public void handleQuery(CreateQueryEvent e) {
		DataSynchroDirection direction = getConfig().getDirection();

		switch (e.queryName) {
		// Select queries : remove unsed columns
		case count:
		case countFromUpdateDate:
		case select:
		case selectFromUpdateDate:
		case selectMaxUpdateDate:
			// Import
			if (direction == DataSynchroDirection.IMPORT_SERVER2TEMP) {
				// Add restriction on person session
				e.sql = addRestrictionOnImport(e.source, e.queryName, e.sql);

				// Add a filter for debug, on ID and VESSEL_FK
				e.sql = addDebugRestriction(e.sql, getConfig().getColumnId());
			}

			// Export
			else if (direction == DataSynchroDirection.EXPORT_LOCAL2TEMP) {
				// Add restriction on person session
				e.sql = addRestrictionOnExport(e.source, e.queryName, e.sql);

				// Add a filter for debug, on REMOTE_ID and VESSEL_FK
				e.sql = addDebugRestriction(e.sql, getConfig().getColumnRemoteId());
			}

			break;

		default:
			break;
		}
	}

	protected String addRestrictionOnImport(SynchroTableMetadata table, SynchroQueryName queryName, String sql) {

		boolean enableUpdateDateFilter = queryName.name().endsWith("ToUpdate");

		Set<String> objectTypes = ObjectTypeHelper.getObjectTypeFromTableName(table.getName());
		Preconditions.checkArgument(CollectionUtils.isNotEmpty(objectTypes));
		String objectTypeList = new StringBuilder().append("'").append(Joiner.on("','").join(objectTypes)).append("'")
				.toString();

		SynchroQueryBuilder queryBuilder = SynchroQueryBuilder.newBuilder(queryName, sql);

		// Retrieve the session id
		int personSessionId = checkAndGetPersonSessionId();

		// select
		// -> need to add a 'distinct' because of inner join on LANDING
		if (queryName == SynchroQueryName.count
				|| queryName == SynchroQueryName.countFromUpdateDate) {
			queryBuilder.replaceColumn("count(*)", "count(distinct t.ID)");
		}
		else {
			queryBuilder.setColumnDistinct(true);
		}

		// Add join on PERSON_SESSION
		queryBuilder.addJoin(
				"LEFT OUTER JOIN LANDING L ON t.ID=L.OBSERVED_LOCATION_FK"
						+ " LEFT OUTER JOIN PERSON_SESSION_VESSEL psv ON psv.vessel_fk=L.vessel_fk AND psv.program_fk = t.program_fk"
						+ " AND psv.person_session_fk=" + personSessionId);

		// where
		if (enableUpdateDateFilter) {
			queryBuilder.addWhere(SynchroQueryOperator.OR, "psv.update_date", ">", ":updateDate");
		}

		// where: select observed_location without landing, OR rights on expected object types
		queryBuilder.addWhere(SynchroQueryOperator.AND,
				"((L.ID is NULL)"
						+ " OR ("
						+ String.format("psv.object_type_fk IN (%s)", objectTypeList)
						+ "))"
				);

		// Filter on programs
		String programFilter = createProgramCodesFilter("t.program_fk IN (%s)");
		if (programFilter.length() > 0) {
			queryBuilder.addWhere(SynchroQueryOperator.AND, programFilter);
		}

		// where: limit to pks (for import by Pk)
		String pkFilter = createPkFilter(table.getName());
		if (StringUtils.isNotBlank(pkFilter)) {
			queryBuilder.addWhere(SynchroQueryOperator.AND, pkFilter);
		}

		// where: period, if a date column exists AND no Pk filter
		else if (getConfig().getDataStartDate() != null
				&& getConfig().getDataEndDate() != null) {

			queryBuilder.addWhere(SynchroQueryOperator.AND, "t.start_date_time >= :startDate AND t.start_date_time <= :endDate");
		}
		return queryBuilder.build();
	}

	protected String addRestrictionOnExport(SynchroTableMetadata table, SynchroQueryName queryName, String sql) {

		Set<String> objectTypes = ObjectTypeHelper.getObjectTypeFromTableName(table.getName());
		Preconditions.checkArgument(CollectionUtils.isNotEmpty(objectTypes));
		String objectTypeList = new StringBuilder().append("'").append(Joiner.on("','").join(objectTypes)).append("'")
				.toString();

		SynchroQueryBuilder queryBuilder = SynchroQueryBuilder.newBuilder(queryName, sql);

		// Retrieve the session id
		int personId = checkAndGetPersonId();

		// select
		// -> need to add a 'distinct' because of inner join on LANDING
		if (queryName == SynchroQueryName.count
				|| queryName == SynchroQueryName.countFromUpdateDate) {
			queryBuilder.replaceColumn("count(*)", "count(distinct t.ID)");
		}
		else {
			queryBuilder.setColumnDistinct(true);
		}

		// Add join on PERSON_SESSION
		queryBuilder.addJoin(
				"LEFT OUTER JOIN LANDING L ON t.ID=L.OBSERVED_LOCATION_FK"
						+ " LEFT OUTER JOIN PERSON_SESSION_VESSEL psv ON psv.vessel_fk=L.vessel_fk AND psv.program_fk = t.program_fk"
						+ " LEFT OUTER JOIN PERSON_SESSION ps ON psv.person_session_fk=ps.id AND ps.person_fk=" + personId);

		// where: synchronisation_status
		queryBuilder.addWhere(SynchroQueryOperator.AND,
				String.format("%s='%s'", COLUMN_SYNCHRONIZATION_STATUS, SynchronizationStatus.READY_TO_SYNCHRONIZE.value()));

		// where: select observed_location without landing
		queryBuilder.addWhere(SynchroQueryOperator.AND,
				"((L.ID is NULL)"
						+ " OR (ps.id is not null"
						+ String.format(" AND psv.object_type_fk IN (%s)", objectTypeList)
						+ "))"
				);

		// where: programs
		String programFilter = createProgramCodesFilter("t.program_fk IN (%s)");
		if (programFilter.length() > 0) {
			queryBuilder.addWhere(SynchroQueryOperator.AND, programFilter);
		}

		return queryBuilder.build();
	}

	protected String addDebugRestriction(String sql, String observedLocationFilterColumnName) {

		String idIncludes = AdagioConfiguration.getInstance().getApplicationConfig().getOption("adagio.synchro.import.observedLocation.includes");
		String vesselIncludesStr = AdagioConfiguration.getInstance().getApplicationConfig().getOption("adagio.synchro.import.vessels.includes");
		if (StringUtils.isBlank(idIncludes) && StringUtils.isBlank(vesselIncludesStr)) {
			return sql;
		}

		SynchroQueryBuilder queryBuilder = SynchroQueryBuilder.newBuilder(sql);

		// IDs to include
		if (StringUtils.isNotBlank(idIncludes)) {
			queryBuilder.addWhere(SynchroQueryOperator.AND, String.format("t.%s in (%s)",
					observedLocationFilterColumnName,
					idIncludes));
		}

		// Vessels to includes
		if (StringUtils.isNotBlank(vesselIncludesStr)) {
			StringBuilder vesselParam = new StringBuilder();
			for (String vesselCode : vesselIncludesStr.split(",")) {
				vesselParam.append(",'")
						.append(vesselCode)
						.append("'");
			}
			queryBuilder.addWhere(SynchroQueryOperator.AND,
					String.format("t.id in (select distinct l.observed_location_fk from landing l where l.vessel_fk in (%s))",
							vesselParam.substring(1)));
		}

		return queryBuilder.build();
	}

	protected String createProgramCodesFilter(String stringToFormat) {
		String programCodes = AdagioConfiguration.getInstance().getImportProgramCodes();
		if (StringUtils.isBlank(programCodes)) {
			return "";
		}
		return String.format(stringToFormat,
				"'"
						+ Joiner.on("','").join(Splitter.on(',').split(programCodes))
						+ "'");
	}

	protected String createPkFilter(String tableName) {
		if (getConfig().getPkIncludes() == null
				|| getConfig().getPkIncludes().isEmpty()) {
			return null;
		}

		Collection<String> pkStrs = getConfig()
				.getPkIncludes()
				.get(tableName.toUpperCase());

		if (CollectionUtils.isEmpty(pkStrs)) {
			// There is filter by ID, but not on this table
			// so exclude this table
			return "1=2";
		}

		StringBuilder inBuilder = new StringBuilder();
		for (String pkStr : pkStrs) {
			inBuilder.append(',').append(pkStr);
		}

		return String.format("t.%s IN (%s)",
				getConfig().getColumnId(),
				inBuilder.substring(1)
				);
	}
}
