package fr.ifremer.adagio.core.service.data.synchro.intercept;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import org.hibernate.tool.hbm2ddl.TableMetadata;

import com.google.common.eventbus.Subscribe;

import fr.ifremer.adagio.core.service.data.synchro.DataSynchroDirection;
import fr.ifremer.adagio.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroJoinMetadata;
import fr.ifremer.adagio.synchro.meta.event.CreateQueryEvent;
import fr.ifremer.adagio.synchro.meta.event.LoadJoinEvent;
import fr.ifremer.adagio.synchro.query.SynchroQueryBuilder;

/**
 * 
 * @author Benoit Lavenier <benoit.lavenier@e-is.pro>
 * @since 3.5.0
 * 
 */
public class FishingTripOriginInterceptor extends DataAbstractSynchroInterceptor {

	public static final String COLUMN_PROGRAM_FK = "program_fk";

	private final String COLUMN_ORIGIN_FISHING_TRIP_REMOTE_ID = "origin_fishing_trip_remote_id";
	private final String COLUMN_ORIGIN_FISHING_TRIP_FK = "origin_fishing_trip_fk";

	public FishingTripOriginInterceptor() {
		super();
	}

	@Override
	public boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table) {
		return "FISHING_TRIP_ORIGIN".equalsIgnoreCase(table.getName());
	}

	@Subscribe
	public void handleQuery(CreateQueryEvent e) {
		DataSynchroDirection direction = getConfig().getDirection();

		// Target database is a mirror (Server DB -> Temp DB)
		if (direction == DataSynchroDirection.IMPORT_SERVER2TEMP) {

			switch (e.queryName) {
			// Select queries : remove unsed columns
			case select:
			case selectFromUpdateDate:
				e.sql = SynchroQueryBuilder.newBuilder(e.queryName, e.sql)
						.deleteColumnIfExists(COLUMN_ORIGIN_FISHING_TRIP_REMOTE_ID)
						.build();
				break;
			case insert:
			case update: {
				e.sql = SynchroQueryBuilder.newBuilder(e.queryName, e.sql)
						.deleteColumnIfExists(COLUMN_ORIGIN_FISHING_TRIP_REMOTE_ID)
						.build();
				break;
			}
			default:
				break;
			}
		}

		// Target database is not a mirror database (Temp DB -> Local DB)
		else if (direction == DataSynchroDirection.IMPORT_TEMP2LOCAL) {

			switch (e.queryName) {
			// Select queries : remove unsed columns
			case select:
			case selectFromUpdateDate:
				e.sql = SynchroQueryBuilder.newBuilder(e.queryName, e.sql)
						.deleteColumnIfExists(COLUMN_ORIGIN_FISHING_TRIP_REMOTE_ID)
						.build();
				break;
			case insert: {
				e.sql = SynchroQueryBuilder.newBuilder(e.queryName, e.sql)
						.deleteColumnIfExists(COLUMN_ORIGIN_FISHING_TRIP_REMOTE_ID)
						.replaceColumn(COLUMN_ORIGIN_FISHING_TRIP_FK, COLUMN_ORIGIN_FISHING_TRIP_REMOTE_ID)
						.build();
				break;
			}
			case update: {
				e.sql = SynchroQueryBuilder.newBuilder(e.queryName, e.sql)
						.deleteColumnIfExists(COLUMN_ORIGIN_FISHING_TRIP_REMOTE_ID)
						.replaceColumn(COLUMN_ORIGIN_FISHING_TRIP_FK, COLUMN_ORIGIN_FISHING_TRIP_REMOTE_ID)
						.build();
				break;
			}
			default:
				break;
			}
		}
	}

	@Subscribe
	public void handleJoinLoad(LoadJoinEvent e) {
		SynchroJoinMetadata join = e.join;

		if (join.isSourceFk()
				&& !getConfig().isMirrorDatabase()
				&& COLUMN_ORIGIN_FISHING_TRIP_FK.equalsIgnoreCase(e.join.getSourceColumn().getName())) {
			// Do not load fishing trip has child
			e.join.setIsValid(false);
		}
	}

}
