package fr.ifremer.adagio.core.service.data.synchro.intercept;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.List;

import org.hibernate.tool.hbm2ddl.TableMetadata;

import com.google.common.collect.ImmutableList;
import com.google.common.eventbus.Subscribe;

import fr.ifremer.adagio.core.service.data.synchro.DataSynchroDirection;
import fr.ifremer.adagio.synchro.dao.Daos;
import fr.ifremer.adagio.synchro.dao.SynchroTableDao;
import fr.ifremer.adagio.synchro.intercept.SynchroOperationRepository;
import fr.ifremer.adagio.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata.DuplicateKeyStrategy;
import fr.ifremer.adagio.synchro.meta.event.CreateQueryEvent;
import fr.ifremer.adagio.synchro.meta.event.LoadJoinEvent;
import fr.ifremer.adagio.synchro.meta.event.LoadTableEvent;
import fr.ifremer.adagio.synchro.query.SynchroQueryBuilder;
import fr.ifremer.adagio.synchro.query.SynchroQueryOperator;

/**
 * 
 * @author Benoit Lavenier <benoit.lavenier@e-is.pro>
 * @since 3.5.0
 * 
 */
public class FishingTripInterceptor extends DataAbstractSynchroInterceptor {

	public static final String TABLE_LANDING = "LANDING";
	public static final String LANDING_COLUMN_FISHING_TRIP_FK = "fishing_trip_fk";

	private static final List<String> FISHING_TRIP_UNIQUE_KEY = ImmutableList.of("DEPARTURE_DATE_TIME", "VESSEL_FK", "PROGRAM_FK");

	private PreparedStatement updateLandingFishingTripFkStatement = null;

	public FishingTripInterceptor() {
		super();
		setEnableOnWrite(true);
	}

	@Override
	public boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table) {
		return "FISHING_TRIP".equalsIgnoreCase(table.getName());
	}

	@Subscribe
	public void handleCreateQuery(CreateQueryEvent e) {
		switch (e.queryName) {
		// Select queries : remove unsed columns
		case count:
		case countFromUpdateDate:
		case select:
		case selectFromUpdateDate:
		case selectMaxUpdateDate:
			// Add restriction on person session
			e.sql = SynchroQueryBuilder.newBuilder(e.sql)
					// Evict if linked to sientific cruise
					.addWhere(SynchroQueryOperator.AND, "t.scientific_cruise_fk is null")
					// Evict if already linked to landing
					.addJoin(" LEFT OUTER JOIN landing la ON la.fishing_trip_fk=t.id")
					.addWhere(SynchroQueryOperator.AND, "la.id is null")
					.build();
			break;

		default:
			break;
		}
	}

	@Subscribe
	public void handleLoadJoin(LoadJoinEvent e) {
		if (!e.join.isValid()) {
			return;
		}

		// Disable child join to LANDING
		// (LANDING should not be seen as a child of FISHING_TRIP)
		String targetTableName = e.join.getTargetTable().getName();
		String targetColumnName = e.join.getTargetColumn().getName();
		boolean isJoinToLandingTable = TABLE_LANDING.equalsIgnoreCase(targetTableName)
				&& LANDING_COLUMN_FISHING_TRIP_FK.equalsIgnoreCase(targetColumnName);

		if (isJoinToLandingTable) {
			e.join.setIsValid(false);
		}
	}

	@Subscribe
	public void handleTableLoad(LoadTableEvent e) {
		e.table.setRoot(true);

		// Define a natural Id, on server
		if (getConfig().getDirection() == DataSynchroDirection.EXPORT_TEMP2SERVER) {
			e.table.addUniqueConstraint("FISHING_TRIP_UNIQUE_KEY", FISHING_TRIP_UNIQUE_KEY, DuplicateKeyStrategy.REJECT);
		}
	}

	@Override
	protected void doOnDelete(List<Object> pk, SynchroTableDao sourceDao, SynchroTableDao targetDao, SynchroOperationRepository buffer)
			throws SQLException {

		int localId = Integer.parseInt(pk.get(0).toString());

		// remove reference contains by LANDING.FISHING_TRIP_FK
		removeLinkToLanding(targetDao.getConnection(), localId);
	}

	@Override
	protected void doClose() throws IOException {
		super.doClose();

		Daos.closeSilently(updateLandingFishingTripFkStatement);
		updateLandingFishingTripFkStatement = null;
	}

	/* -- Internal methods -- */

	protected void removeLinkToLanding(Connection connection, int fishingTripId) throws SQLException {
		if (updateLandingFishingTripFkStatement == null) {
			updateLandingFishingTripFkStatement = connection.prepareStatement(initUpdateLandingFishingTripFkStatement());
		}

		updateLandingFishingTripFkStatement.setInt(1, fishingTripId);
		updateLandingFishingTripFkStatement.executeUpdate();
	}

	protected String initUpdateLandingFishingTripFkStatement() {
		return String.format("UPDATE %s SET %s = null WHERE %s = ?",
				TABLE_LANDING,
				LANDING_COLUMN_FISHING_TRIP_FK,
				LANDING_COLUMN_FISHING_TRIP_FK
				);
	}
}
