package fr.ifremer.adagio.core.service.data.synchro.intercept;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.util.Set;

import org.apache.commons.collections4.CollectionUtils;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import com.google.common.base.Joiner;
import com.google.common.collect.Sets;
import com.google.common.eventbus.Subscribe;

import fr.ifremer.adagio.core.service.data.synchro.DataSynchroDirection;
import fr.ifremer.adagio.synchro.config.SynchroConfiguration;
import fr.ifremer.adagio.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.adagio.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.adagio.synchro.meta.event.CreateQueryEvent;
import fr.ifremer.adagio.synchro.meta.event.LoadTableEvent;
import fr.ifremer.adagio.synchro.query.SynchroQueryBuilder;
import fr.ifremer.adagio.synchro.query.SynchroQueryOperator;

/**
 * Import DELETED_ITEM_HISTORY rows, limited to data tables
 * 
 * @author Benoit Lavenier <benoit.lavenier@e-is.pro>
 * @since 3.6.3
 * 
 */
public class DeletedItemHistoryInterceptor extends DataAbstractSynchroInterceptor {

	private static String TABLE_NAME = "DELETED_ITEM_HISTORY";
	private static String COLUMN_OBJECT_TYPE = "object_type_fk";
	private static String COLUMN_REMOTE_ID = "remote_id";

	private String whereClauseOnTableIncludes = null;

	private String whereClauseForExportLocal2Temp = null;

	public DeletedItemHistoryInterceptor() {
		// Exclude Temp-> Local (process in a special code)
		super(DataSynchroDirection.IMPORT_SERVER2TEMP,
				DataSynchroDirection.EXPORT_LOCAL2TEMP,
				DataSynchroDirection.EXPORT_TEMP2SERVER);
	}

	@Override
	public SynchroInterceptorBase clone() {
		DeletedItemHistoryInterceptor newBean = (DeletedItemHistoryInterceptor) super.clone();
		newBean.whereClauseOnTableIncludes = this.whereClauseOnTableIncludes;
		return newBean;
	}

	@Override
	public boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table) {
		return TABLE_NAME.equalsIgnoreCase(table.getName());
	}

	@Subscribe
	public void handleCreateQuery(CreateQueryEvent e) {
		switch (e.queryName) {
		case count:
		case countFromUpdateDate:
		case select:
		case selectFromUpdateDate:
		case selectMaxUpdateDate:
			e.sql = addRestrictions(e.sql);
			break;

		default:
			break;
		}
	}

	@Subscribe
	public void handleTableLoad(LoadTableEvent e) {
		// Do not import this table to local
		boolean isRoot = isInDirections(DataSynchroDirection.IMPORT_SERVER2TEMP,
				DataSynchroDirection.EXPORT_LOCAL2TEMP,
				DataSynchroDirection.EXPORT_TEMP2SERVER);

		e.table.setRoot(isRoot);
	}

	/* -- Internal methods -- */

	private String addRestrictions(String sql) {

		SynchroQueryBuilder query = SynchroQueryBuilder.newBuilder(sql);

		// Add limit on included tables
		if (whereClauseOnTableIncludes == null) {
			whereClauseOnTableIncludes = createWhereClauseOnTableIncludes();
		}
		query.addWhere(SynchroQueryOperator.AND, whereClauseOnTableIncludes);

		// Export: only delete done by connected user, and not exported deletion
		if (getConfig().getDirection() == DataSynchroDirection.EXPORT_LOCAL2TEMP) {
			if (whereClauseForExportLocal2Temp == null) {
				whereClauseForExportLocal2Temp = createWhereClauseForExportLocal2Temp();
			}

			query.addWhere(SynchroQueryOperator.AND, whereClauseForExportLocal2Temp);
		}

		return query.build();
	}

	private String createWhereClauseOnTableIncludes() {
		Set<String> tableToIncludes = SynchroConfiguration.getInstance().getImportDataTablesIncludes();

		// If no tables configured : should return no lines
		if (CollectionUtils.isEmpty(tableToIncludes)) {
			return "1=2";
		}

		Set<String> allObjectTypes = Sets.newHashSet();
		for (String tableName : tableToIncludes) {
			Set<String> objectTypes = ObjectTypeHelper.getObjectTypeFromTableName(tableName);
			if (CollectionUtils.isNotEmpty(objectTypes)) {
				allObjectTypes.addAll(objectTypes);
			}
			else {
				// By default, add the table name itself (i.e. PERSON_SESSION_VESSEL)
				allObjectTypes.add(tableName);
			}
		}

		// Remove itself
		allObjectTypes.remove(TABLE_NAME);

		return new StringBuilder()
				.append(COLUMN_OBJECT_TYPE)
				.append(" IN ('")
				.append(Joiner.on("','").join(allObjectTypes))
				.append("')")
				.toString();
	}

	private String createWhereClauseForExportLocal2Temp() {

		// mantis #22850 : do not limit to the recorder of the deletion
		// Integer recorderPersonId = checkAndGetPersonId();
		// return String.format("%s = %s AND %s is null",COLUMN_RECORDER_PERSON,recorderPersonId,COLUMN_REMOTE_ID);

		return COLUMN_REMOTE_ID + " is null";
	}
}
