package fr.ifremer.adagio.core.service.data.synchro.intercept;

/*
 * #%L
 * SIH-Adagio :: Core for Allegro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.sql.Timestamp;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.eventbus.Subscribe;

import fr.ifremer.adagio.core.service.data.synchro.DataSynchroDirection;
import fr.ifremer.adagio.core.service.data.synchro.intercept.internal.ExportFkRemoteIdInterceptor;
import fr.ifremer.adagio.core.service.data.synchro.intercept.internal.ExportPkRemoteIdInterceptor;
import fr.ifremer.adagio.core.service.data.synchro.intercept.internal.ImportRemoteIdInterceptor;
import fr.ifremer.adagio.synchro.config.SynchroConfiguration;
import fr.ifremer.adagio.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.adagio.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroJoinMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata.DuplicateKeyStrategy;
import fr.ifremer.adagio.synchro.meta.event.CreateQueryEvent;
import fr.ifremer.adagio.synchro.meta.event.LoadJoinEvent;
import fr.ifremer.adagio.synchro.meta.event.LoadTableEvent;
import fr.ifremer.adagio.synchro.query.SynchroQueryBuilder;
import fr.ifremer.adagio.synchro.query.SynchroQueryOperator;
import fr.ifremer.adagio.synchro.service.SynchroDatabaseConfiguration;

/**
 * Manage only data table with columns 'id' AND 'remote_id' :
 * <ul>
 * <li>Set as root only if the table as an object_type (OBJECT_TYPE is need for join with PERSON_SESION_VESSEL)
 * <li>when exporting to server, override 'update_date' column with a systimestamp
 * <p/>
 * 
 * 
 * @author Benoit Lavenier <benoit.lavenier@e-is.pro>
 * @since 3.6.3
 * 
 */
public class DataTableInterceptor extends DataAbstractSynchroInterceptor {

	private Set<String> dataTableIncludes;
	private Timestamp systimestamp = null;

	public DataTableInterceptor() {
		super();
		dataTableIncludes = SynchroConfiguration.getInstance().getImportDataTablesIncludes();
	}

	@Override
	public boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table) {
		if (CollectionUtils.isEmpty(dataTableIncludes)) {
			return false;
		}

		boolean isDataTable = dataTableIncludes.contains(table.getName());

		return isDataTable && hasColumns(table, getConfig().getColumnId(), getConfig().getColumnRemoteId());
	}

	@Override
	public SynchroInterceptorBase clone() {
		DataTableInterceptor result = (DataTableInterceptor) super.clone();
		result.dataTableIncludes = this.dataTableIncludes;
		result.systimestamp = this.systimestamp;
		return result;
	}

	@Subscribe
	public void handleQuery(CreateQueryEvent e) {
		DataSynchroDirection direction = getConfig().getDirection();

		// IMPORT: Server DB -> Temp DB
		if (direction == DataSynchroDirection.IMPORT_SERVER2TEMP) {

			switch (e.queryName) {
			// Select queries : remove unsed columns
			case select:
			case selectFromUpdateDate:
				e.sql = SynchroQueryBuilder.newBuilder(e.queryName, e.sql)
						.deleteColumnIfExists(getConfig().getColumnRemoteId())
						.deleteColumnIfExists(getConfig().getColumnSynchronizationStatus())
						.build();
				break;
			case insert:
			case update: {
				SynchroQueryBuilder qb = SynchroQueryBuilder.newBuilder(e.queryName, e.sql)
						.deleteColumnIfExists(getConfig().getColumnRemoteId());
				if (qb.constainsColumn(getConfig().getColumnSynchronizationStatus())) {
					qb.setColumnValue(getConfig().getColumnSynchronizationStatus(), "'SYNC'");
				}
				e.sql = qb.build();
				break;
			}
			default:
				break;
			}
		}

		// IMPORT: Temp DB -> Local DB
		else if (direction == DataSynchroDirection.IMPORT_TEMP2LOCAL) {

			switch (e.queryName) {
			// Select queries : remove unsed columns
			case select:
			case selectFromUpdateDate:
				e.sql = SynchroQueryBuilder.newBuilder(e.queryName, e.sql)
						.deleteColumnIfExists(getConfig().getColumnRemoteId())
						.deleteColumnIfExists(getConfig().getColumnSynchronizationStatus())
						.build();
				break;
			case insert: {
				SynchroQueryBuilder qb = SynchroQueryBuilder.newBuilder(e.queryName, e.sql)
						.deleteColumn(getConfig().getColumnRemoteId())
						.replaceColumn(getConfig().getColumnId(), getConfig().getColumnRemoteId());
				if (qb.constainsColumn(getConfig().getColumnSynchronizationStatus())) {
					qb.setColumnValue(getConfig().getColumnSynchronizationStatus(), "'SYNC'");
				}
				qb.addColumn(getConfig().getColumnId(), e.source.getSelectSequenceNextValString());
				e.sql = qb.build();
				break;
			}
			case update: {
				SynchroQueryBuilder qb = SynchroQueryBuilder.newBuilder(e.queryName, e.sql)
						.deleteColumn(getConfig().getColumnRemoteId())
						.replaceColumn(getConfig().getColumnId(), getConfig().getColumnRemoteId());
				if (qb.constainsColumn(getConfig().getColumnSynchronizationStatus())) {
					qb.setColumnValue(getConfig().getColumnSynchronizationStatus(), "'SYNC'");
				}
				e.sql = qb.build();
				break;
			}
			default:
				break;
			}
		}

		// EXPORT: Local DB -> Temp DB
		else if (direction == DataSynchroDirection.EXPORT_LOCAL2TEMP) {

			// Nothing to do ?
		}

		// EXPORT: Temp DB -> Server DB
		else if (direction == DataSynchroDirection.EXPORT_TEMP2SERVER) {

			switch (e.queryName) {
			case selectMaxUpdateDate:
				// Do not try to retrieve max update_date on server
				e.sql = SynchroQueryBuilder.newBuilder(e.queryName, e.sql)
						.addWhere(SynchroQueryOperator.AND, "1=2")
						.build();
				break;
			// Select queries : remove unsed columns
			case select:
			case selectFromUpdateDate:
				e.sql = SynchroQueryBuilder.newBuilder(e.queryName, e.sql)
						.deleteColumnIfExists(getConfig().getColumnSynchronizationStatus())
						.deleteColumnIfExists(getConfig().getColumnRemoteId())
						.build();
				break;
			case insert: {
				SynchroQueryBuilder qb = SynchroQueryBuilder.newBuilder(e.queryName, e.sql)
						.deleteColumnIfExists(getConfig().getColumnSynchronizationStatus())
						.deleteColumnIfExists(getConfig().getColumnRemoteId());
				e.sql = qb.build();
				break;
			}
			case update: {
				SynchroQueryBuilder qb = SynchroQueryBuilder.newBuilder(e.queryName, e.sql)
						.deleteColumnIfExists(getConfig().getColumnSynchronizationStatus())
						.deleteColumnIfExists(getConfig().getColumnRemoteId());
				e.sql = qb.build();
				break;
			}
			default:
				break;
			}
		}
	}

	@Subscribe
	public void handleTableLoad(LoadTableEvent e) {

		SynchroTableMetadata table = e.table;

		// Set table as root if synchronization status column exists
		boolean hasSynchronizationStatus = table.getColumnIndex(getConfig().getColumnSynchronizationStatus()) != -1;
		boolean hasObjectType = ObjectTypeHelper.getObjectTypeFromTableName(e.table.getName()) != null;

		// Set isRoot if not already done
		// for example, DeletedItemHistoryInterceptor already set as true, so do not override this to false !
		if (!table.isRoot() && hasSynchronizationStatus && hasObjectType) {
			// BLA 23/09/2014 : avoid Landing to be et as root
			// table.setRoot(true);
		}

		String remoteIdColumn = getConfig().getColumnRemoteId();
		String idColumn = getConfig().getColumnId();
		boolean hasRemoteIdColumn = table.getColumnIndex(remoteIdColumn) != -1;
		DataSynchroDirection direction = getConfig().getDirection();

		// Temp DB -> Local DB
		if (hasRemoteIdColumn && direction == DataSynchroDirection.IMPORT_TEMP2LOCAL) {
			// Define a natural id on REMOTE_ID, if not already define
			// (could have been override - e.g. ProduceSortingMeasurementInterceptor)
			if (!table.hasUniqueConstraint(remoteIdColumn)) {
				table.addUniqueConstraint(remoteIdColumn, ImmutableList.of(remoteIdColumn), DuplicateKeyStrategy.REPLACE);
			}
		}

		// Temp DB -> Server DB
		else if (hasRemoteIdColumn && direction == DataSynchroDirection.EXPORT_TEMP2SERVER) {
			if (systimestamp == null) {
				systimestamp = checkAndGetSystemTimestamp(getConfig());
			}

			// Define a natural id on ID column, if not already define
			// (could have been override - e.g. ProduceSortingMeasurementInterceptor)
			if (!table.hasUniqueConstraint(remoteIdColumn)) {
				table.addUniqueConstraint(remoteIdColumn, ImmutableList.of(idColumn), DuplicateKeyStrategy.REPLACE);
			}

			int updateDateColumnIndex = table.getSelectColumnIndex(getConfig().getColumnUpdateDate());
			boolean hasUpdateDate = updateDateColumnIndex != -1;

			ExportPkRemoteIdInterceptor remoteIdInterceptor = new ExportPkRemoteIdInterceptor(
					getConfig(),
					table.getName(),
					getConfig().getColumnId(),
					table.getSelectColumnIndex(getConfig().getColumnId()),
					hasSynchronizationStatus,
					hasUpdateDate,
					updateDateColumnIndex,
					systimestamp,
					table.hasChildJoins());

			table.getInterceptors().add(remoteIdInterceptor);
		}

	}

	@Subscribe
	public void handleJoinLoad(LoadJoinEvent e) {
		SynchroJoinMetadata join = e.join;
		// Do not apply if mirror database
		if (!join.isValid()) {
			return;
		}

		SynchroTableMetadata fkTable = join.getFkTable();
		DataSynchroDirection direction = getConfig().getDirection();

		// If the FK table is the current table, or not processed by this interceptor
		if (fkTable == e.source || isNotInterceptedTable(fkTable)) {

			if (direction == DataSynchroDirection.IMPORT_TEMP2LOCAL) {
				SynchroTableMetadata pkTable = join.getPkTable();

				// Create and configure a interceptor, to rewrite remote id
				String pkTableName = pkTable.getName().toLowerCase();
				String fkColumnName = join.getFkColumn().getName().toLowerCase();
				int fkColumnIndex = fkTable.getSelectColumnIndex(fkColumnName);

				ImportRemoteIdInterceptor remoteIdInterceptor = new ImportRemoteIdInterceptor(
						getConfig(),
						pkTableName,
						fkColumnName,
						fkColumnIndex,
						join.getFkColumn().isNullable());

				fkTable.getInterceptors().add(remoteIdInterceptor);
			}

			else if (direction == DataSynchroDirection.EXPORT_TEMP2SERVER) {
				SynchroTableMetadata pkTable = join.getPkTable();

				// Create and configure a interceptor, to rewrite remote id
				String pkTableName = pkTable.getName().toLowerCase();
				String fkColumnName = join.getFkColumn().getName().toLowerCase();
				int fkColumnIndex = fkTable.getSelectColumnIndex(fkColumnName);

				ExportFkRemoteIdInterceptor remoteIdInterceptor = new ExportFkRemoteIdInterceptor(
						getConfig(),
						pkTableName,
						fkColumnName,
						fkColumnIndex,
						join.getFkColumn().isNullable());

				fkTable.getInterceptors().add(remoteIdInterceptor);

			}

		}

	}

	/* -- Internal methods -- */

	protected boolean isNotInterceptedTable(SynchroTableMetadata table) {

		Set<String> columnNames = table.getColumnNames();

		return !columnNames.contains(getConfig().getColumnId())
				|| !columnNames.contains(getConfig().getColumnRemoteId());
	}

	/* -- Protected methods -- */

	protected Timestamp checkAndGetSystemTimestamp(SynchroDatabaseConfiguration configuration) {
		Timestamp systimestamp = configuration.getSystemTimestamp();
		Preconditions.checkNotNull(systimestamp,
				String.format("Could not found system timestamp in database configuration. This is need for %s", getClass().getSimpleName()));
		return systimestamp;
	}
}
