package fr.ifremer.adagio.core.service.data.synchro.intercept;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.util.Arrays;
import java.util.Set;

import org.hibernate.tool.hbm2ddl.TableMetadata;

import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;

import fr.ifremer.adagio.core.service.data.synchro.DataSynchroDatabaseConfiguration;
import fr.ifremer.adagio.core.service.data.synchro.DataSynchroDirection;
import fr.ifremer.adagio.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.adagio.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.adagio.synchro.service.SynchroDatabaseConfiguration;

public abstract class DataAbstractSynchroInterceptor extends SynchroInterceptorBase {

	private SynchroDatabaseMetadata meta;
	private Integer personSessionId;
	private Integer personId;
	private Set<DataSynchroDirection> allowDirections = Sets.newHashSet();

	public DataAbstractSynchroInterceptor() {
		// All direction are allowed
		this.allowDirections = null;
	}

	public DataAbstractSynchroInterceptor(DataSynchroDirection... directions) {
		Preconditions.checkArgument(directions != null && directions.length > 0);

		// Fill allowed directions
		this.allowDirections = Sets.newHashSet(directions);
	}

	@Override
	public SynchroInterceptorBase clone() {
		DataAbstractSynchroInterceptor result = (DataAbstractSynchroInterceptor) super.clone();
		result.meta = this.meta;
		result.personSessionId = this.personSessionId;
		result.personId = this.personId;
		result.allowDirections = allowDirections;
		return result;
	}

	@Override
	public final boolean apply(SynchroDatabaseConfiguration config) {
		boolean result = config instanceof DataSynchroDatabaseConfiguration
				/* super is need to store the config into a field */
				&& super.apply(config);

		if (result) {
			// Init the interceptor
			init((DataSynchroDatabaseConfiguration) config);
		}
		return result;
	}

	@Override
	public final boolean apply(SynchroDatabaseMetadata meta, TableMetadata table) {
		Preconditions.checkNotNull(meta);
		if (this.meta != null) {
			Preconditions.checkState(this.meta == meta);
		}

		// Store meta and configuration
		else {
			this.meta = meta;

			// Retrieve the person id
			personId = getConfig().getPersonId();

			// Retrieve the session id
			personSessionId = getConfig().getPersonSessionId();
		}

		return (allowDirections == null || allowDirections.contains(getConfig().getDirection()))
				&& doApply(meta, table);
	}

	public abstract boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table);

	public Integer checkAndGetPersonSessionId() {
		Preconditions
				.checkNotNull(
						personSessionId,
						String.format(
								"Could not retrieve user session (in database configuration). %s need a not null sessionId. Please set property %s in configuration.",
								getClass().getSimpleName(),
								DataSynchroDatabaseConfiguration.PERSON_SESSION_ID
								));
		return personSessionId;
	}

	public Integer checkAndGetPersonId() {
		Preconditions
				.checkNotNull(
						personId,
						String.format(
								"Could not retrieve user id (in database configuration). %s need a not null personId. Please set property %s in configuration.",
								getClass().getSimpleName(),
								DataSynchroDatabaseConfiguration.PERSON_ID
								));
		return personId;
	}

	/* -- delegate methods -- */

	public DataSynchroDatabaseConfiguration getConfig() {
		return (DataSynchroDatabaseConfiguration) getDefaultDatabaseConfiguration();
	}

	public Integer getPersonSessionId() {
		return personSessionId;
	}

	/* -- internal methods -- */

	protected void init(DataSynchroDatabaseConfiguration config) {
		// could be override
	}

	protected boolean isInDirections(DataSynchroDirection... allowDirections) {
		return Arrays.asList(allowDirections).contains(getConfig().getDirection());
	}
}
