package fr.ifremer.adagio.core.service.data.synchro.intercept;

/*
 * #%L
 * SIH-Adagio :: Synchronization
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.List;

import org.hibernate.tool.hbm2ddl.TableMetadata;

import com.google.common.eventbus.Subscribe;

import fr.ifremer.adagio.core.service.data.synchro.DataSynchroDatabaseConfiguration;
import fr.ifremer.adagio.core.service.data.synchro.DataSynchroDirection;
import fr.ifremer.adagio.core.service.data.synchro.intercept.internal.ExportFkRemoteIdInterceptor;
import fr.ifremer.adagio.core.service.data.synchro.intercept.internal.ImportRemoteIdInterceptor;
import fr.ifremer.adagio.synchro.dao.Daos;
import fr.ifremer.adagio.synchro.dao.SynchroBaseDao;
import fr.ifremer.adagio.synchro.dao.SynchroTableDao;
import fr.ifremer.adagio.synchro.intercept.SynchroInterceptorBase;
import fr.ifremer.adagio.synchro.intercept.SynchroOperationRepository;
import fr.ifremer.adagio.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata;
import fr.ifremer.adagio.synchro.meta.event.LoadJoinEvent;
import fr.ifremer.adagio.synchro.meta.event.LoadTableEvent;

/**
 * 
 * @author Lionel Touseau <lionel.touseau@e-is.pro>
 * @since 3.8.1
 */
public class BatchInterceptor extends DataAbstractSynchroInterceptor {

	private int rootBatchFkColumnIndex = -1;
	private int parentBatchFkColumnIndex = -1;
	private int batchIdColumnIndex = -1;
	private boolean enableIntegrityConstraints = true;
	private boolean hasParentBatchExportFkRemoteIdInterceptor = false;
	private boolean hasRootBatchExportFkRemoteIdInterceptor = false;
	private boolean hasParentBatchImportRemoteIdInterceptor = false;
	private boolean hasRootBatchImportRemoteIdInterceptor = false;

	private static final String TABLE_BATCH = "BATCH";
	private static final String TABLE_OPERATION = "OPERATION";

	public static final String COLUMN_ID = "id";
	public static final String COLUMN_ROOT_BATCH = "root_batch_fk";
	public static final String COLUMN_PARENT_BATCH = "parent_batch_fk";

	public static final String COLUMN_OPERATION_CATCH_BATCH_FK = "catch_batch_fk";

	private PreparedStatement updateOperationCatchBatchFkStatement = null;
	private PreparedStatement deleteChildrenBatchesByRootBatchFkStatement = null;

	public BatchInterceptor() {
		super();
		setEnableOnWrite(true);
	}

	@Override
	public void init(DataSynchroDatabaseConfiguration config) {
		super.init(config);
		enableIntegrityConstraints = config.getDirection() == DataSynchroDirection.EXPORT_TEMP2SERVER
				|| config.getDirection() == DataSynchroDirection.IMPORT_TEMP2LOCAL;
	}

	@Override
	public SynchroInterceptorBase clone() {
		BatchInterceptor newBean = (BatchInterceptor) super.clone();
		newBean.batchIdColumnIndex = this.batchIdColumnIndex;
		newBean.rootBatchFkColumnIndex = this.rootBatchFkColumnIndex;
		newBean.parentBatchFkColumnIndex = this.parentBatchFkColumnIndex;
		newBean.enableIntegrityConstraints = this.enableIntegrityConstraints;
		newBean.hasParentBatchExportFkRemoteIdInterceptor = this.hasParentBatchExportFkRemoteIdInterceptor;
		newBean.hasRootBatchExportFkRemoteIdInterceptor = this.hasRootBatchExportFkRemoteIdInterceptor;
		newBean.hasParentBatchImportRemoteIdInterceptor = this.hasParentBatchImportRemoteIdInterceptor;
		newBean.hasRootBatchImportRemoteIdInterceptor = this.hasRootBatchImportRemoteIdInterceptor;
		return newBean;
	}

	@Override
	public boolean doApply(SynchroDatabaseMetadata meta, TableMetadata table) {
		return TABLE_BATCH.equalsIgnoreCase(table.getName());
	}

	@Subscribe
	public void handleTableLoad(LoadTableEvent e) {
		// batches are not root
		e.table.setRoot(false);

		// get column indexes
		batchIdColumnIndex = e.table.getSelectColumnIndex(COLUMN_ID);
		rootBatchFkColumnIndex = e.table.getSelectColumnIndex(COLUMN_ROOT_BATCH);
		parentBatchFkColumnIndex = e.table.getSelectColumnIndex(COLUMN_PARENT_BATCH);
	}

	/**
	 * Disables parent-child relationships when loading
	 * 
	 * @param e
	 */
	@Subscribe
	public void handleJoinLoad(LoadJoinEvent e) {
		if (!e.join.isValid()) {
			return;
		}

		// Disable reverse Operation-CatchBatch relationship
		// (OPERATION should not be seen as a child of BATCH)
		if (TABLE_OPERATION.equalsIgnoreCase(e.join.getTargetTable().getName())) {
			// this parent link can be disabled
			e.join.setIsValid(false);
		}

		// Disable join between BATCH and BATCH (root_batch_fk, parent_batch_fk)
		String targetTableName = e.join.getTargetTable().getName();
		String targetColumnName = e.join.getTargetColumn().getName();
		boolean isJoinToRootOrParentTable = TABLE_BATCH.equalsIgnoreCase(targetTableName)
				&& (COLUMN_ROOT_BATCH.equalsIgnoreCase(targetColumnName)
				|| COLUMN_PARENT_BATCH.equalsIgnoreCase(targetColumnName));

		if (isJoinToRootOrParentTable) {
			// disable join
			e.join.setIsValid(false);

			// create ExportFk interceptor for root_batch_fk and parent_batch_fk when exporting from temp 2 server
			// ensure we register only once
			if ((!hasParentBatchExportFkRemoteIdInterceptor || !hasRootBatchExportFkRemoteIdInterceptor)
					&& getConfig().getDirection() == DataSynchroDirection.EXPORT_TEMP2SERVER) {

				SynchroTableMetadata batchTable = e.join.getFkTable();

				// Create and configure a interceptor, to rewrite remote id
				ExportFkRemoteIdInterceptor batchRemoteIdInterceptor = new ExportFkRemoteIdInterceptor(
						getConfig(),
						TABLE_BATCH.toLowerCase(),
						targetColumnName.toLowerCase(),
						batchTable.getSelectColumnIndex(targetColumnName),
						e.join.getFkColumn().isNullable());

				batchTable.getInterceptors().add(batchRemoteIdInterceptor);

				if (COLUMN_PARENT_BATCH.equalsIgnoreCase(targetColumnName)) {
					hasParentBatchExportFkRemoteIdInterceptor = true;
				} else if (COLUMN_ROOT_BATCH.equalsIgnoreCase(targetColumnName)) {
					hasRootBatchExportFkRemoteIdInterceptor = true;
				}

			}

			// create ImportRemoteId Interceptor for root_batch_fk and parent_batch_fk when importing from temp 2 local
			// ensure we register only once
			if ((!hasParentBatchImportRemoteIdInterceptor || hasRootBatchImportRemoteIdInterceptor)
					&& getConfig().getDirection() == DataSynchroDirection.IMPORT_TEMP2LOCAL) {

				SynchroTableMetadata batchTable = e.join.getPkTable();

				// Create and configure a interceptor, to rewrite remote id
				ImportRemoteIdInterceptor batchRemoteIdInterceptor = new ImportRemoteIdInterceptor(
						getConfig(),
						TABLE_BATCH.toLowerCase(),
						targetColumnName.toLowerCase(),
						batchTable.getSelectColumnIndex(targetColumnName),
						e.join.getFkColumn().isNullable());

				batchTable.getInterceptors().add(batchRemoteIdInterceptor);

				if (COLUMN_PARENT_BATCH.equalsIgnoreCase(targetColumnName)) {
					hasParentBatchImportRemoteIdInterceptor = true;
				} else if (COLUMN_ROOT_BATCH.equalsIgnoreCase(targetColumnName)) {
					hasRootBatchImportRemoteIdInterceptor = true;
				}

			}
		}

	}

	@Override
	protected void doOnWrite(Object[] data, List<Object> pk, SynchroTableDao sourceDao, SynchroTableDao targetDao, SynchroOperationRepository buffer)
			throws SQLException {

		// If root entity: no transformation
		if (buffer == null) {
			return;
		}

		// delay rootBatchFk write
		Object rootBatchFk = data[rootBatchFkColumnIndex];

		// root (catch) batch
		if (rootBatchFk == null) {
			// manually add root batch children
			Object rootBatchId = data[batchIdColumnIndex];
			buffer.addChildToUpdateFromOneColumn(TABLE_BATCH, COLUMN_ROOT_BATCH, rootBatchId);
		}

		// also delay parentBatchFk write
		Object parentBatchFk = data[parentBatchFkColumnIndex];
		if (parentBatchFk != null) {
			if (enableIntegrityConstraints) {
				// set as Null, and update later
				data[parentBatchFkColumnIndex] = null;

				buffer.addMissingColumnUpdate(COLUMN_PARENT_BATCH, pk, parentBatchFk);
			}
		}

	}

	@Override
	protected void doOnDelete(List<Object> pk, SynchroTableDao sourceDao, SynchroTableDao targetDao, SynchroOperationRepository buffer)
			throws SQLException {

		int localId = Integer.parseInt(pk.get(0).toString());

		// remove reference contains by OPERATION.CATCH_BATCH_FK
		removeLinkToOperation(targetDao, localId);

		// delete children batches (if root)
		deleteChildrenBatchesByRootBatch(targetDao, localId);
	}

	@Override
	protected void doClose() throws IOException {
		super.doClose();

		Daos.closeSilently(updateOperationCatchBatchFkStatement);
		Daos.closeSilently(deleteChildrenBatchesByRootBatchFkStatement);
		updateOperationCatchBatchFkStatement = null;
		deleteChildrenBatchesByRootBatchFkStatement = null;
	}

	/* -- Internal methods -- */

	protected void removeLinkToOperation(SynchroBaseDao targetDao, int catchBatchId) throws SQLException {
		if (updateOperationCatchBatchFkStatement == null || updateOperationCatchBatchFkStatement.isClosed()) {

			updateOperationCatchBatchFkStatement = targetDao.getPreparedStatement(getUpdateOperationCatchBatchFkQuery());
		}

		updateOperationCatchBatchFkStatement.setInt(1, catchBatchId);
		updateOperationCatchBatchFkStatement.executeUpdate();
	}

	protected String getUpdateOperationCatchBatchFkQuery() {
		return String.format("UPDATE %s SET %s = null WHERE %s = ?",
				TABLE_OPERATION,
				COLUMN_OPERATION_CATCH_BATCH_FK,
				COLUMN_OPERATION_CATCH_BATCH_FK
				);
	}

	protected void deleteChildrenBatchesByRootBatch(SynchroBaseDao targetDao, int rootBatchId) throws SQLException {
		if (deleteChildrenBatchesByRootBatchFkStatement == null || deleteChildrenBatchesByRootBatchFkStatement.isClosed()) {

			deleteChildrenBatchesByRootBatchFkStatement = targetDao.getPreparedStatement(getDeleteBatchByRootBatchFkQuery());
		}

		deleteChildrenBatchesByRootBatchFkStatement.setInt(1, rootBatchId);
		deleteChildrenBatchesByRootBatchFkStatement.executeUpdate();
	}

	protected String getDeleteBatchByRootBatchFkQuery() {
		return String.format("DELETE FROM %s WHERE %s = ?",
				TABLE_BATCH,
				COLUMN_ROOT_BATCH
				);
	}

}
