package fr.ifremer.adagio.core.service.data.synchro;

/*
 * #%L
 * SIH-Adagio :: Core for Allegro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import static org.nuiton.i18n.I18n.t;

import java.io.File;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Deque;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.annotation.Resource;
import javax.sql.DataSource;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.dialect.Dialect;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Queues;

import fr.ifremer.adagio.core.AdagioTechnicalException;
import fr.ifremer.adagio.core.config.AdagioConfiguration;
import fr.ifremer.adagio.core.dao.administration.user.PersonSessionDaoImpl;
import fr.ifremer.adagio.core.dao.administration.user.PersonSessionExtendDao;
import fr.ifremer.adagio.core.dao.technical.DaoUtils;
import fr.ifremer.adagio.core.dao.technical.DateUtils;
import fr.ifremer.adagio.core.dao.technical.hibernate.ConfigurationHelper;
import fr.ifremer.adagio.core.dao.technical.synchronization.SynchronizationStatus;
import fr.ifremer.adagio.core.service.data.synchro.intercept.ObjectTypeHelper;
import fr.ifremer.adagio.synchro.config.SynchroConfiguration;
import fr.ifremer.adagio.synchro.dao.DaoFactoryImpl;
import fr.ifremer.adagio.synchro.dao.SynchroTableDao;
import fr.ifremer.adagio.synchro.meta.SynchroDatabaseMetadata;
import fr.ifremer.adagio.synchro.meta.SynchroTableMetadata;
import fr.ifremer.adagio.synchro.service.RejectedRowStatus;
import fr.ifremer.adagio.synchro.service.RejectedRowStrategy;
import fr.ifremer.adagio.synchro.service.SynchroContext;
import fr.ifremer.adagio.synchro.service.SynchroDatabaseConfiguration;
import fr.ifremer.adagio.synchro.service.SynchroResult;
import fr.ifremer.adagio.synchro.service.SynchroTableOperation;

@Service("dataSynchroService")
@Lazy
public class DataSynchroServiceImpl
		extends fr.ifremer.adagio.synchro.service.SynchroServiceImpl
		implements DataSynchroService {

	private static final Log log = LogFactory.getLog(DataSynchroServiceImpl.class);

	private static boolean DISABLE_INTEGRITY_CONSTRAINTS = false;
	private static boolean ALLOW_MISSING_OPTIONAL_COLUMN = true;
	private static boolean ALLOW_ADDITIONAL_MANDATORY_COLUMN_IN_SOURCE_SCHEMA = false;
	private static boolean KEEP_WHERE_CLAUSE_ON_QUERIES_BY_FKS = false;

	private static final String TABLE_DELETED_ITEM_HISTORY = "DELETED_ITEM_HISTORY";
	private static final String TABLE_PERSON_SESSION = "PERSON_SESSION";
	private static final String TABLE_PERSON_SESSION_VESSEL = "PERSON_SESSION_VESSEL";
	private static final String COLUMN_SYNCHRONIZATION_STATUS = "synchronization_status";
	private static final String COLUMN_REMOTE_ID = "remote_id";

	private static final String TABLE_SORTING_MEASUREMENT_P = "SORTING_MEASUREMENT_P";

	@Resource
	private PersonSessionExtendDao personSessionDao = null;

	private final int exportUpdateDateDelayInSecond;

	@Autowired
	public DataSynchroServiceImpl(DataSource dataSource, SynchroConfiguration config) {
		super(dataSource, config,
				DISABLE_INTEGRITY_CONSTRAINTS,
				ALLOW_MISSING_OPTIONAL_COLUMN,
				ALLOW_ADDITIONAL_MANDATORY_COLUMN_IN_SOURCE_SCHEMA,
				KEEP_WHERE_CLAUSE_ON_QUERIES_BY_FKS);
		exportUpdateDateDelayInSecond = AdagioConfiguration.getInstance().getExportUpdateDateDelayInSecond();
	}

	public DataSynchroServiceImpl() {
		super(DISABLE_INTEGRITY_CONSTRAINTS,
				ALLOW_MISSING_OPTIONAL_COLUMN,
				ALLOW_ADDITIONAL_MANDATORY_COLUMN_IN_SOURCE_SCHEMA,
				KEEP_WHERE_CLAUSE_ON_QUERIES_BY_FKS);
		exportUpdateDateDelayInSecond = AdagioConfiguration.getInstance().getExportUpdateDateDelayInSecond();
		personSessionDao = new PersonSessionDaoImpl();
	}

	@Override
	public DataSynchroContext createSynchroContext(File sourceDbDirectory, DataSynchroDirection direction, int userId) {
		Preconditions.checkNotNull(sourceDbDirectory);
		Preconditions.checkArgument(sourceDbDirectory.exists() && sourceDbDirectory.isDirectory());
		return createSynchroContext(sourceDbDirectory, direction, userId, null);
	}

	@Override
	public DataSynchroContext createSynchroContext(File sourceDbDirectory, DataSynchroDirection direction, int userId,
			Timestamp lastSynchronizationDate) {
		SynchroContext delegateContext = super.createSynchroContext(sourceDbDirectory, config.getImportDataTablesIncludes());
		delegateContext.setLastSynchronizationDate(lastSynchronizationDate);
		return new DataSynchroContext(delegateContext, direction, userId);
	}

	@Override
	public DataSynchroContext createSynchroContext(Properties sourceConnectionProperties, DataSynchroDirection direction, int userId) {
		return createSynchroContext(sourceConnectionProperties, direction, userId, null);
	}

	@Override
	public DataSynchroContext createSynchroContext(Properties sourceConnectionProperties, DataSynchroDirection direction, int userId,
			Timestamp lastSynchronizationDate) {
		Preconditions.checkNotNull(sourceConnectionProperties);

		SynchroContext delegateContext = super.createSynchroContext(sourceConnectionProperties, config.getImportDataTablesIncludes());
		delegateContext.setLastSynchronizationDate(lastSynchronizationDate);

		return new DataSynchroContext(delegateContext, direction, userId);
	}

	@Override
	public void prepare(SynchroContext synchroContext) {
		Preconditions.checkArgument(synchroContext != null && synchroContext instanceof DataSynchroContext,
				String.format("The context must be a instance of %s", DataSynchroContext.class.getName()));

		DataSynchroContext dataSynchroContext = (DataSynchroContext) synchroContext;
		SynchroResult result = dataSynchroContext.getResult();

		DataSynchroDatabaseConfiguration target = (DataSynchroDatabaseConfiguration) dataSynchroContext.getTarget();
		DataSynchroDatabaseConfiguration source = (DataSynchroDatabaseConfiguration) dataSynchroContext.getSource();

		// If server DB -> Temp DB
		if (dataSynchroContext.getDirection() == DataSynchroDirection.IMPORT_SERVER2TEMP) {

			// Set target database as mirror AND temporary
			target.setIsMirrorDatabase(true);

			// Ignore some not used columns
			target.addColumnExclude("synchronization_status");
			target.addColumnExclude("remote_id");

			// Disable integrity constraints
			{
				disableIntegrityConstraints(target.getProperties(), result);
				if (!result.isSuccess()) {
					return;
				}
			}

			// Make sure personSessionId is filled
			if (dataSynchroContext.getSessionId() == null) {
				fillPersonSessionId(dataSynchroContext, result);

				if (!result.isSuccess()) {
					return;
				}
			}

			// Copy session id to target and source:
			if (dataSynchroContext.getSessionId() != null) {
				// set person session id
				source.setPersonSessionId(dataSynchroContext.getSessionId());
				target.setPersonSessionId(dataSynchroContext.getSessionId());
			}

			// Copy person id to target and source:
			if (dataSynchroContext.getUserId() != null) {
				source.setPersonId(dataSynchroContext.getUserId());
				target.setPersonId(dataSynchroContext.getUserId());
			}

			// Compute the default period, if need
			if (dataSynchroContext.getDataEndDate() == null) {
				fillDefaultDataPeriod(dataSynchroContext);
			}

			// Copy start/end date
			target.setDataStartDate(dataSynchroContext.getDataStartDate());
			target.setDataEndDate(dataSynchroContext.getDataEndDate());
			source.setDataStartDate(dataSynchroContext.getDataStartDate());
			source.setDataEndDate(dataSynchroContext.getDataEndDate());

			// Compute the default PK restriction, if need
			if (dataSynchroContext.getPkIncludes() == null) {
				fillDefaultPkIncludes(dataSynchroContext);
			}

			// Copy PKs by table
			target.setPkIncludes(dataSynchroContext.getPkIncludes());
			source.setPkIncludes(dataSynchroContext.getPkIncludes());
		}

		// If Temp DB -> Local DB
		else if (dataSynchroContext.getDirection() == DataSynchroDirection.IMPORT_TEMP2LOCAL) {

			// Set target database as not a Mirror
			target.setIsMirrorDatabase(false);
		}

		// If local -> temporary database
		else if (dataSynchroContext.getDirection() == DataSynchroDirection.EXPORT_LOCAL2TEMP) {

			// Set target database as mirror AND temporary
			target.setIsMirrorDatabase(true);

			// Disable integrity constraints
			{
				disableIntegrityConstraints(target.getProperties(), result);
				if (!result.isSuccess()) {
					return;
				}
			}

			// Copy person id to target and source:
			if (dataSynchroContext.getUserId() != null) {
				source.setPersonId(dataSynchroContext.getUserId());
				target.setPersonId(dataSynchroContext.getUserId());
			}
		}

		// If Temp DB -> server DB
		else if (dataSynchroContext.getDirection() == DataSynchroDirection.EXPORT_TEMP2SERVER) {

			// Ignore some not used columns
			source.addColumnExclude("synchronization_status");
			source.addColumnExclude("remote_id");
			source.setFullMetadataEnable(true);

			// Set target database as not a Mirror
			target.setIsMirrorDatabase(false);
			target.setFullMetadataEnable(false);

			// Fill system timestamp into context, and copy it to source
			fillSystemTimestampWithDelay(result, target);
			source.setSystemTimestamp(target.getSystemTimestamp());

			// Copy person id to target and source:
			if (dataSynchroContext.getUserId() != null) {
				source.setPersonId(dataSynchroContext.getUserId());
				target.setPersonId(dataSynchroContext.getUserId());
			}
		}

		super.prepare(synchroContext);
	}

	@Override
	public void synchronize(SynchroContext synchroContext) {
		Preconditions.checkArgument(synchroContext != null && synchroContext instanceof DataSynchroContext,
				String.format("The context must be a instance of %s", DataSynchroContext.class.getName()));

		DataSynchroContext dataSynchroContext = (DataSynchroContext) synchroContext;
		SynchroResult result = dataSynchroContext.getResult();

		DataSynchroDatabaseConfiguration target = (DataSynchroDatabaseConfiguration) dataSynchroContext.getTarget();
		DataSynchroDatabaseConfiguration source = (DataSynchroDatabaseConfiguration) dataSynchroContext.getSource();

		// If server -> temporary database
		if (dataSynchroContext.getDirection() == DataSynchroDirection.IMPORT_SERVER2TEMP) {

			// Enable protected columns
			target.removeColumnExclude("synchronization_status");
			target.removeColumnExclude("remote_id");

			// Disable integrity constraints
			{
				disableIntegrityConstraints(dataSynchroContext.getTarget().getProperties(), result);
				if (!result.isSuccess()) {
					return;
				}
			}
		}

		// If temp DB -> server DB
		else if (dataSynchroContext.getDirection() == DataSynchroDirection.EXPORT_TEMP2SERVER) {

			// Enable protected columns
			source.removeColumnExclude("synchronization_status");
			source.removeColumnExclude("remote_id");

			target.setReadOnly(true);
		}

		super.synchronize(synchroContext);
	}

	@Override
	public void finish(SynchroContext synchroContext, SynchroResult serverExportResult, Map<RejectedRowStatus, RejectedRowStrategy> rejectStrategies) {
		super.finish(synchroContext, serverExportResult, rejectStrategies);
	}

	/* -- Internal methods -- */

	@Override
	protected void prepareRootTable(
			DaoFactoryImpl sourceDaoFactory,
			DaoFactoryImpl targetDaoFactory,
			SynchroTableMetadata table,
			SynchroContext context,
			SynchroResult result) throws SQLException {

		// Prepare from super class
		super.prepareRootTable(sourceDaoFactory,
				targetDaoFactory,
				table,
				context,
				result);

		// Add deleted items
		prepareDeletedRootTable(sourceDaoFactory,
				targetDaoFactory,
				table,
				context,
				result);
	}

	/**
	 * Count number of row to delete, for the given table
	 * 
	 * @param sourceDaoFactory
	 * @param targetDaoFactory
	 * @param table
	 * @param context
	 * @param result
	 * @throws SQLException
	 */
	protected void prepareDeletedRootTable(
			DaoFactoryImpl sourceDaoFactory,
			DaoFactoryImpl targetDaoFactory,
			SynchroTableMetadata table,
			SynchroContext context,
			SynchroResult result) throws SQLException {

		String tableName = table.getName();
		Set<String> objectTypeFks = ObjectTypeHelper.getObjectTypeFromTableName(tableName);

		if (CollectionUtils.isEmpty(objectTypeFks)) {
			return;
		}

		SynchroTableDao dihSourceDao = sourceDaoFactory.getSourceDao(TABLE_DELETED_ITEM_HISTORY);

		List<List<Object>> columnValues = Lists.newArrayListWithCapacity(objectTypeFks.size());
		for (String objectTypeFk : objectTypeFks) {
			columnValues.add(ImmutableList.<Object> of(objectTypeFk));
		}

		// Read rows of DELETED_ITEM_HISTORY (from the temp DB)
		Map<String, Object> bindings = createSelectBindingsForTable(context, TABLE_DELETED_ITEM_HISTORY);
		long count = dihSourceDao.countDataByFks(
				ImmutableSet.of("OBJECT_TYPE_FK"),
				columnValues,
				bindings
				);
		if (count > 0) {
			result.addRows(tableName, (int) count);
		}
	}

	@Override
	protected Map<String, Object> createDefaultSelectBindings(SynchroContext context) {

		Map<String, Object> bindings = super.createDefaultSelectBindings(context);

		if (context instanceof DataSynchroContext) {
			DataSynchroContext dataContext = (DataSynchroContext) context;

			// Fill with the current person_fk (need for Dao and insert into TempQueryParemeter)
			bindings.put("userId", dataContext.getUserId());

			// Fill the period (using in interceptor queries - see ObservedLocationInterceptor)
			if (dataContext.getDataStartDate() != null) {
				bindings.put("startDate", new Timestamp(dataContext.getDataStartDate().getTime()));
				bindings.put("endDate", new Timestamp(dataContext.getDataEndDate().getTime()));
			}
		}

		return bindings;
	}

	/**
	 * Override the default method, to disable this binding: <br/>
	 * <code>bindings.put(SynchroTableMetadata.UPDATE_DATE_BINDPARAM, context.getResult().getUpdateDate(tableName))</code>
	 */
	@Override
	protected Map<String, Object> createSelectBindingsForTable(SynchroContext context, String tableName) {
		// Get defaults binding from context
		Map<String, Object> bindings = Maps.newHashMap(getSelectBindings(context));

		Timestamp lastSynchronizationDate = context.getLastSynchronizationDate();
		boolean notEmptyTargetTable = (context.getResult().getUpdateDate(tableName) != null);
		boolean enableUpdateDateFilter = (lastSynchronizationDate != null)
				&& (context.getTarget().isMirrorDatabase() || notEmptyTargetTable);

		if (enableUpdateDateFilter) {
			bindings.put(SynchroTableMetadata.UPDATE_DATE_BINDPARAM, lastSynchronizationDate);
		}

		return bindings;
	}

	protected void fillPersonSessionId(DataSynchroContext dataSynchroContext, SynchroResult result) {
		Preconditions.checkNotNull(dataSynchroContext.getUserId(),
				"One of 'userId' or 'sessionId' must be set in the synchro context");

		result.getProgressionModel().setMessage(t("adagio.synchro.synchronizeData.initPersonSession"));
		if (log.isInfoEnabled()) {
			log.info(t("adagio.synchro.synchronizeData.initPersonSession.log", dataSynchroContext.getUserId()));
		}

		try {
			int sessionId = personSessionDao.initPersonSession(
					dataSynchroContext.getSource().getProperties(),
					dataSynchroContext.getUserId());
			if (log.isDebugEnabled()) {
				log.debug(String.format("Session initialized: id=%s", sessionId));
			}

			dataSynchroContext.setSessionId(sessionId);
		} catch (AdagioTechnicalException e) {
			result.setError(e);
		}

	}

	protected void fillSystemTimestampWithDelay(SynchroResult result, SynchroDatabaseConfiguration databaseConfiguration) {

		result.getProgressionModel().setMessage(t("adagio.synchro.synchronizeData.initSystemTimestamp"));
		if (log.isInfoEnabled()) {
			log.info(t("adagio.synchro.synchronizeData.initSystemTimestamp"));
		}

		Connection connection = null;
		try {
			connection = createConnection(databaseConfiguration);
			Dialect dialect = Dialect.getDialect(databaseConfiguration.getProperties());
			Timestamp currentTimestamp = DaoUtils.getCurrentTimestamp(connection, dialect);

			// Add a delay, to allow concurrent importation to reimport this data later
			DateUtils.addSeconds(currentTimestamp, exportUpdateDateDelayInSecond);

			if (log.isDebugEnabled()) {
				log.debug(String.format("Current timestamp: %s", currentTimestamp));
			}

			databaseConfiguration.setSystemTimestamp(currentTimestamp);
		} catch (AdagioTechnicalException e) {
			result.setError(e);
		} catch (SQLException e) {
			result.setError(e);
		} finally {
			closeSilently(connection);
		}
	}

	protected void fillDefaultDataPeriod(DataSynchroContext dataSynchroContext) {
		int nbYearDataHistory = AdagioConfiguration.getInstance().getImportNbYearDataHistory();

		// Compute the start date : X years ago, starting at January 1th
		Date dataStartDate = DateUtils.addYears(new Date(), -1 * nbYearDataHistory);
		dataStartDate = DateUtils.truncate(dataStartDate, Calendar.YEAR);
		dataSynchroContext.setDataStartDate(dataStartDate);

		// Compute the end : last hour of the day
		Date dataEndDate = DateUtils.lastSecondOfTheDay(new Date());
		dataSynchroContext.setDataEndDate(dataEndDate);
	}

	protected void fillDefaultPkIncludes(DataSynchroContext dataSynchroContext) {
		String confProperty = AdagioConfiguration.getInstance().getImportDataPkIncludes();
		Multimap<String, String> pkIncludes = ConfigurationHelper.getMultimap(confProperty);
		dataSynchroContext.setPkIncludes(pkIncludes);
	}

	@Override
	protected List<SynchroTableOperation> getRootOperations(
			DaoFactoryImpl sourceDaoFactory,
			DaoFactoryImpl targetDaoFactory,
			SynchroDatabaseMetadata dbMeta,
			SynchroContext context) throws SQLException {
		DataSynchroContext dataContext = (DataSynchroContext) context;
		List<SynchroTableOperation> result = Lists.newArrayList();

		// Add default operations
		Collection<SynchroTableOperation> defaultOperations = super.getRootOperations(sourceDaoFactory, targetDaoFactory, dbMeta, context);
		result.addAll(defaultOperations);

		if (dataContext.getDirection() == DataSynchroDirection.EXPORT_TEMP2SERVER
				|| dataContext.getDirection() == DataSynchroDirection.IMPORT_TEMP2LOCAL) {

			// Add delete items history operation
			Collection<SynchroTableOperation> deletedItemOperations = getDeleteOperations(sourceDaoFactory, targetDaoFactory, dbMeta, dataContext);
			result.addAll(deletedItemOperations);
		}

		if (dataContext.getDirection() == DataSynchroDirection.IMPORT_TEMP2LOCAL) {

			// Add delete items history operation
			SynchroTableOperation deletePsvOperation = getDeletePersonSessionVesselOperations(sourceDaoFactory, targetDaoFactory, dbMeta, dataContext);
			if (deletePsvOperation != null && !deletePsvOperation.isEmpty()) {
				result.add(deletePsvOperation);
			}
		}

		return result;
	}

	private Collection<SynchroTableOperation> getDeleteOperations(
			DaoFactoryImpl sourceDaoFactory,
			DaoFactoryImpl targetDaoFactory,
			SynchroDatabaseMetadata dbMeta,
			DataSynchroContext context) throws SQLException {
		Preconditions.checkArgument(dbMeta.getConfiguration().isFullMetadataEnable());

		Deque<SynchroTableOperation> result = Queues.newArrayDeque();

		SynchroTableDao dihSourceDao = sourceDaoFactory.getSourceDao(TABLE_DELETED_ITEM_HISTORY);
		Set<String> includedDataTables = config.getImportDataTablesIncludes();

		// Read rows of DELETED_ITEM_HISTORY (from the temp DB)
		Map<String, Object> bindings = createSelectBindingsForTable(context, TABLE_DELETED_ITEM_HISTORY);
		ResultSet dihResultSet = dihSourceDao.getData(bindings);

		List<Object> dihIdsToRemove = Lists.newArrayList();

		while (dihResultSet.next()) {

			long dihId = dihResultSet.getLong("ID");
			String objectType = dihResultSet.getString("OBJECT_TYPE_FK");
			String tableName = ObjectTypeHelper.getTableNameFromObjectType(objectType);

			boolean isDataTable = StringUtils.isNotBlank(tableName)
					&& includedDataTables.contains(tableName.toUpperCase());

			if (isDataTable) {
				SynchroTableDao targetDao = targetDaoFactory.getSourceDao(tableName);
				SynchroTableMetadata table = targetDao.getTable();
				boolean deleteOnRemoteId = (context.getDirection() == DataSynchroDirection.IMPORT_TEMP2LOCAL)
						&& (table.getColumnIndex(COLUMN_REMOTE_ID) != -1);

				String objectCode = dihResultSet.getString("OBJECT_CODE");
				String objectId = dihResultSet.getString("OBJECT_ID");

				// Delete by a PK (id or code)
				if (StringUtils.isNotBlank(objectCode) || StringUtils.isNotBlank(objectId)) {
					List<Object> pk = StringUtils.isNotBlank(objectCode)
							? ImmutableList.<Object> of(objectCode)
							: ImmutableList.<Object> of(objectId);

					boolean hasChildTables = targetDao.getTable().hasChildJoins();

					SynchroTableOperation operation = new SynchroTableOperation(tableName, context);
					operation.setEnableProgress(true);
					result.add(operation);

					// If import to local DB : object_id = table's remote_id
					if (deleteOnRemoteId) {
						operation.addChildrenToDeleteFromOneColumn(tableName, COLUMN_REMOTE_ID, pk);

						// Make sure DIH is deleted, if exported from this local DB
						dihIdsToRemove.add(dihId);
					}
					// If export to server DB : object_id = table's id
					else {
						operation.addMissingDelete(pk);
					}

					// If has children, add child deletion to result
					if (hasChildTables) {
						addDeleteChildrenToDeque(table, ImmutableList.of(pk), result, context);
					}
				}
			}
		}

		if (CollectionUtils.isNotEmpty(dihIdsToRemove)) {
			SynchroTableOperation operation = new SynchroTableOperation(TABLE_DELETED_ITEM_HISTORY, context);
			operation.addChildrenToDeleteFromOneColumn(TABLE_DELETED_ITEM_HISTORY, COLUMN_REMOTE_ID, dihIdsToRemove);
			result.add(operation);
		}

		return result;
	}

	private SynchroTableOperation getDeletePersonSessionVesselOperations(
			DaoFactoryImpl sourceDaoFactory,
			DaoFactoryImpl targetDaoFactory,
			SynchroDatabaseMetadata dbMeta,
			DataSynchroContext context) throws SQLException {
		Preconditions.checkArgument(dbMeta.getConfiguration().isFullMetadataEnable());

		// Retrieve the person session Id
		Integer personId = ((DataSynchroContext) context).getUserId();
		Integer personSessionId = getPersonSessionId(
				sourceDaoFactory,
				targetDaoFactory,
				dbMeta,
				context
				);

		// If no personSession in local DB (e.g. empty DB - mantis #23022): exit (nothing to delete)
		if (personSessionId == null) {
			return null;
		}

		Map<String, Object> emptyBindings = ImmutableMap.of();
		SynchroTableOperation result = new SynchroTableOperation(TABLE_PERSON_SESSION_VESSEL, context);
		result.setEnableProgress(true);

		// DAO for person session vessel
		SynchroTableMetadata psvTable = dbMeta.getTable(TABLE_PERSON_SESSION_VESSEL);
		SynchroTableDao psvSourceDao = sourceDaoFactory.getSourceDao(TABLE_PERSON_SESSION_VESSEL);
		SynchroTableDao psvTargetDao = targetDaoFactory.getTargetDao(TABLE_PERSON_SESSION_VESSEL, psvSourceDao, null);
		String psvObjectType = ObjectTypeHelper.getFirstObjectTypeFromTableName(TABLE_PERSON_SESSION_VESSEL, TABLE_PERSON_SESSION_VESSEL);
		int psvIdColumnIndex = psvTable.getSelectColumnIndex("id") + 1;
		int psvUpdateDateColumnIndex = psvTable.getSelectColumnIndex("update_date") + 1;
		Set<String> psvColumnNames = ImmutableSet.of("vessel_fk", "person_session_fk");

		// DAO for Deleted item history
		SynchroTableDao dihSourceDao = sourceDaoFactory.getSourceDao(TABLE_DELETED_ITEM_HISTORY);
		Set<String> dihColumnNames = ImmutableSet.of("object_type_fk", "recorder_person_fk");
		List<List<Object>> dihColumnValues = ImmutableList.<List<Object>> of(ImmutableList.<Object> of(psvObjectType, personId));
		Map<String, Object> dihBindings = createSelectBindingsForTable(context, TABLE_DELETED_ITEM_HISTORY);

		ResultSet dihResultSet = null;
		ResultSet psvResultSet = null;
		try {
			// Get deleted item history on PSV (on temp DB)
			dihResultSet = dihSourceDao.getDataByFks(dihColumnNames,
					dihColumnValues,
					dihBindings);

			while (dihResultSet.next()) {

				String vesselFk = dihResultSet.getString("VESSEL_FK");
				Timestamp updateDate = dihResultSet.getTimestamp("UPDATE_DATE");

				if (StringUtils.isNotBlank(vesselFk)) {

					// Retrieve corresponding PSV row (on local DB)
					List<List<Object>> psvColumnValues = ImmutableList.<List<Object>> of(ImmutableList.<Object> of(vesselFk, personSessionId));
					psvResultSet = psvTargetDao.getDataByFks(
							psvColumnNames,
							psvColumnValues,
							emptyBindings);

					while (psvResultSet.next()) {
						Integer psvId = psvResultSet.getInt(psvIdColumnIndex);
						Timestamp psvUpdateDate = psvResultSet.getTimestamp(psvUpdateDateColumnIndex);

						// Add to delete operation only if the deletion occur after the row date
						if (updateDate.after(psvUpdateDate)) {
							List<Object> pk = ImmutableList.<Object> of(psvId);
							result.addMissingDelete(pk);
						}
					}
				}
			}

			return result;
		} finally {
			DaoUtils.closeSilently(dihResultSet);
			DaoUtils.closeSilently(psvResultSet);
		}
	}

	@SuppressWarnings("resource")
	protected Integer getPersonSessionId(
			DaoFactoryImpl sourceDaoFactory,
			DaoFactoryImpl targetDaoFactory,
			SynchroDatabaseMetadata dbMeta,
			SynchroContext context) throws SQLException {

		Preconditions.checkNotNull(context);
		Integer personId = ((DataSynchroContext) context).getUserId();
		Preconditions.checkNotNull(personId);

		Map<String, Object> emptyBindings = ImmutableMap.of();

		// Person session Daos
		SynchroTableDao psSourceDao = sourceDaoFactory.getSourceDao(TABLE_PERSON_SESSION);
		SynchroTableDao psTargetDao = targetDaoFactory.getTargetDao(TABLE_PERSON_SESSION, psSourceDao, null);
		int psIdColumnIndex = psTargetDao.getTable().getSelectColumnIndex("id") + 1;

		ResultSet rs = null;
		try {
			rs = psTargetDao.getDataByFks(
					ImmutableSet.of("person_fk"),
					ImmutableList.<List<Object>> of(ImmutableList.<Object> of(personId)),
					emptyBindings);

			while (rs.next()) {
				int result = rs.getInt(psIdColumnIndex);
				if (result > 0) {
					return result;
				}
			}

			return null;
		} finally {
			DaoUtils.closeSilently(rs);
		}
	}

	@Override
	protected void resolveTableRow(SynchroTableMetadata table,
			String pkStr,
			String[] rejectInfos,
			RejectedRowStatus rejectStatus,
			RejectedRowStrategy rejectStrategy,
			SynchroTableOperation operation,
			SynchroContext context,
			SynchroResult result) {

		// Execute default implementation
		super.resolveTableRow(
				table,
				pkStr,
				rejectInfos,
				rejectStatus,
				rejectStrategy,
				operation,
				context,
				result
				);

		boolean hasSynchronizationStatus = table.getColumnNames().contains(COLUMN_SYNCHRONIZATION_STATUS);
		if (hasSynchronizationStatus) {
			return;
		}

		// Add a special case, when keep local data : make sure all entity will be re-export,
		// by changing synchronizaton_status to READ_TO_SYNC
		if (rejectStatus == RejectedRowStatus.BAD_UPDATE_DATE
				&& rejectStrategy == RejectedRowStrategy.KEEP_LOCAL) {

			operation.addMissingColumnUpdate(
					COLUMN_SYNCHRONIZATION_STATUS,
					pkStr,
					SynchronizationStatus.READY_TO_SYNCHRONIZE.getValue());
		}
	}

	@Override
	protected void detachRows(
			SynchroTableDao targetDao,
			List<List<Object>> pks,
			SynchroContext context,
			SynchroResult result,
			Deque<SynchroTableOperation> pendingOperations)
			throws SQLException {

		// Call default implementation
		super.detachRows(targetDao, pks, context, result, pendingOperations);

		// Then override to reset remote_id:

		SynchroTableMetadata table = targetDao.getTable();
		boolean hasSynchronizationStatus = table.getColumnNames().contains(COLUMN_SYNCHRONIZATION_STATUS);
		boolean hasRemoteId = table.getColumnNames().contains(COLUMN_REMOTE_ID);
		if (!hasSynchronizationStatus && !hasRemoteId) {
			return;
		}

		String tableName = table.getName();
		String tablePrefix = table.getTableLogPrefix() + " - " + result.getNbRows(tableName);
		String readyToSynchronize = SynchronizationStatus.READY_TO_SYNCHRONIZE.getValue();
		int countR = 0;

		// For each pk to detach: reset remote_id
		if (hasRemoteId) {
			for (List<Object> pk : pks) {
				targetDao.executeUpdateColumn(COLUMN_REMOTE_ID, pk, null);
				countR++;
				reportProgress(result, targetDao, countR, tablePrefix);
			}
		}

		// For each pk to detach: set synchronization_status to READY_TO_SYNC
		// (in case the status was SYNC)
		if (hasSynchronizationStatus) {
			for (List<Object> pk : pks) {
				targetDao.executeUpdateColumn(COLUMN_SYNCHRONIZATION_STATUS, pk, readyToSynchronize);
				countR++;
				reportProgress(result, targetDao, countR, tablePrefix);
			}
		}

		if (countR == 0) {
			return;
		}

		targetDao.flush();

		result.addTableName(tableName);
		result.addUpdates(tableName, countR);

		if (log.isInfoEnabled()) {
			log.info(String.format("%s done: %s (detachment: %s)", tablePrefix, countR, countR));
		}

		if (targetDao.getCurrentOperation().isEnableProgress()) {
			result.getProgressionModel().increments(countR % batchSize);
		}

	}

}
