package fr.ifremer.adagio.core.service.data.synchro;

/*
 * #%L
 * SIH-Adagio :: Core for Allegro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.util.Date;
import java.util.Properties;

import org.apache.commons.lang3.StringUtils;

import com.google.common.collect.ImmutableMultimap;
import com.google.common.collect.Multimap;

import fr.ifremer.adagio.core.dao.technical.hibernate.ConfigurationHelper;
import fr.ifremer.adagio.synchro.config.SynchroConfiguration;
import fr.ifremer.adagio.synchro.service.SynchroDatabaseConfiguration;

public class DataSynchroDatabaseConfiguration extends SynchroDatabaseConfiguration {

	/**
	 * Person session Id
	 */
	public static String PERSON_SESSION_ID = "adagio.synchro.database.personSessionId";

	/**
	 * Person Id
	 */
	public static String PERSON_ID = "adagio.synchro.database.personId";

	/**
	 * Temporary database
	 */
	public static String IS_TEMPORARY_DATABASE = "adagio.synchro.database.isTemporary";

	/**
	 * data start date
	 */
	public static String DATA_START_DATE = "adagio.synchro.import.dataStartDate";

	/**
	 * data end date
	 */
	public static String DATA_END_DATE = "adagio.synchro.import.dataEndDate";

	/**
	 * IDs to import, by table name
	 */
	public static String PK_INCLUDES = "adagio.synchro.import.pk.includes";

	/**
	 * Column remote id
	 */
	public static String COLUMN_REMOTE_ID = "synchro.database.column.remoteId";

	/**
	 * Column synchronization_status
	 */
	public static String COLUMN_SYNCHRONIZATION_STATUS = "synchro.database.column.synchronizationStatus";

	private final DataSynchroContext context;

	private Multimap<String, String> pkIncludesCache = null;

	private String columnRemoteId;

	private String columnSynchronizationStatus;

	public DataSynchroDatabaseConfiguration(SynchroConfiguration config, DataSynchroContext context, Properties settings, boolean isTarget) {
		super(config, settings, isTarget);
		this.context = context;
		this.columnRemoteId = ConfigurationHelper.getString(COLUMN_REMOTE_ID, settings, "remote_id");
		this.columnSynchronizationStatus = ConfigurationHelper.getString(COLUMN_SYNCHRONIZATION_STATUS, settings, "synchronization_status");
	}

	public String getColumnRemoteId() {
		return columnRemoteId;
	}

	public SynchroDatabaseConfiguration setColumnRemoteId(String columnName) {
		this.columnRemoteId = columnName;
		setProperty(COLUMN_REMOTE_ID, columnName);
		return this;
	}

	public String getColumnSynchronizationStatus() {
		return columnSynchronizationStatus;
	}

	public SynchroDatabaseConfiguration setColumnSynchronizationStatus(String columnName) {
		this.columnSynchronizationStatus = columnName;
		setProperty(COLUMN_SYNCHRONIZATION_STATUS, columnName);
		return this;
	}

	public Integer getPersonSessionId() {
		return ConfigurationHelper.getInteger(PERSON_SESSION_ID, getProperties());
	}

	public void setPersonSessionId(Integer personSessionId) {
		if (personSessionId == null) {
			getProperties().remove(PERSON_SESSION_ID);
		}
		else {
			setProperty(PERSON_SESSION_ID, personSessionId.toString());
		}
	}

	public Integer getPersonId() {
		return ConfigurationHelper.getInteger(PERSON_ID, getProperties());
	}

	public void setPersonId(Integer personId) {
		if (personId == null) {
			getProperties().remove(PERSON_ID);
		}
		else {
			setProperty(PERSON_ID, personId.toString());
		}
	}

	public void setIsTemporary(boolean isTemporayDatabase) {
		setProperty(IS_TEMPORARY_DATABASE, String.valueOf(isTemporayDatabase));
	}

	public void setDataStartDate(Date dataStartDate) {
		if (dataStartDate == null) {
			getProperties().remove(DATA_START_DATE);
		}
		else {
			setProperty(DATA_START_DATE, String.valueOf(dataStartDate.getTime()));
		}
	}

	public Date getDataStartDate() {
		long time = ConfigurationHelper.getLong(DATA_START_DATE, getProperties(), 0);
		if (time == 0) {
			return null;
		}
		return new Date(time);
	}

	public void setDataEndDate(Date dataEndDate) {
		if (dataEndDate == null) {
			getProperties().remove(DATA_END_DATE);
		}
		else {
			setProperty(DATA_END_DATE, String.valueOf(dataEndDate.getTime()));
		}
	}

	public Date getDataEndDate() {
		long time = ConfigurationHelper.getLong(DATA_END_DATE, getProperties(), 0);
		if (time == 0) {
			return null;
		}
		return new Date(time);
	}

	public void setPkIncludes(String pkIncludesProperty) {
		if (StringUtils.isBlank(pkIncludesProperty)) {
			getProperties().remove(PK_INCLUDES);
		}
		else {
			setProperty(PK_INCLUDES, pkIncludesProperty);
		}

		// clean cache
		pkIncludesCache = null;
	}

	public void setPkIncludes(Multimap<String, String> pkIncludes) {
		ConfigurationHelper.setMultimap(PK_INCLUDES, pkIncludes, getProperties());

		// update cache
		pkIncludesCache = pkIncludes != null ? ImmutableMultimap.copyOf(pkIncludes) : null;
	}

	public Multimap<String, String> getPkIncludes() {
		if (pkIncludesCache != null) {
			return pkIncludesCache;
		}

		// e.g. <TABLE1>|<ID1>:ID2,<TABLE2>|<ID2>;<ID2>
		Multimap<String, String> result = ConfigurationHelper.getMultimap(PK_INCLUDES, getProperties());

		// update cache
		pkIncludesCache = result != null ? ImmutableMultimap.copyOf(result) : null;

		return pkIncludesCache;
	}

	public DataSynchroDirection getDirection() {
		return context.getDirection();
	}

	public boolean isIntegrityConstraintEnable() {
		return !isMirrorDatabase();
	}

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder(super.toString());

		Integer personSessionId = getPersonSessionId();
		sb.append("\n    person session id: ").append(personSessionId == null ? "" : personSessionId);

		return sb.toString();
	}
}
