package fr.ifremer.adagio.core.dao.technical.synchronization;

/*
 * #%L
 * SIH-Adagio :: Core for Allegro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2015 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.Types;
import java.util.List;

import javax.sql.DataSource;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.SessionFactory;
import org.hibernate.dialect.Dialect;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.jdbc.datasource.DataSourceUtils;
import org.springframework.stereotype.Repository;

import com.google.common.base.Preconditions;

import fr.ifremer.adagio.core.AdagioTechnicalException;
import fr.ifremer.adagio.core.config.AdagioConfiguration;
import fr.ifremer.adagio.core.dao.technical.DaoUtils;
import fr.ifremer.adagio.synchro.config.SynchroConfiguration;

@Repository("tempQueryParameterDao")
@Lazy
public class TempQueryParameterDaoImpl extends TempQueryParameterDaoBase implements TempQueryParameterExtendDao {

	public static final String TEMP_QUERY_PARAMETER_TABLE = "TEMP_QUERY_PARAMETER";
	public static final String TEMP_QUERY_PARAMETER_VALUE_COLUMN = "ALPHANUMERICAL_VALUE";
	public static final String TEMP_QUERY_PARAMETER_PARAM_COLUMN = "PARAMETER_NAME";
	public static final String TEMP_QUERY_PARAMETER_SEQUENCE = "TEMP_QUERY_PARAMETER_SEQ";

	/** Logger. */
	private static final Log log =
			LogFactory.getLog(TempQueryParameterDaoImpl.class);

	private DataSource dataSource;
	private final Dialect dialect;
	private final String deleteTempQueryParameterQuery;
	private final String deleteTempQueryParameterQueryWithLike;
	private final String insertIntoTempQueryParameterQuery;

	private final int batchSize;

	@Autowired
	public TempQueryParameterDaoImpl(SessionFactory sessionFactory, DataSource dataSource, AdagioConfiguration config) {
		super();
		setSessionFactory(sessionFactory);
		this.dataSource = dataSource;
		this.batchSize = config.getJdbcBatchSize();
		this.dialect = initDialect(config);
		this.insertIntoTempQueryParameterQuery = initInsertIntoTempQueryParameterQuery(this.dialect);
		this.deleteTempQueryParameterQuery = initDeleteTempQueryParameterQuery(false);
		this.deleteTempQueryParameterQueryWithLike = initDeleteTempQueryParameterQuery(true);

	}

	@Override
	public void removeByParameterName(String queryParameterName, boolean likeOperatorForParameterName, int queryPersonId) {
		if (log.isDebugEnabled()) {
			log.debug(String.format("Deleting rows from %s", TEMP_QUERY_PARAMETER_TABLE.toUpperCase()));
		}

		String sql = likeOperatorForParameterName ? deleteTempQueryParameterQueryWithLike : deleteTempQueryParameterQuery;

		Connection connection = null;
		PreparedStatement statement = null;
		try {
			// Get statement from pool
			connection = DataSourceUtils.getConnection(dataSource);
			statement = connection.prepareStatement(sql);

			// execute the delete statement
			statement.setString(1, queryParameterName);
			statement.setInt(2, queryPersonId);
			int deletedRows = statement.executeUpdate();
			if (log.isTraceEnabled()) {
				log.trace(String.format("%s rows delete from TEMP_QUERY_PAREMETER", deletedRows));
			}
		} catch (Exception e) {
			throw new AdagioTechnicalException(String.format("Could not delete from table %s", TEMP_QUERY_PARAMETER_TABLE), e);
		} finally {
			DaoUtils.closeSilently(statement);
			DataSourceUtils.releaseConnection(connection, dataSource);
		}
	}

	@Override
	public void insertAll(String queryParameterName,
			int queryPersonId,
			List<Number> numericalValues,
			List<Object> alphanumerialValues) {

		boolean noNumericalValues = CollectionUtils.isEmpty(numericalValues);
		boolean noAlphanumerialValues = CollectionUtils.isEmpty(alphanumerialValues);
		Preconditions.checkArgument((noNumericalValues && !noAlphanumerialValues)
				|| (!noNumericalValues && noAlphanumerialValues)
				|| (CollectionUtils.size(numericalValues) == CollectionUtils.size(alphanumerialValues)),
				"Only value list must filled, or both list must have the same size");
		int valuesSize = (!noNumericalValues) ? numericalValues.size()
				: alphanumerialValues.size();

		Connection connection = null;
		PreparedStatement statement = null;
		try {

			// Delete existing rows in TEMP_QUERY_PARAMETER
			removeByParameterName(queryParameterName, false/* = */, queryPersonId);

			if (log.isDebugEnabled()) {
				log.debug(String.format("Setting query parameters into %s", TEMP_QUERY_PARAMETER_TABLE.toUpperCase()));
			}

			// Get statement
			connection = DataSourceUtils.getConnection(dataSource);
			statement = connection.prepareStatement(insertIntoTempQueryParameterQuery);

			int i = 0;
			while (i < valuesSize) {
				statement.setString(1, queryParameterName);

				// numerical value
				if (noNumericalValues) {
					statement.setNull(2, Types.INTEGER);
				}
				else {
					statement.setObject(2, numericalValues.get(i));
				}

				// alphanumerical value
				if (noAlphanumerialValues) {
					statement.setNull(3, Types.VARCHAR);
				}
				else {
					statement.setObject(3, alphanumerialValues.get(i));
				}

				statement.setInt(4, queryPersonId);
				statement.addBatch();

				i++;
				if (i % batchSize == 0) {
					statement.executeBatch();
					statement.clearBatch();
				}
			}

			if (i % batchSize != 0) {
				statement.executeBatch();
				statement.clearBatch();
			}

		} catch (Exception e) {
			throw new AdagioTechnicalException(String.format("Could not insert into table %s", TEMP_QUERY_PARAMETER_TABLE), e);
		} finally {
			DaoUtils.closeSilently(statement);
			DataSourceUtils.releaseConnection(connection, dataSource);
		}
	}

	/* -- internal methods -- */

	protected Dialect initDialect(AdagioConfiguration config) {
		return Dialect.getDialect(config.getConnectionProperties());
	}

	protected String initInsertIntoTempQueryParameterQuery(Dialect dialect) {

		String sequenceNextValString = dialect.getSelectSequenceNextValString(TEMP_QUERY_PARAMETER_SEQUENCE);

		String sql = String.format("INSERT INTO %s (ID, PARAMETER_NAME, NUMERICAL_VALUE, ALPHANUMERICAL_VALUE, PERSON_FK)"
				+ " VALUES (%s, ?, ?, ?, ?)",
				TEMP_QUERY_PARAMETER_TABLE,
				sequenceNextValString);

		return sql;
	}

	protected String initDeleteTempQueryParameterQuery(boolean likeOperatorForParameterName) {
		String sql = String.format("DELETE FROM %s WHERE parameter_name %s ? AND %s=?",
				TEMP_QUERY_PARAMETER_TABLE,
				likeOperatorForParameterName ? "LIKE" : "=",
				SynchroConfiguration.getInstance().getTempQueryParemeterUserIdColumn());

		return sql;
	}
}
